/********** MOP03File.C ***********************************************************************************************\

 PURPOSE
   To create MOP03.hdf --- Level 3 output file

 REVISION HISTORY
   11/02   Debbie Mao 
   01/03   Debbie Mao -- change the mop03 file from Grid to Swath      
   03/04   Debbie Mao -- change the mop03 file from Swath back to Grid 
   07/12   Debbie Mao -- change HDF-EOS2 to CF HDF-EOS5 CF with Augmentation, accessible for netCDF4 libs
   08/16   Debbie Mao -- add 6 new diagnostics fields

\**********************************************************************************************************************/
#include "MOP03File.h"
#include "MOP03Metadata.h"
#include "DiagnosticReporter.h"
#include <string>

extern diagnostic_reporter diagnosticreporter;

mop03_file :: mop03_file ()
            : name ()
{
  fileid = 0;
  gdid = 0;
  metadataid = 0;

  Initialize ();
}


mop03_file :: ~mop03_file ()
{
 ;
}

int mop03_file :: defFieldG ( string fn, string dimlist, string attrval )
{
  int status = 1;
  hsize_t attrcount[3];

  attrcount[0] = attrval.size();

  status = HE5_GDdeffield (gdid, fn.c_str(), ((char*)dimlist.c_str()), NULL, H5T_NATIVE_FLOAT, HE5_HDFE_NOMERGE);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 523, "Could not def Geo field"); 

  status = HE5_GDwritelocattr (gdid, fn.c_str(), "units", H5T_NATIVE_CHAR, attrcount, (VOIDP)attrval.c_str());
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 524, "Could not write units for Geo field"); 

  return status;
}

int mop03_file :: defFieldF ( string fn, string dimlist, string attrval, string lfn )
{
  string fieldname, lfieldname;
  int compparm[5];  
  int status = 1;
  float tempfill;
  hsize_t tiledims[2], attrcount[3];

  compparm[0] = 5;
  tempfill = -9999.0;
  tiledims[0] = 60;
  tiledims[1] = 30;
  attrcount[0] = attrval.size();

  fieldname = fn  + "Day";
  lfieldname = lfn + " Day";
  status = HE5_GDsetfillvalue(gdid, fieldname.c_str(), H5T_NATIVE_FLOAT, &tempfill);
  if ( status != 0 ) 	
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 520, "Could not set Day fill value");
  status = HE5_GDdeftile(gdid, HE5_HDFE_TILE, 2, tiledims);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 521, "Could not def Day tile");
  status = HE5_GDdefcomp(gdid, HE5_HDFE_COMP_DEFLATE, compparm);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 522, "Could not def Day compression");   
  status = HE5_GDdeffield (gdid, fieldname.c_str(), ((char*)dimlist.c_str()), NULL, H5T_NATIVE_FLOAT, HE5_HDFE_NOMERGE);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 523, "Could not def Day field"); 
  status = HE5_GDwritelocattr (gdid, fieldname.c_str(), "units", H5T_NATIVE_CHAR, attrcount, (VOIDP)attrval.c_str());
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 524, "Could not write units for Day field"); 
  attrcount[0] = lfieldname.size();
  status = HE5_GDwritelocattr (gdid,fieldname.c_str(),"long_name",H5T_NATIVE_CHAR, attrcount,(VOIDP)lfieldname.c_str());
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 524, "Could not write long_name for Day field"); 

  fieldname = fn + "Night";
  lfieldname = lfn + " Night";
  status = HE5_GDsetfillvalue(gdid, fieldname.c_str(), H5T_NATIVE_FLOAT, &tempfill);
  if ( status != 0 ) 	
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 525, "Could not set Night fill value"); 
  status = HE5_GDdeftile(gdid, HE5_HDFE_TILE, 2, tiledims);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 526, "Could not def Night tile");
  status = HE5_GDdefcomp(gdid, HE5_HDFE_COMP_DEFLATE, compparm);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 527, "Could not def Night compression");   
  status = HE5_GDdeffield (gdid, fieldname.c_str(), ((char*)dimlist.c_str()), NULL, H5T_NATIVE_FLOAT, HE5_HDFE_NOMERGE);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 528, "Could not def Night field"); 
  status = HE5_GDwritelocattr (gdid, fieldname.c_str(), "units", H5T_NATIVE_CHAR, attrcount, (VOIDP)attrval.c_str());
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 529, "Could not write units for Night field"); 
  attrcount[0] = lfieldname.size();
  status = HE5_GDwritelocattr (gdid,fieldname.c_str(),"long_name",H5T_NATIVE_CHAR, attrcount,(VOIDP)lfieldname.c_str());
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 529, "Could not write long_name for Night field"); 

  return status;
}

int mop03_file :: defFieldI ( string fn, string dimlist, string attrval, string lfn )
{
  string fieldname, lfieldname;
  intn compparm[5]; 
  int status = 1;
  int tempfill;
  hsize_t tiledims[2], attrcount[3];

  compparm[0] = 5;
  tempfill = -9999;
  tiledims[0] = 60;
  tiledims[1] = 30;
  attrcount[0] = attrval.size();

  fieldname = fn + "Day";
  lfieldname = lfn + " Day";
  status = HE5_GDsetfillvalue(gdid, fieldname.c_str(), H5T_NATIVE_INT32, &tempfill);
  if ( status != 0 ) 	
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 530, "Could not set Day fill value");   
  status = HE5_GDdeftile(gdid, HE5_HDFE_TILE, 2, tiledims);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 531, "Could not def Day tile"); 
  status = HE5_GDdefcomp(gdid, HE5_HDFE_COMP_DEFLATE, compparm);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 532, "Could not def Day compression");          
  status = HE5_GDdeffield (gdid, fieldname.c_str(), ((char*)dimlist.c_str()), NULL, H5T_NATIVE_INT32, HE5_HDFE_NOMERGE);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 533, "Could not def Day field"); 
  status = HE5_GDwritelocattr (gdid, fieldname.c_str(), "units", H5T_NATIVE_CHAR, attrcount, (VOIDP)attrval.c_str());
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 534, "Could not write units for Day field"); 
  attrcount[0] = lfieldname.size();
  status = HE5_GDwritelocattr (gdid,fieldname.c_str(),"long_name",H5T_NATIVE_CHAR, attrcount,(VOIDP)lfieldname.c_str());
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 524, "Could not write long_name for Day field"); 

  fieldname = fn + "Night";
  lfieldname = lfn + " Night";
  status = HE5_GDsetfillvalue(gdid, fieldname.c_str(), H5T_NATIVE_INT32, &tempfill);
  if ( status != 0 ) 	
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 535, "Could not set Night fill value"); 
  status = HE5_GDdeftile(gdid, HE5_HDFE_TILE, 2, tiledims);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 536, "Could not def Night tile");
  status = HE5_GDdefcomp(gdid, HE5_HDFE_COMP_DEFLATE, compparm);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 537, "Could not def Night compression");   
  status = HE5_GDdeffield (gdid, fieldname.c_str(), ((char*)dimlist.c_str()), NULL, H5T_NATIVE_INT32, HE5_HDFE_NOMERGE);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 538, "Could not def Night field"); 
  status = HE5_GDwritelocattr (gdid, fieldname.c_str(), "units", H5T_NATIVE_CHAR, attrcount, (VOIDP)attrval.c_str());
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 539, "Could not write units for Night field"); 
  attrcount[0] = lfieldname.size();
  status = HE5_GDwritelocattr (gdid,fieldname.c_str(),"long_name",H5T_NATIVE_CHAR, attrcount,(VOIDP)lfieldname.c_str());
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 529, "Could not write long_name for Night field"); 
  
  return status;
}

int mop03_file :: defFieldF3 ( string fn, string dimlist, string attrval, string lfn, int tile290 )
{
  string fieldname, lfieldname;
  int compparm[5];  
  int status = 1;
  float tempfill;
  hsize_t tiledims[3], attrcount[3];

  compparm[0] = 5;
  tempfill = -9999.0;
  tiledims[0] = 60;
  tiledims[1] = 30;
  tiledims[2] = 9;
  if ( tile290 == 2 ) 
    tiledims[2] = 2;
  else if ( tile290 == 10 ) 
    tiledims[2] = 10;
  attrcount[0] = attrval.size();

  fieldname = fn  + "Day";
  lfieldname = lfn + " Day";
  status = HE5_GDsetfillvalue(gdid, fieldname.c_str(), H5T_NATIVE_FLOAT, &tempfill);
  if ( status != 0 ) 	
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 520, "Could not set Day fill value");
  status = HE5_GDdeftile(gdid, HE5_HDFE_TILE, 3, tiledims);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 521, "Could not def Day tile");
  status = HE5_GDdefcomp(gdid, HE5_HDFE_COMP_DEFLATE, compparm);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 522, "Could not def Day compression");   
  status = HE5_GDdeffield (gdid, fieldname.c_str(), ((char*)dimlist.c_str()), NULL, H5T_NATIVE_FLOAT, HE5_HDFE_NOMERGE);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 523, "Could not def Day field"); 
  status = HE5_GDwritelocattr (gdid, fieldname.c_str(), "units", H5T_NATIVE_CHAR, attrcount, (VOIDP)attrval.c_str());
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 524, "Could not write units for Day field");
  attrcount[0] = lfieldname.size();
  status = HE5_GDwritelocattr (gdid,fieldname.c_str(),"long_name",H5T_NATIVE_CHAR, attrcount,(VOIDP)lfieldname.c_str());
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 524, "Could not write long_name for Day field");  

  fieldname = fn + "Night";
  lfieldname = lfn + " Night";
  status = HE5_GDsetfillvalue(gdid, fieldname.c_str(), H5T_NATIVE_FLOAT, &tempfill);
  if ( status != 0 ) 	
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 525, "Could not set Night fill value"); 
  status = HE5_GDdeftile(gdid, HE5_HDFE_TILE, 3, tiledims);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 526, "Could not def Night tile");
  status = HE5_GDdefcomp(gdid, HE5_HDFE_COMP_DEFLATE, compparm);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 527, "Could not def Night compression");   
  status = HE5_GDdeffield (gdid, fieldname.c_str(), ((char*)dimlist.c_str()), NULL, H5T_NATIVE_FLOAT, HE5_HDFE_NOMERGE);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 528, "Could not def Night field"); 
  status = HE5_GDwritelocattr (gdid, fieldname.c_str(), "units", H5T_NATIVE_CHAR, attrcount, (VOIDP)attrval.c_str());
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 529, "Could not write units for Night field"); 
  attrcount[0] = lfieldname.size();
  status = HE5_GDwritelocattr (gdid,fieldname.c_str(),"long_name",H5T_NATIVE_CHAR, attrcount,(VOIDP)lfieldname.c_str());
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 529, "Could not write long_name for Night field"); 

  return status;
}

int mop03_file :: defFieldF4 ( string fn, string dimlist, string attrval, string lfn )
{
  string fieldname, lfieldname;
  int compparm[5];  
  int status = 1;
  float tempfill;
  hsize_t tiledims[4], attrcount[3];

  compparm[0] = 5;
  tempfill = -9999.0;
  tiledims[0] = 60;
  tiledims[1] = 30;
  tiledims[2] = 10;
  tiledims[3] = 10;
  attrcount[0] = attrval.size();

  fieldname = fn  + "Day";
  lfieldname = lfn + " Day";
  status = HE5_GDsetfillvalue(gdid, fieldname.c_str(), H5T_NATIVE_FLOAT, &tempfill);
  if ( status != 0 ) 	
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 520, "Could not set Day fill value");
  status = HE5_GDdeftile(gdid, HE5_HDFE_TILE, 4, tiledims);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 521, "Could not def Day tile");
  status = HE5_GDdefcomp(gdid, HE5_HDFE_COMP_DEFLATE, compparm);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 522, "Could not def Day compression");   
  status = HE5_GDdeffield (gdid, fieldname.c_str(), ((char*)dimlist.c_str()), NULL, H5T_NATIVE_FLOAT, HE5_HDFE_NOMERGE);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 523, "Could not def Day field"); 
  status = HE5_GDwritelocattr (gdid, fieldname.c_str(), "units", H5T_NATIVE_CHAR, attrcount, (VOIDP)attrval.c_str());
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 524, "Could not write units for Day field");
  attrcount[0] = lfieldname.size();
  status = HE5_GDwritelocattr (gdid,fieldname.c_str(),"long_name",H5T_NATIVE_CHAR, attrcount,(VOIDP)lfieldname.c_str());
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 524, "Could not write long_name for Day field");  

  fieldname = fn + "Night";
  lfieldname = lfn + " Night";
  status = HE5_GDsetfillvalue(gdid, fieldname.c_str(), H5T_NATIVE_FLOAT, &tempfill);
  if ( status != 0 ) 	
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 525, "Could not set Night fill value"); 
  status = HE5_GDdeftile(gdid, HE5_HDFE_TILE, 4, tiledims);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 526, "Could not def Night tile");
  status = HE5_GDdefcomp(gdid, HE5_HDFE_COMP_DEFLATE, compparm);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 527, "Could not def Night compression");   
  status = HE5_GDdeffield (gdid, fieldname.c_str(), ((char*)dimlist.c_str()), NULL, H5T_NATIVE_FLOAT, HE5_HDFE_NOMERGE);
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 528, "Could not def Night field"); 
  status = HE5_GDwritelocattr (gdid, fieldname.c_str(), "units", H5T_NATIVE_CHAR, attrcount, (VOIDP)attrval.c_str());
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 529, "Could not write units for Night field"); 
  attrcount[0] = lfieldname.size();
  status = HE5_GDwritelocattr (gdid,fieldname.c_str(),"long_name",H5T_NATIVE_CHAR, attrcount,(VOIDP)lfieldname.c_str());
  if ( status != 0 ) 
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 529, "Could not write long_name for Night field"); 

  return status;
}

int mop03_file :: wrField2F ( string fieldname, float data[XDim][YDim] )
{
  hssize_t start2[2];
  hsize_t edge2[2];
  int status = 1;

  start2[0] = 0;       start2[1] = 0;
  edge2 [0] = XDim;    edge2 [1] = YDim;

  status = HE5_GDwritefield( gdid, fieldname.c_str(), start2, NULL, edge2, data );
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 570, "Could not write 2DF field");

  return status;
}

int mop03_file :: wrField2I ( string fieldname, int data[XDim][YDim])
{
  hssize_t start2[2];
  hsize_t edge2[2];
  int status = 1;

  start2[0] = 0;       start2[1] = 0;
  edge2 [0] = XDim;    edge2 [1] = YDim;

  status = HE5_GDwritefield( gdid, fieldname.c_str(), start2, NULL, edge2, data );
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 572, "Could not write 2DT field");

  return status;
}

int mop03_file :: wrField3F ( string fieldname, float data[XDim][YDim][NTWO])
{
  hssize_t start3[3];
  hsize_t edge3[3];
  int status = 1;

  start3[0] = 0;       start3[1] = 0;        start3[2] = 0;
  edge3 [0] = XDim;    edge3 [1] = YDim;     edge3 [2] = NTWO;

  status = HE5_GDwritefield( gdid, fieldname.c_str(), start3, NULL, edge3, data );
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 573, "Could not write 3DF NTWO field");

  return status;
}

int mop03_file :: wrField3 ( string fieldname, float data[XDim][YDim][Prs])
{
  hssize_t start3[3];
  hsize_t edge3[3];
  int status = 1;

  start3[0] = 0;       start3[1] = 0;        start3[2] = 0;
  edge3 [0] = XDim;    edge3 [1] = YDim;     edge3 [2] = Prs;

  status = HE5_GDwritefield( gdid, fieldname.c_str(), start3, NULL, edge3, data );
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 573, "Could not write 3D Prs field");

  return status;
}
int mop03_file :: wrField3P ( string fieldname, float data[XDim][YDim][Prs1])
{
  hssize_t start3[3];
  hsize_t edge3[3];
  int status = 1;

  start3[0] = 0;       start3[1] = 0;        start3[2] = 0;
  edge3 [0] = XDim;    edge3 [1] = YDim;     edge3 [2] = Prs1;

  status = HE5_GDwritefield( gdid, fieldname.c_str(), start3, NULL, edge3, data );
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 573, "Could not write 3D Prs1 field");

  return status;
}

int mop03_file :: wrField4 ( string fieldname, float data[XDim][YDim][Prs1][Prs2])
{
  hssize_t start4[4];
  hsize_t edge4[4];
  int status = 1;

  start4[0] = 0;       start4[1] = 0;        start4[2] = 0;      start4[3] = 0;
  edge4 [0] = XDim;    edge4 [1] = YDim;     edge4 [2] = Prs1;   edge4 [3] = Prs2;

  status = HE5_GDwritefield( gdid, fieldname.c_str(), start4, NULL, edge4, data );
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 574, "Could not write 4D field");

  return status;
}

bool mop03_file :: Initialize ()
{
  intn status;
  int  tile2, tile9, tile0;
  bool isinit = false;
  PGSt_integer version = 1;
  char filename [PGSd_PC_FILE_PATH_MAX];

  if (PGS_PC_GetReference (M3LOGICAL, &version, filename) != PGS_S_SUCCESS)
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 500, "Could not get MOP03 filename");
  else {
    isinit = true;
    name = filename;
  }

  if (isinit) {

    float64 upleft[2], lowright[2];
    upleft[0] = -180000000.00;    lowright[0] = 180000000.00;
    upleft[1] =  -90000000.00;    lowright[1] =  90000000.00;
    tile2 = 2;
    tile9 = 9;
    tile0 = 10;

    if ((fileid = HE5_GDopen ((char*) name.c_str (), H5F_ACC_TRUNC)) == FAIL)
      diagnosticreporter.Write (DIAGNOSTICS_ERROR, 501, "Could not open MOP03");   

    if ( (gdid = HE5_GDcreate(fileid, "MOP03", XDim, YDim, upleft, lowright)) == FAIL)
      diagnosticreporter.Write (DIAGNOSTICS_ERROR, 502, "Could not create MOP03");

    status = HE5_GDdefproj(gdid, GCTP_GEO, NULL, NULL, NULL );
    if ( status == FAIL )
      diagnosticreporter.Write (DIAGNOSTICS_ERROR, 503, "Could not define projectopn for MOP03");

    // set up dimensions XDim=360, YDim=180, Prs=9, Prs1=10, Prs2=10, NTWO=2
    status = HE5_GDdefdim(gdid, "XDim", XDim );
    if ( status != 0 ) 
      diagnosticreporter.Write (DIAGNOSTICS_ERROR, 504, "Could not set up XDim");
    status = HE5_GDdefdim(gdid, "YDim", YDim );
    if ( status != 0 ) 
      diagnosticreporter.Write (DIAGNOSTICS_ERROR, 505, "Could not set up YDim");
    status = HE5_GDdefdim(gdid, "Prs", Prs);
    if ( status != 0 ) 
      diagnosticreporter.Write (DIAGNOSTICS_ERROR, 506, "Could not set up Prs");
    status = HE5_GDdefdim(gdid, "Prs1", Prs1);
    if ( status != 0 ) 
      diagnosticreporter.Write (DIAGNOSTICS_ERROR, 507, "Could not set up Prs1");
    status = HE5_GDdefdim(gdid, "Prs2", Prs2);
    if ( status != 0 ) 
      diagnosticreporter.Write (DIAGNOSTICS_ERROR, 508, "Could not set up Prs2");
    status = HE5_GDdefdim(gdid, "NTWO", NTWO);
    if ( status != 0 ) 
      diagnosticreporter.Write (DIAGNOSTICS_ERROR, 509, "Could not set up NTWO");

    // set up geo-data fileds
    status = defFieldG  ("Latitude", "YDim",  "degrees_north");
    status = defFieldG  ("Longitude", "XDim", "degrees_east");
    status = defFieldG  ("Pressure",  "Prs",  "hPa");
    status = defFieldG  ("Pressure2", "Prs2", "hPa");

    // set up data fileds
    status = defFieldF  ("SolarZenithAngle", "XDim,YDim", "deg","Solar Zenith Angle");
    status = defFieldF  ("SatelliteZenithAngle", "XDim,YDim", "deg","Satellite Zenith Angle");
    status = defFieldF  ("SurfacePressure", "XDim,YDim", "hPa","Surface Pressure");
    status = defFieldF  ("RetrievedSurfaceTemperature", "XDim,YDim", "K","Retrieved Surface Temperature");
    status = defFieldF  ("RetrievedSurfaceTemperatureMeanUncertainty", "XDim,YDim", "K",
			 "Retrieved Surface Temperature Mean Uncertainty");
    status = defFieldF  ("RetrievedSurfaceTemperatureVariability", "XDim,YDim", "K",
			 "Retrieved Surface Temperature Variability");
    status = defFieldF  ("RetrievedSurfaceEmissivity", "XDim,YDim","NA","Retrieved Surface Emissivity");
    status = defFieldF  ("RetrievedSurfaceEmissivityMeanUncertainty", "XDim,YDim", "NA",
			 "Retrieved Surface Emissivity Mean Uncertainty");
    status = defFieldF  ("RetrievedSurfaceEmissivityVariability", "XDim,YDim", "NA",
			 "Retrieved Surface Emissivity Variability");
    status = defFieldF3 ("RetrievedCOMixingRatioProfile", "XDim,YDim,Prs","ppbv", 
			 "Retrieved CO Mixing Ratio Profile",tile9);
    status = defFieldF3 ("RetrievedCOMixingRatioProfileMeanUncertainty", "XDim,YDim,Prs", "ppbv",
			 "Retrieved CO Mixing Ratio Profile Mean Uncertainty", tile9);
    status = defFieldF3 ("RetrievedCOMixingRatioProfileVariability", "XDim,YDim,Prs", "ppbv", 
			 "Retrieved CO Mixing Ratio Profile Variability", tile9);
    status = defFieldF  ("RetrievedCOSurfaceMixingRatio", "XDim,YDim", "ppbv", "Retrieved CO Surface Mixing Ratio");
    status = defFieldF  ("RetrievedCOSurfaceMixingRatioMeanUncertainty", "XDim,YDim", "ppbv",
			 "Retrieved CO Surface Mixing Ratio Mean Uncertainty");
    status = defFieldF  ("RetrievedCOSurfaceMixingRatioVariability", "XDim,YDim", "ppbv",
			 "Retrieved CO Surface Mixing Ratio Variability");
    status = defFieldF  ("RetrievedCOTotalColumn", "XDim,YDim", "mol/cm^2", "Retrieved CO Total Column");
    status = defFieldF  ("RetrievedCOTotalColumnMeanUncertainty", "XDim,YDim", "mol/cm^2",
			 "Retrieved CO Total Column Mean Uncertainty");
    status = defFieldF  ("RetrievedCOTotalColumnVariability", "XDim,YDim", "mol/cm^2",
			 "Retrieved CO Total Column Variability");
    status = defFieldF3 ("RetrievedCOTotalColumnDiagnostics", "XDim,YDim,NTWO", "mol/cm^2", 
			 "Retrieved CO Total Column Diagnostics", tile2);
    status = defFieldF4 ("RetrievalAveragingKernelMatrix", "XDim,YDim,Prs1,Prs2", "NA",
			 "Retrieval Averaging Kernel Matrix");
    status = defFieldF  ("APrioriSurfaceTemperature", "XDim,YDim", "K","A Priori Surface Temperature");
    status = defFieldF  ("APrioriSurfaceEmissivity", "XDim,YDim", "NA", "A Priori Surface Emissivity");
    status = defFieldF3 ("APrioriCOMixingRatioProfile", "XDim,YDim,Prs", "ppbv", 
			 "A Priori CO Mixing Ratio Profile",tile9);
    status = defFieldF  ("APrioriCOSurfaceMixingRatio", "XDim,YDim", "ppbv","A Priori CO Surface Mixing Ratio");
    status = defFieldF  ("APrioriCOTotalColumn", "XDim,YDim", "mol/cm^2", "A Priori CO Total Column");
    status = defFieldF  ("DegreesofFreedomforSignal", "XDim,YDim", "NA", "Degrees of Freedom for Signal");
    status = defFieldI  ("SurfaceIndex", "XDim,YDim", "NA", "Surface Index");
    status = defFieldI  ("NumberofPixels", "XDim,YDim", "NA", "Number of Pixel");
    status = defFieldF  ("DEMAltitude", "XDim,YDim", "m", "DEM Altitude");
    status = defFieldF  ("DEMAltitudeVariability", "XDim,YDim", "m", "DEM Altitude Variability");
    status = defFieldF  ("SignalChi2", "XDim,YDim", "NA", "Signal Chi2");
    status = defFieldF  ("SignalChi2Variability", "XDim,YDim", "NA", "Signal Chi2 Variability");

    status = defFieldF  ("DryAirColumn", "XDim,YDim", "mol/cm^2", "Dry Air Column");
    status = defFieldF  ("WaterVaporColumn", "XDim,YDim", "mol/cm^2", "Water Vapor Column");
    status = defFieldF3 ("TotalColumnAveragingKernel", "XDim,YDim,Prs1", "mol/(cm^2 log(VMR))", 
			 "Total Column Averaging Kernel",tile0);
    status = defFieldF4 ("MeasurementErrorCovarianceMatrix", "XDim,YDim,Prs1,Prs2", "NA",
			 "Measurement Error Covariance Matrix");
    status = defFieldF4 ("SmoothingErrorCovarianceMatrix", "XDim,YDim,Prs1,Prs2", "NA",
			 "Smoothing Error Covariance Matrix");
    status = defFieldF4 ("RetrievalErrorCovarianceMatrix", "XDim,YDim,Prs1,Prs2", "NA", 
			 "Retrieval Error Covariance Matrix");
  }     
  return isinit;
}


void  mop03_file :: Write ( Product Prod )
{
  intn status;
  int32 i;
  string attrname;
  hsize_t attrcount[3];
  hssize_t start1[1];
  hsize_t edge1[1], count[1];

  hsize_t dtwo, dprs, dprs1, dprs2, dlat, dlon;
  int32 ctwo[NTWO] = {0, 1};
  float mopittlev[Prs] = { 900.0, 800.0, 700.0, 600.0, 500.0, 400.0, 300.0, 200.0, 100.0 };
  float mopittlev2[Prs2] = { 1000.0, 900.0, 800.0, 700.0, 600.0, 500.0, 400.0, 300.0, 200.0, 100.0 };
  float lat[YDim],lon[XDim];
  char dimattr[50];

  for ( i=0; i<YDim; i++)
    lat[i] = i - 89.5;
  for ( i=0; i<XDim; i++)
    lon[i] = i - 179.5;

  
  //def dim scales
  dtwo = 2;
  status = HE5_GDdefdimscale(gdid, "NTWO", dtwo, HE5T_NATIVE_INT, ctwo);
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 550, "Could not defdim NTWO");

  dlat = 180;
  status = HE5_GDdefdimscale(gdid, "YDim", dlat, HE5T_NATIVE_DOUBLE, NULL);
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 551, "Could not defdim YDim");
  count[0]= 15;
  (void) strcpy (dimattr, "degrees_north");
  status = HE5_GDwritedscaleattr(gdid, "YDim", "units", H5T_NATIVE_CHAR, count, dimattr);
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 551, "Could not write dim attr for YDim");

  dlon = 360;
  status = HE5_GDdefdimscale(gdid, "XDim", dlon, HE5T_NATIVE_DOUBLE, NULL);
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 552, "Could not defdim XDim");
  count[0]= 14;
  (void) strcpy (dimattr, "degrees_east");
  status = HE5_GDwritedscaleattr(gdid, "XDim", "units", H5T_NATIVE_CHAR, count,  dimattr); 
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 552, "Could not write dim attr for XDim");

  dprs = 9;
  status = HE5_GDdefdimscale(gdid, "Prs", dprs, HE5T_NATIVE_FLOAT, mopittlev);
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 553, "Could not defdim Prs");
  count[0]= 5;
  (void) strcpy (dimattr, "hPa");
  status = HE5_GDwritedscaleattr(gdid, "Prs", "units", H5T_NATIVE_CHAR, count, dimattr);
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 553, "Could not write dim attr for Prs");

  dprs1 = 10;
  status = HE5_GDdefdimscale(gdid, "Prs1", dprs1, HE5T_NATIVE_FLOAT, mopittlev2);
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 554, "Could not defdim Prs1");
  status = HE5_GDwritedscaleattr(gdid, "Prs1", "units", H5T_NATIVE_CHAR, count, dimattr);
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 554, "Could not write dim attr for Prs1");

  dprs2 = 10;
  status = HE5_GDdefdimscale(gdid, "Prs2", dprs2, HE5T_NATIVE_FLOAT, mopittlev2);
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 555, "Could not defdim Prs2");
  status = HE5_GDwritedscaleattr(gdid, "Prs2", "units", H5T_NATIVE_CHAR, count, dimattr);
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 555, "Could not write dim attr for Prs2");


  // write file attributes
  attrname = "MOPITT Level 3 Monthly File";
  attrcount[0] = attrname.size();
  status = HE5_EHwriteglbattr (fileid, "title", H5T_NATIVE_CHAR, attrcount, (VOIDP)attrname.c_str());
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 560, "Could not write title");
  attrname = "MOPITT at ACOM of NCAR";
  attrcount[0] = attrname.size();
  status = HE5_EHwriteglbattr (fileid, "institution", H5T_NATIVE_CHAR, attrcount, (VOIDP)attrname.c_str());
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 560, "Could not write institution");

  count[0] = 1;
  status = HE5_EHwriteglbattr (fileid, "StartTime", H5T_NATIVE_DOUBLE, count, (VOIDP) &Prod.Fstarttime);
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 560, "Could not write Start Time");
  status = HE5_EHwriteglbattr (fileid, "StopTime",  H5T_NATIVE_DOUBLE, count, (VOIDP) &Prod.Fstoptime);
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 560, "Could not write Stop Time");
  float fillvalue = -9999.0;
  status = HE5_EHwriteglbattr (fileid, "FillValue", H5T_NATIVE_FLOAT, count, (VOIDP) &fillvalue);
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 560, "Could not write Fill Value.");

  // write lat / lon
  start1[0] = 0;   edge1[0] = YDim; 
  status = HE5_GDwritefield( gdid, "Latitude", start1, NULL, edge1, lat );
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 561, "Could not write Latitude");
  start1[0] = 0;   edge1[0] = XDim; 
  status = HE5_GDwritefield( gdid, "Longitude", start1, NULL, edge1, lon );
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 562, "Could not write Longitude");

  // write Pressure Grid -- standard 9 & 10 levels
  start1[0] = 0;   edge1[0] = Prs; 
  status = HE5_GDwritefield( gdid, "Pressure", start1, NULL, edge1, mopittlev );
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 563, "Could not write Pressure Grid");
  start1[0] = 0;   edge1[0] = Prs2; 
  status = HE5_GDwritefield( gdid, "Pressure2", start1, NULL, edge1, mopittlev2 );
  if ( status != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 563, "Could not write Pressure Grid 2");

  // write date fields 
  status = wrField2F("SolarZenithAngleDay", Prod.zensol[0]);
  status = wrField2F("SolarZenithAngleNight", Prod.zensol[1]);
  status = wrField2F("SatelliteZenithAngleDay", Prod.zensat[0]);
  status = wrField2F("SatelliteZenithAngleNight", Prod.zensat[1]);
  status = wrField2F("SurfacePressureDay", Prod.psurf[0]);
  status = wrField2F("SurfacePressureNight", Prod.psurf[1]);

  status = wrField2F("RetrievedSurfaceTemperatureDay", Prod.tsurf[0][0]);
  status = wrField2F("RetrievedSurfaceTemperatureNight", Prod.tsurf[1][0]);
  status = wrField2F("RetrievedSurfaceTemperatureMeanUncertaintyDay", Prod.tsurf[0][1]);
  status = wrField2F("RetrievedSurfaceTemperatureMeanUncertaintyNight", Prod.tsurf[1][1]);
  status = wrField2F("RetrievedSurfaceTemperatureVariabilityDay", Prod.tsurf[0][2]);
  status = wrField2F("RetrievedSurfaceTemperatureVariabilityNight", Prod.tsurf[1][2]);
  status = wrField2F("RetrievedSurfaceEmissivityDay", Prod.esurf[0][0]);
  status = wrField2F("RetrievedSurfaceEmissivityNight", Prod.esurf[1][0]);
  status = wrField2F("RetrievedSurfaceEmissivityMeanUncertaintyDay", Prod.esurf[0][1]);
  status = wrField2F("RetrievedSurfaceEmissivityMeanUncertaintyNight", Prod.esurf[1][1]);
  status = wrField2F("RetrievedSurfaceEmissivityVariabilityDay", Prod.esurf[0][2]);
  status = wrField2F("RetrievedSurfaceEmissivityVariabilityNight", Prod.esurf[1][2]);
 
  status = wrField3("RetrievedCOMixingRatioProfileDay", Prod.co_mix[0][0]);
  status = wrField3("RetrievedCOMixingRatioProfileNight", Prod.co_mix[1][0]);
  status = wrField3("RetrievedCOMixingRatioProfileMeanUncertaintyDay", Prod.co_mix[0][1]);
  status = wrField3("RetrievedCOMixingRatioProfileMeanUncertaintyNight", Prod.co_mix[1][1]);
  status = wrField3("RetrievedCOMixingRatioProfileVariabilityDay", Prod.co_mix[0][2]);
  status = wrField3("RetrievedCOMixingRatioProfileVariabilityNight", Prod.co_mix[1][2]);

  status = wrField2F("RetrievedCOSurfaceMixingRatioDay", Prod.co_surf[0][0]);
  status = wrField2F("RetrievedCOSurfaceMixingRatioNight", Prod.co_surf[1][0]);
  status = wrField2F("RetrievedCOSurfaceMixingRatioMeanUncertaintyDay", Prod.co_surf[0][1]);
  status = wrField2F("RetrievedCOSurfaceMixingRatioMeanUncertaintyNight", Prod.co_surf[1][1]);
  status = wrField2F("RetrievedCOSurfaceMixingRatioVariabilityDay", Prod.co_surf[0][2]);
  status = wrField2F("RetrievedCOSurfaceMixingRatioVariabilityNight", Prod.co_surf[1][2]);

  status = wrField2F("RetrievedCOTotalColumnDay", Prod.co_colm[0][0]);
  status = wrField2F("RetrievedCOTotalColumnNight", Prod.co_colm[1][0]);
  status = wrField2F("RetrievedCOTotalColumnMeanUncertaintyDay", Prod.co_colm[0][1]);
  status = wrField2F("RetrievedCOTotalColumnMeanUncertaintyNight", Prod.co_colm[1][1]);
  status = wrField2F("RetrievedCOTotalColumnVariabilityDay", Prod.co_colm[0][2]);
  status = wrField2F("RetrievedCOTotalColumnVariabilityNight", Prod.co_colm[1][2]);
  status = wrField3F("RetrievedCOTotalColumnDiagnosticsDay", Prod.co_colm_diag[0]);
  status = wrField3F("RetrievedCOTotalColumnDiagnosticsNight", Prod.co_colm_diag[1]);

  status = wrField4("RetrievalAveragingKernelMatrixDay", Prod.avgker[0]);
  status = wrField4("RetrievalAveragingKernelMatrixNight", Prod.avgker[1]);

  status = wrField2F("APrioriSurfaceTemperatureDay", Prod.ap_tsurf[0]);
  status = wrField2F("APrioriSurfaceTemperatureNight", Prod.ap_tsurf[1]);
  status = wrField2F("APrioriSurfaceEmissivityDay", Prod.ap_esurf[0]);
  status = wrField2F("APrioriSurfaceEmissivityNight", Prod.ap_esurf[1]);
  status = wrField3("APrioriCOMixingRatioProfileDay", Prod.ap_co_mix[0]);
  status = wrField3("APrioriCOMixingRatioProfileNight", Prod.ap_co_mix[1]);
  status = wrField2F("APrioriCOSurfaceMixingRatioDay", Prod.ap_co_surf[0]);
  status = wrField2F("APrioriCOSurfaceMixingRatioNight", Prod.ap_co_surf[1]);
  status = wrField2F("APrioriCOTotalColumnDay", Prod.ap_co_tot[0]);
  status = wrField2F("APrioriCOTotalColumnNight", Prod.ap_co_tot[1]);

  status = wrField2F("DegreesofFreedomforSignalDay", Prod.DFS[0]); 
  status = wrField2F("DegreesofFreedomforSignalNight", Prod.DFS[1]); 
  status = wrField2I("SurfaceIndexDay", Prod.isurf[0]);
  status = wrField2I("SurfaceIndexNight", Prod.isurf[1]);
  status = wrField2I("NumberofPixelsDay", Prod.nco_colm[0]); 
  status = wrField2I("NumberofPixelsNight", Prod.nco_colm[1]);

  status = wrField2F("DEMAltitudeDay", Prod.alt[0][0]);
  status = wrField2F("DEMAltitudeNight", Prod.alt[1][0]);
  status = wrField2F("DEMAltitudeVariabilityDay", Prod.alt[0][1]);
  status = wrField2F("DEMAltitudeVariabilityNight", Prod.alt[1][1]);
  status = wrField2F("SignalChi2Day", Prod.chi[0][0]);
  status = wrField2F("SignalChi2Night", Prod.chi[1][0]);
  status = wrField2F("SignalChi2VariabilityDay", Prod.chi[0][1]);
  status = wrField2F("SignalChi2VariabilityNight", Prod.chi[1][1]);

  status = wrField2F("DryAirColumnDay", Prod.da_col[0]);
  status = wrField2F("DryAirColumnNight", Prod.da_col[1]);
  status = wrField2F("WaterVaporColumnDay", Prod.wa_col[0]);
  status = wrField2F("WaterVaporColumnNight", Prod.wa_col[1]);
  status = wrField3P("TotalColumnAveragingKernelDay", Prod.ak_col[0]);
  status = wrField3P("TotalColumnAveragingKernelNight", Prod.ak_col[1]);
  status = wrField4 ("MeasurementErrorCovarianceMatrixDay", Prod.me_cov[0]);
  status = wrField4 ("MeasurementErrorCovarianceMatrixNight", Prod.me_cov[1]);
  status = wrField4 ("SmoothingErrorCovarianceMatrixDay", Prod.sm_cov[0]);
  status = wrField4 ("SmoothingErrorCovarianceMatrixNight", Prod.sm_cov[1]);
  status = wrField4 ("RetrievalErrorCovarianceMatrixDay", Prod.re_cov[0]);
  status = wrField4 ("RetrievalErrorCovarianceMatrixNight", Prod.re_cov[1]);
}


void mop03_file :: Close (double starttime, double stoptime)
{
  
  // detach from the grid group
  if (HE5_GDdetach (gdid) == FAIL) 
    diagnosticreporter.Write (DIAGNOSTICS_WARNING, 560, "Could not detach MOP03 hdf." );
  else if (HE5_GDclose(fileid) == FAIL )
    diagnosticreporter.Write (DIAGNOSTICS_WARNING, 562, "Could not close MOP03 hdf file." );  
  else {

    //MOP03 meta data
    mop03_metadata mop03metadata;
    mop03metadata.Write ( starttime, stoptime, GetMetadataID () );
     
    diagnosticreporter.Write (DIAGNOSTICS_WARNING, 564, "MOP03 file is closed." );
  }
}
