/************************ MOP02_Read_Geo.C ****************************************************************************\

 PURPOSE
   To read Geo data from MOP02.hdf

 REVISION HISTORY
   02/09   Debbie Mao

\**********************************************************************************************************************/
#include "MOP02.h"
#include "DiagnosticReporter.h"
#include <string>

extern diagnostic_reporter diagnosticreporter;

using namespace std;

bool MOP02_Read_Geo( int f, double &starttime, double &stoptime, MOP02_Geo &Geo )

{
  bool   isread = false;
  intn   errstat;
  hid_t  fileid, swid;
  hssize_t start1[1], start2[2];
  hsize_t  edge1[1],  edge2[2];
  char   filename [PGSd_PC_FILE_PATH_MAX];
  double time[NTIME];
  PGSt_integer version=1;
  int ierr;

  // initialize the arrays
  for ( int t = 0; t < NTIME; t++) {
    Geo.lat[t] = MISSING_VALUE;
    Geo.lon[t] = MISSING_VALUE;
    Geo.solzen[t] = MISSING_VALUE;
    Geo.presurf[t]= MISSING_VALUE;
    Geo.indsurf[t]= MISSING_INT;
  }
  starttime = MISSING_VALUE;
  stoptime  = MISSING_VALUE;
  Geo.timecount = MISSING_INT;
  version = f + 1;

  // read the MOP02 file name
  if (PGS_PC_GetReference (M2LOGICAL, &version, filename) != PGS_S_SUCCESS){
    diagnosticreporter.Write (DIAGNOSTICS_WARNING, 300, "Could not get MOP02 filename");
    return isread;
  }

  // open the MOP02 file 
  if ((fileid = HE5_SWopen ((char*) filename, H5F_ACC_RDONLY)) == FAIL) {
    diagnosticreporter.Write (DIAGNOSTICS_WARNING, 302, "Could not open MOP02 file");
    return isread;
  }

  // read Time Count
  errstat = HE5_EHreadglbattr(fileid, "TimeCount", (VOIDP) &Geo.timecount);	
  if (errstat == FAIL || Geo.timecount < (int32) 1){
    (void) HE5_SWclose (fileid);
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 204, "Could not read time count from MOP02 file");
    return isread;
  }
  
  // attach swath
  swid = HE5_SWattach (fileid, "MOP02");
  if (swid < (int32) 0) {
    (void) HE5_SWclose (fileid);
    diagnosticreporter.Write (DIAGNOSTICS_WARNING, 304, "Could not attach MOP02 file");
    return isread;
  }

  // read time [NTIME] for Start Time and Stop Time
  start1[0] = 0;   
  edge1 [0] = Geo.timecount;
  ierr = HE5_SWreadfield(swid,"Time",start1, NULL,edge1, (VOIDP)time);
  if ( ierr != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 307, "Could not read Time" );
  else {
    starttime = time[0];
    stoptime  = time[Geo.timecount-1];
  }

  // read latitude [NTIME]
  ierr = HE5_SWreadfield(swid,"Latitude",start1, NULL,edge1, (VOIDP)Geo.lat);
  if ( ierr != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 308, "Could not read Latitude" );
  
  // read longitude [NTIME]
  ierr = HE5_SWreadfield(swid,"Longitude",start1, NULL,edge1, (VOIDP)Geo.lon);
  if ( ierr != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 310, "Could not read Longitude" );
  
  // read Solar Zenith Angle [NTIME]
  ierr = HE5_SWreadfield(swid,"SolarZenithAngle",start1, NULL,edge1, (VOIDP)Geo.solzen);
  if ( ierr != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR,312, "Could not read Solar Zenith Angle" );

  // read Surface Index [NTIME]
  ierr = HE5_SWreadfield(swid,"SurfaceIndex",start1, NULL,edge1, (VOIDP)Geo.indsurf);
  if ( ierr != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 314, "Could not read Surface Index" );

  // read Surface Pressure  [NTIME][0]
  start2[0] = 0;                start2[1] = 0;
  edge2 [0] = Geo.timecount;    edge2 [1] = 1;   
  ierr = HE5_SWreadfield(swid,"SurfacePressure",start2, NULL,edge2, (VOIDP)Geo.presurf) ;
  if ( ierr != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 320, "Could not read Surface Pressure" );

  
  (void) HE5_SWdetach (swid); 
  (void) HE5_SWclose(fileid);

  isread = true;
  return isread;
}
