/************************ MOP02_Read_Data.C ***************************************************************************\

 PURPOSE
   To read data from MOP02.hdf

 REVISION HISTORY
   11/02   Debbie Mao
   03/05   Debbie Mao
   02/09   Debbie Mao
   08/16   DM: read in more varibles from V7 MOP02 files

\**********************************************************************************************************************/
#include "MOP02.h"
#include "DiagnosticReporter.h"
#include <string>

extern diagnostic_reporter diagnosticreporter;

using namespace std;



bool MOP02_Read_Data(int fil, MOP02_Geo &Geo, MOP02_Dat1 &Dat1, MOP02_Dat2 &Dat2)
{
  intn   errstat;
  hid_t  fileid, swid;
  hssize_t start1[1],  start2[2], start3[3];
  hsize_t  edge1[1], edge2[2], edge3[3];
  char   filename [PGSd_PC_FILE_PATH_MAX];
  PGSt_integer version = 1;
  bool isread = false;
  double time[NTIME];
  int    ierr, nt, na, nd;

  // initialize all the variables
  for ( int t = 0; t < NTIME; t++) {
    Geo.lat[t] = MISSING_VALUE;
    Geo.lon[t] = MISSING_VALUE;
    Geo.solzen[t] = MISSING_VALUE;
    Geo.presurf[t]= MISSING_VALUE;
    Geo.indsurf[t]= MISSING_INT;

    Dat1.satzen[t]   = MISSING_VALUE;
    Dat1.ap_tsurf[t] = MISSING_VALUE;
    Dat1.ap_esurf[t] = MISSING_VALUE;
    Dat1.ap_cosurf[t]= MISSING_VALUE;
    Dat1.ap_cotot[t] = MISSING_VALUE;
    Dat1.dfs[t]      = MISSING_VALUE;
    Dat1.dryair_col[t]   = MISSING_VALUE;
    Dat1.watervap_col[t] = MISSING_VALUE;

    for ( int k = 0; k < NTHREE; k++) {
      Dat1.swInd[t][k] = MISSING_INT;
    }

    for ( int np1 = 0; np1 < Prs; np1++){
      Dat1.ap_comix[t][np1] = MISSING_VALUE;
    }
    for ( int np1 = 0; np1 < Prs1; np1++) {
      Dat1.ak_tot[t][np1] = MISSING_VALUE;
      for ( int np2 = 0; np2 < Prs2; np2++) {
	Dat1.aker[t][np1][np2] = MISSING_VALUE;
	Dat1.measure_cov[t][np1][np2] = MISSING_VALUE;
	Dat1.smooth_cov[t][np1][np2] = MISSING_VALUE;
	Dat1.retrival_cov[t][np1][np2] = MISSING_VALUE;
      }
    }

    Dat2.demalt[t] = MISSING_VALUE;
    Dat2.chi2[t]   = MISSING_VALUE;

    for ( int n = 0; n < NTWO; n++) {
      Dat1.co_diag[t][n] = MISSING_VALUE;
      Dat2.tsurf[t][n]   = MISSING_VALUE;
      Dat2.esurf[t][n]   = MISSING_VALUE;
      Dat2.cosurf[t][n]  = MISSING_VALUE;
      Dat2.cotot[t][n]   = MISSING_VALUE;
      for ( int np1 = 0; np1 < Prs; np1++){
	Dat2.comix[t][np1][n] = MISSING_VALUE;
      }
      for ( int nr1 = 0; nr1 < NRES; nr1++){
	Dat1.L1rad[t][nr1][n] = MISSING_VALUE;
      }
    }
  }
  version = fil + 1;

  // read the MOP02 file name
  if (PGS_PC_GetReference (M2LOGICAL, &version, filename) != PGS_S_SUCCESS) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 200, "Could not get MOP02 filename");
    return isread;
  }

  // open MOP02 file 
  if ((fileid = HE5_SWopen ((char*) filename, H5F_ACC_RDONLY)) == FAIL) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 202, "Could not open MOP02 file");
    return isread;
  }

  // read Time Count
  errstat = HE5_EHreadglbattr(fileid, "TimeCount", (VOIDP) &Geo.timecount);	
  if (errstat == FAIL || Geo.timecount < (int32) 1){
    (void) HE5_SWclose (fileid);
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 204, "Could not read time count from MOP02 file");
    return isread;
  }

  // attach swath
  swid = HE5_SWattach (fileid, "MOP02");
  if (swid < (int32) 0) {
    (void) HE5_SWclose (fileid);
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 204, "Could not attach MOP02 file");
    return isread;
  }

  start1[0] = 0;   
  edge1 [0] = Geo.timecount;

  // read latitude [NTIME]
  ierr = HE5_SWreadfield(swid,"Latitude",start1, NULL,edge1, (VOIDP)Geo.lat);
  if ( ierr != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 308, "Could not read Latitude" );
  
  // read longitude [NTIME]
  ierr = HE5_SWreadfield(swid,"Longitude",start1, NULL,edge1, (VOIDP)Geo.lon);
  if ( ierr != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 310, "Could not read Longitude" );
  
  // read Solar Zenith Angle [NTIME]
  ierr = HE5_SWreadfield(swid,"SolarZenithAngle",start1, NULL,edge1, (VOIDP)Geo.solzen);
  if ( ierr != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR,312, "Could not read Solar Zenith Angle" );

  // read Surface Index [NTIME]
  ierr = HE5_SWreadfield(swid,"SurfaceIndex",start1, NULL,edge1, (VOIDP)Geo.indsurf);
  if ( ierr != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 314, "Could not read Surface Index" );

  // read DEM altitude [NTIME]
  ierr = HE5_SWreadfield(swid,"DEMAltitude",start1, NULL,edge1, (VOIDP)Dat2.demalt);
  if ( ierr != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 316, "Could not read DEM Altitude" );

  // read Surface Pressure  [NTIME][0]
  start2[0] = 0;                start2[1] = 0;
  edge2 [0] = Geo.timecount;    edge2 [1] = 1;   
  ierr = HE5_SWreadfield(swid,"SurfacePressure",start2, NULL,edge2, (VOIDP)Geo.presurf) ;
  if ( ierr != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 320, "Could not read Surface Pressure" );

  // read Satllite Zenith Angle [NTIME]
  ierr = HE5_SWreadfield(swid,"SatelliteZenithAngle",start1, NULL,edge1, (VOIDP)Dat1.satzen);
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 211, "Could not read Satellite Zenith Angle" );
    return isread;
  }

  // read DFS [NTIME]
  ierr = HE5_SWreadfield(swid,"DegreesofFreedomforSignal",start1, NULL,edge1, (VOIDP)Dat1.dfs) ;
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 213, "Could not read DFS" );
    return isread;
  }

  // read Signal Chi2 [NTIME]
  ierr = HE5_SWreadfield(swid,"SignalChi2",start1, NULL,edge1, (VOIDP)Dat2.chi2);
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 219, "Could not read Signal Chi2" );
    return isread;
  }

  // read DryAirColumn [NTIME]
  ierr = HE5_SWreadfield(swid,"DryAirColumn",start1, NULL,edge1, (VOIDP)Dat1.dryair_col);
  if ( ierr != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 218, "Could not read DryAirColumn" );

  // read WaterVaporColumn [NTIME]
  ierr = HE5_SWreadfield(swid,"WaterVaporColumn",start1, NULL,edge1, (VOIDP)Dat1.watervap_col);
  if ( ierr != 0 )
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 219, "Could not read WaterVaporColumn" );

  // read A Priori Surface Temperature [0][NTIME]
  ierr = HE5_SWreadfield(swid,"APrioriSurfaceTemperature",start2, NULL,edge2, (VOIDP)Dat1.ap_tsurf) ;
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 216, "Could not read AP Surface Temperature");
    return isread;
  }

  // read A Priori Surface Emissivity [0][NTIME]
  ierr = HE5_SWreadfield(swid,"APrioriSurfaceEmissivity",start2, NULL,edge2, (VOIDP)Dat1.ap_esurf) ;
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 217, "Could not read AP Surface Emissivity");
    return isread;
  }
   
  // read A Priori CO Surface Mixing Ratio [0][NTIME]
  ierr = HE5_SWreadfield(swid,"APrioriCOSurfaceMixingRatio",start2, NULL,edge2, (VOIDP)Dat1.ap_cosurf) ;
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 218, "Could not read AP CO Surface Mixing Ratio");
    return isread;
  }

  // read A Priori CO Total Column [NTIME]
  ierr = HE5_SWreadfield(swid,"APrioriCOTotalColumn",start1, NULL,edge1, (VOIDP)Dat1.ap_cotot) ;
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 218, "Could not read AP CO Total Column");
    return isread;
  }

  start2[0] = 0;                start2[1] = 0;
  edge2 [0] = Geo.timecount;    edge2 [1] = NTWO; 

  // read Retrieved CO Total Column Diag [NTIME][NTWO]
  ierr = HE5_SWreadfield(swid,"RetrievedCOTotalColumnDiagnostics",start2, NULL,edge2, (VOIDP)Dat1.co_diag) ;
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 222, "Could not read Retrieved CO Total Column Diagnostics" );
    return isread;
  }

  // read Retrieved Surface Temperature  [NTIME][NTWO]
  ierr = HE5_SWreadfield(swid,"RetrievedSurfaceTemperature",start2, NULL,edge2, (VOIDP)Dat2.tsurf) ;
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 219, "Could not read Retrieved Surface Temperature");
    return isread;
  }
  
  // read Retrieved Surface Emissivity  [NTIME][NTWO]
  ierr = HE5_SWreadfield(swid,"RetrievedSurfaceEmissivity",start2, NULL,edge2, (VOIDP)Dat2.esurf) ;
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 220, "Could not read Retrieved Surface Emissivity");
    return isread;
  }
 
  // read Retrieved CO Total Column Data [NTWO][NTIME]
  ierr = HE5_SWreadfield(swid,"RetrievedCOTotalColumn",start2, NULL,edge2, (VOIDP)Dat2.cotot) ;
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 221, "Could not read Retrieved CO Total Column" );
    return isread;
  }

  // read Retrieved CO Surface Mixing Ratio [NTIME][NTWO]
  ierr = HE5_SWreadfield(swid,"RetrievedCOSurfaceMixingRatio",start2, NULL,edge2, (VOIDP)Dat2.cosurf) ;
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 223, "Could not read Retrieved CO Surface Mixing Ratio");
    return isread;
  }

  // add Swath Index here for filtering out noisy pixels
  start2[0] = 0;                start2[1] = 0;
  edge2 [0] = Geo.timecount;    edge2 [1] = NTHREE; 
  ierr = HE5_SWreadfield(swid, "SwathIndex", start2, NULL, edge2, (VOIDP)Dat1.swInd);
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 223, "Could not read Swath Index");
    return isread;
  }

  start2[0] = 0;                start2[1] = 0;
  edge2 [0] = Geo.timecount;    edge2 [1] = Prs1;   

  // read TotalColumnAveragingKernel [NTIME][Prs1]
  ierr = HE5_SWreadfield(swid, "TotalColumnAveragingKernel", start2, NULL, edge2, (VOIDP)Dat1.ak_tot);
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 224, "Could not read TotalColumnAveragingKernel");
    return isread;
  }

  start3[0] = 0;                start3[1] = 0;       start3[2] = 0;
  edge3 [0] = Geo.timecount;    edge3 [1] = Prs;     edge3 [2] = 1;

  // read A Priori CO Mixing Ratio Profile [NTIME][Prs][0]
  ierr = HE5_SWreadfield(swid,"APrioriCOMixingRatioProfile",start3, NULL,edge3, (VOIDP)Dat1.ap_comix) ;
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 224, "Could not read AP CO Mixing Ratio Profile" );
    return isread;
  }

  start3[0] = 0;                start3[1] = 0;       start3[2] = 0;
  edge3 [0] = Geo.timecount;    edge3 [1] = Prs;     edge3 [2] = NTWO;

  // read Retrieved CO Mixing Ratio Profile [NTIME][Prs][NTWO]
  ierr = HE5_SWreadfield(swid,"RetrievedCOMixingRatioProfile",start3, NULL,edge3, (VOIDP)Dat2.comix) ;
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 225, "Could not read Retrieved CO Mixing Ratio Profile" );
    return isread;
  } 

  start3[0] = 0;                start3[1] = 0;        start3[2] = 0;
  edge3 [0] = Geo.timecount;    edge3 [1] = Prs1;     edge3 [2] = Prs2;

  // read Retrieval Averaging Kernel Matrix [NTIME][Prs1][Prs2]
  ierr = HE5_SWreadfield(swid,"RetrievalAveragingKernelMatrix",start3, NULL,edge3, (VOIDP)Dat1.aker) ;
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 226, "Could not read Retrieval Averaging Kernel Matrix");
    return isread;
  }

  // read MeasurementErrorCovarianceMatrix [NTIME][Prs1][Prs2]
  ierr = HE5_SWreadfield(swid,"MeasurementErrorCovarianceMatrix",start3, NULL,edge3, (VOIDP)Dat1.measure_cov) ;
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 228, "Could not read MeasurementErrorCovarianceMatrix");
    return isread;
  }

  // read SmoothingErrorCovarianceMatrix [NTIME][Prs1][Prs2]
  ierr = HE5_SWreadfield(swid,"SmoothingErrorCovarianceMatrix",start3, NULL,edge3, (VOIDP)Dat1.smooth_cov) ;
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 229, "Could not read SmoothingErrorCovarianceMatrix");
    return isread;
  }

  // read RetrievalErrorCovarianceMatrix [NTIME][Prs1][Prs2]
  ierr = HE5_SWreadfield(swid,"RetrievalErrorCovarianceMatrix",start3, NULL,edge3, (VOIDP)Dat1.retrival_cov) ;
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 230, "Could not read RetrievalErrorCovarianceMatrix");
    return isread;
  }

  start3[0] = 0;                start3[1] = 0;        start3[2] = 0;
  edge3 [0] = Geo.timecount;    edge3 [1] = NRES;     edge3 [2] = NTWO;

  // read Leverl1 Radiance and Errors [NTIME][NRES][NTWO]
  ierr = HE5_SWreadfield(swid,"Level1RadiancesandErrors",start3, NULL,edge3, (VOIDP)Dat1.L1rad) ;
  if ( ierr != 0 ) {
    diagnosticreporter.Write (DIAGNOSTICS_ERROR, 227, "Could not read Leverl1 Radiance and Errors");
    return isread;
  }


  (void) HE5_SWdetach (swid); 
  (void) HE5_SWclose(fileid);

  isread = true;
  return isread;

}
