/********** L3AvgProcessor.C *******************************************************************************************\

 PURPOSE
   Main driver for L3 Monthly processor.

 ASSUMPTIONS ON ENTRY
   FILES
     The PGE PCF has staged all required files.

 ASSUMPTIONS ON EXIT
   FILES
     MOP03 and MOP3Q files have been produced for this processing data.
   ERROR HANDLING
     Errors detected during PGE run have been written to the LogStatus file.
   MISCELLANEOUS
     PGE information has been written to the LogReport file.

 REVISION HISTORY
   02/11   Debbie Mao -- for V5 new release
   12/12   Debbie Mao -- for V6 in HDF-EOS5 format
   07/13   Debbie Mao -- filter out noisy pixels: TIR & JNT-keep Pix1&2; NIR-keep all pixels
   08/16   Debbie Mao -- for V7 by using Merritt's new pixel filtering method

\**********************************************************************************************************************/
#include "Driver.h" 
#include "MOP02.h" 
#include "Grid.h"
#include "MOP03File.h" 
#include "DiagnosticReporter.h"

// make the diagnostic reporter globally accessible
diagnostic_reporter diagnosticreporter;

using namespace std;

int main ()
{

  MOP02_Geo  M2geo;
  MOP02_Dat1 M2dat1;
  MOP02_Dat2 M2dat2;
  MOP02_Num  M2num;
  MOP03_Num  M3num;
  Product    Prod;  

  double starttime, stoptime;
  int    f, fil;
  float  psurfmaj[DAY][XDim][YDim]; 
  bool   isread = true;
  int    rmPix;

  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 900,"Welcome to MOPITT V6 Monthly Level 3 Processor.");

  // get to be removed pixel number
  isread = Get_rmPixel(rmPix);
  if ( isread ) {
    cout << "removing pixel info:  " << rmPix << endl;
    diagnosticreporter.Write(DIAGNOSTICS_WARNING, 901,"After read rePix info.");
  }
  else {
    diagnosticreporter.Write(DIAGNOSTICS_ERROR, 902, "Failed at reading rmPix switch. STOP the processor!");
  }

  // get the geo information from MOP02.hdf file
  for (f=0; f<NFIL; f++) {
    cout << "file-Geo = " << f << endl;
    isread = MOP02_Read_Geo(f, starttime, stoptime, M2geo);
    if ( isread ) {
      // get the final start and stop time
      if ( f == 0 ) {
	Prod.Fstarttime = starttime;
	Prod.Fstoptime = stoptime;
      }
      else {
	if ( Prod.Fstoptime < stoptime) Prod.Fstoptime = stoptime;
      }
      // For psurfmaj & isurf
      MOP02_Get_Num( f, M2geo, M2num );
    }
    else if ( f > MINFILE )
      break;
    else {
      diagnosticreporter.Write(DIAGNOSTICS_ERROR, 903, "There are not enough daily files. STOP the processor!");
    }
  }
  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 910,"After read geo info.");

  // use the geo info to set the majority surface pressure and majority surface index for each grid
  GetMaj_PsurfInd( M2num, psurfmaj, Prod.isurf );
  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 920,"After GetMaj_PsurfInd.");

  // get info from MOP02.hdf file into mop03 grid
  for (fil=0; fil<f; fil++) {
    isread = MOP02_Read_Data(fil, M2geo, M2dat1, M2dat2);
    if ( isread ) {
      PixeltoGrid (rmPix, fil, psurfmaj, M2geo, M2dat1, M2dat2, M3num, Prod);
    }
    else {
      cout <<"file " << fil << "failed on mop02 read data " << endl;
      diagnosticreporter.Write(DIAGNOSTICS_ERROR, 922, "failed on reading mop02 data, stop!");
    }
  }
  cout << " Pix to Grid -- no: " <<  M3num.nco_totmp[0][0][30][30] <<"  "<< M3num.nco_totmp[0][0][40][40] << endl;
  cout << " Pix to Grid: " << Prod.co_colm[0][0][30][30] <<"  "<< Prod.co_colm[0][0][40][40] << endl;
  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 930, "After PixeltoGrid");

  Avg_Grid ( M3num, Prod );
  cout << " Avg to Grid: " << Prod.co_colm[0][0][30][30] <<"  "<< Prod.co_colm[0][0][40][40] << endl;
  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 940, "After Avg_Grid");

  // get info for std dev
  for (fil=0; fil<f; fil++) {
    isread = MOP02_Read_Dat2(fil, M2geo, M2dat2);
    if ( isread ) {
      Stddev_Grid (rmPix, M2geo, M2dat1, M2dat2, psurfmaj, Prod);
    }
    else {
      cout <<"file " << fil << "failed on mop02 read geo & data2 " << endl;
      diagnosticreporter.Write(DIAGNOSTICS_WARNING, 922, "failed on reading mop02 data2");
    }
  }
  cout << " stddev to Grid: " << Prod.co_colm[0][2][30][30] <<"  "<< Prod.co_colm[0][2][40][40] << endl;
  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 934, "After Stddev_Grid");

  Var_Grid (M3num, Prod);
  cout << " var to Grid: " << Prod.co_colm[0][2][30][30] <<"  "<< Prod.co_colm[0][2][40][40] << endl;
  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 946, "After Var_Grid");

  //initialize MOP03 file
  mop03_file mop03file;
  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 950, "After mop03file initialize");

  //write info in to MOP03 file 
  mop03file.Write( Prod );
  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 960, "After mop03_write");

  // write mop03 metadata file and close mop03 file
  mop03file.Close (Prod.Fstarttime,Prod.Fstoptime);
  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 970, "After mop03_close");

  diagnosticreporter.Write(DIAGNOSTICS_WARNING, 980, "Normal Termination");
  return 0;
}
