!=====================================================================
! ATMOSPHERE & HEALTH EFFECTS FRAMEWORK (AHEF) MODEL: EFFECTS SUBMODEL
!                  COPYRIGHT US-EPA, 2022 
!=====================================================================
! THIS FILE : AHEF_modules.f90
! PURPOSE : all modules used in the AHEF model. Defines variables,
!           dimensions & parameters for sharing among program units.
!=====================================================================
! GLOBAL VERSION: includes O3 data for ALL LATITUDES
! CURRENT SETUP:  US population, geographic, and health data
!=====================================================================
      MODULE GLOBAL
!=====================================================================
! THIS MODULE : GLOBAL
! PURPOSE : Module containing flags and names used throughout
!           AHEF model and submodels
!=====================================================================
      IMPLICIT NONE
      SAVE
!---------------------------------------------------

! flags
      LOGICAL :: eof      ! end-of-file
      LOGICAL :: exf      ! existence-of-file
      LOGICAL :: errflag  ! error flag
      LOGICAL :: keyflag  ! keyword quality/existence
      LOGICAL :: blflag   ! baseline flag
!      LOGICAL :: odsflag  ! if using ODS data summary file
      LOGICAL :: testflag ! do a limited run for test location/population
      LOGICAL :: emiflag  ! call EMISSIONS submodel: calculate emissions
      LOGICAL :: oznflag  ! call ATMOS submodel: calculate ozone
      LOGICAL :: expflag  ! call EXPOSURE submodel: calculate exposure arrays
      LOGICAL :: effflag  ! call EFFECTS submodel: population-dependent effects 
      LOGICAL :: superflag ! allow ozone super-recovery (> 1980 values)
      LOGICAL :: earlyflag ! use early-life weighting (F = whole-life)
      LOGICAL :: extendflag ! extend calculation through existing cohort
                            ! lifetimes after last output year

! flags for effects output design
      LOGICAL :: byyr    ! T/F = output by effects yr / by birth cohort
      LOGICAL :: byage   ! T/F = output by age-yr / sum of all ages
      LOGICAL :: bypop   ! T/F = output by sub-pop / sum of all pops 
      LOGICAL :: bycty   ! T/F = output by county / by state + national

! common counter indices
      INTEGER :: iyear, ilat

! text names assigned in and used throughout the model

      CHARACTER(len=25) :: scenario   ! 
      CHARACTER(len=8)  :: indexname  ! 
      CHARACTER(len=5)  :: drtype     ! 
      CHARACTER(len=12) :: lookupname !
      CHARACTER(len=50) :: expname    !
      CHARACTER(len=3)  :: baftype    ! 
      CHARACTER(len=12) :: endpoint   ! 
      CHARACTER(len=12) :: ageadj     !
      CHARACTER(len=20) :: agename    ! 
      CHARACTER(len=20) :: coeffname  ! 
      CHARACTER(len=20) :: popname    ! 
      CHARACTER(len=50) :: effname    ! 
      CHARACTER(len=50) :: effagename ! 

!---------------------------------------------------
      END MODULE GLOBAL
!---------------------------------------------------

!=====================================================================
      MODULE IOINFO
!=====================================================================
! THIS MODULE : IOINFO
! PURPOSE : Module containing i/o definitions: i/o unit numbers,
!           file locations relative to dir AHEF_MODEL/SOURCE*,
!           filename extensions
!=====================================================================
      IMPLICIT NONE
      SAVE
!----------------------------------------------------------------
! I/O UNITS
!----------------------------------------------------------------

      INTEGER,PARAMETER :: inpunit   = 10 ! global instructions file
      INTEGER,PARAMETER :: logunit   = 20 ! global log file
      INTEGER,PARAMETER :: oznlog    = 25 ! ozone log file
      INTEGER,PARAMETER :: iunit     = 30 ! input
      INTEGER,PARAMETER :: ounit     = 40 ! output
      INTEGER,PARAMETER :: ounit1    = 41 ! output (use if ounit already open)

!----------------------------------------------------------------
! filenames for model instructions input and log output
      CHARACTER(len=12),PARAMETER :: inpfile = "AHEF_RUN.INP"
      CHARACTER(len=8),PARAMETER  :: logfile = "AHEF.LOG"

!----------------------------------------------------------------
! filename for ODS intrisic molecular properties
      CHARACTER(len=13),PARAMETER :: molpropfile = "INTRINSIC.MWT"

!----------------------------------------------------------------
! filenames for TOMS ozone data
      CHARACTER(len=23),PARAMETER :: tomsdatafile = &
                                    "O3data_global_1980.TOMS"
      CHARACTER(len=29),PARAMETER :: tomstrendfile =  &
                                    "O3trend_global_1980-1990.TOMS"
!----------------------------------------------------------------
! filenames for UV irradiance tables
      CHARACTER(len=15),PARAMETER :: scupfile = "SCUP-H.2018.IRR"
      CHARACTER(len=14),PARAMETER :: catafile = "CATAR.2018.IRR"
      CHARACTER(len=12),PARAMETER :: uvifile  = "UVI.2018.IRR"

!----------------------------------------------------------------
! HEALTH INPUT TABLES: EDIT HERE IF USER SUPPLIES NEW FILES
! filename for baseline incidence tables
      CHARACTER(len=11),PARAMETER :: basincfile = "BASEINC.TXT"
! filename for biological amplification factors
      CHARACTER(len=11),PARAMETER :: baffile = "BAF.DAT"
! filename for age-dependent exposure weighting, governed by "EARLY"
! flag in input file
      CHARACTER(len=17),PARAMETER :: wgtwholefile = "EXPWGTS_WHOLE.DAT" !
      CHARACTER(len=17),PARAMETER :: wgtearlyfile = "EXPWGTS_EARLY.DAT" !
      !CHARACTER(len=12) :: cohwgtfile 
!----------------------------------------------------------------
! FILE EXTENSIONS:
!----------------------------------------------------------------
! data: emissions, EESC, ozone, population

      CHARACTER(LEN= 4),PARAMETER :: log_ext=".LOG" ! ODS fract. rel. factors
      CHARACTER(LEN= 4),PARAMETER :: frf_ext=".FRF" ! ODS fract. rel. factors
      CHARACTER(LEN= 4),PARAMETER :: tau_ext=".TAU" ! ODS lifetimes
      CHARACTER(LEN= 4),PARAMETER :: ODS_ext=".ODS" ! all ODS info in 1 file
      CHARACTER(LEN= 4),PARAMETER :: emi_ext=".EMI"
      CHARACTER(LEN= 4),PARAMETER :: esc_ext=".ESC"
      CHARACTER(LEN= 4),PARAMETER :: ozn_ext=".OZN"
      CHARACTER(LEN= 4),PARAMETER :: pop_ext=".pop"

! intermediate output/input (exposure output, effects input)
!                           (scenario, baseline: by age)
      CHARACTER(LEN= 4),PARAMETER :: xpa_ext=".XPA"
      CHARACTER(LEN= 4),PARAMETER :: xpb_ext=".XPB"

! final output: from effects submodel: scenario, baseline, difference
      CHARACTER(LEN= 4),PARAMETER :: efa_ext=".EFA"
      CHARACTER(LEN= 4),PARAMETER :: efb_ext=".EFB"
      CHARACTER(LEN= 4),PARAMETER :: efd_ext=".EFD"

!----------------------------------------------------------------
! DIRECTORIES:
!----------------------------------------------------------------

! instructions file for AHEF model
      CHARACTER(LEN=13),PARAMETER :: dir_in= "../RUN_INPUT/"
! final output: from effects submod
      CHARACTER(LEN=14),PARAMETER :: dir_out="../RUN_OUTPUT/"

! input data: emissions 
      CHARACTER(LEN=17),PARAMETER :: dir_emi="../DATA/EMISSION/"
! input data: ODS parameters
      CHARACTER(LEN=12),PARAMETER :: dir_ods="../DATA/ODS/"
! input data: TOMS ozone distribution & EESC relationship 
      CHARACTER(LEN=13),PARAMETER :: dir_tom="../DATA/TOMS/"
! input data: action spectra-weighted irradiance tables
      CHARACTER(LEN=11),PARAMETER :: dir_irr="../DATA/UV/"
! input data: county latitudes 
      CHARACTER(LEN=19),PARAMETER :: dir_lat="../DATA/POPULATION/"
! input data: population by county
      CHARACTER(LEN=19),PARAMETER :: dir_pop="../DATA/POPULATION/"
! input data: biological weighting functions
      CHARACTER(LEN=15),PARAMETER :: dir_eff="../DATA/HEALTH/"

! generated data: emissions (copies of originals or combined output)
      CHARACTER(LEN=20),PARAMETER :: dir_emo="../DATA_IO/EMISSION/"
! generated data: EESC
      CHARACTER(LEN=16),PARAMETER :: dir_esc="../DATA_IO/EESC/"
! generated data: ozone distributions
      CHARACTER(LEN=17),PARAMETER :: dir_ozn="../DATA_IO/OZONE/"
! generated data: cumulative exposure by age, year, county
      CHARACTER(LEN=17),PARAMETER :: dir_exp ="../DATA_IO/EXPOS/"

!---------------------------------------------------
      END MODULE IOINFO
!---------------------------------------------------

!=====================================================================
      MODULE TESTREFS
!=====================================================================
! THIS MODULE : testrefs
! PURPOSE : Module containing index definitions for reference output
!=====================================================================
!
! Test diagnostic output is for the following conditions:
! (User must specify these in input file AHEF_RUN.INP)
! 
! SCEN WMOA1_cut.EMI
! GLOB WMO2010.GLOB
! FRF  WMO2010.FRF
! TAU  WMO2010.TAU
! ENDPT  MELAINC
! COHYRF 1890
! COHYRL 2100
! OUTYRF 1940 * first output year
! OUTYRL 2100 * last output year
! OUTYRD 1   * output resolution (years)
!
!---------------------------------------------------
      IMPLICIT NONE
      SAVE
!---------------------------------------------------
      INTEGER :: xcty, xlat

!! NB: "light-skinned" =  everyone identifying as other than "black
!only"

!! Test 1)
!! location: Los Angeles, CA
!! sub-population: light-skinned male 
!! county data : cty_fip = 6037; irgn = 6, icty = 72, cty_lat=34.2620
!! incidence data : ilat = 2
      INTEGER,PARAMETER :: xrgn = 6     ! South&West US
      INTEGER,PARAMETER :: xfip = 06037 ! county code for L.A.
      INTEGER,PARAMETER :: xpop = 1     ! LIGHT-SKINNED MALE 

!! Test 2)
!! location: Honolulu
!! sub-population: light-skinned female
!! county data : cty_fip=15003; icty = 34, cty_lat=21.42612
!! incidence data : ilat = 3
!      INTEGER,PARAMETER :: xrgn = 7     ! AK & HI
!      INTEGER,PARAMETER :: xfip = 15003 ! county code for Honolulu
!      INTEGER,PARAMETER :: xpop = 2     ! LIGHT-SKINNED FEMALE

!! Test 3)
!! location: Washington D.C.
!! sub-population: dark-skinned female 
!! county data : cty_fip=11001; irgn = 1, icty = 11, cty_lat=38.891693
!! incidence data : ilat = 2
!      INTEGER,PARAMETER :: xrgn = 1     ! North-East US
!      INTEGER,PARAMETER :: xfip = 11001 ! county code for Washington D.C.
!      INTEGER,PARAMETER :: xpop = 3     ! DARK-SKINNED MALE 

!! Test 4)
!! location: Seattle, WA
!! sub-population: dark-skinned female 
!! county data : cty_fip=53033; irgn = 6, icty = 431, cty_lat=47.467503
!! incidence data : ilat = 1
!      INTEGER,PARAMETER :: xrgn = 6     ! SW & West Coast US
!      INTEGER,PARAMETER :: xfip = 53033 ! county code for Seattle
!      INTEGER,PARAMETER :: xpop = 4     ! DARK-SKINNED FEMALE 

!-------------------------------------------------------!
      END MODULE TESTREFS
!-------------------------------------------------------!

!=====================================================================
      MODULE OZONE
!=====================================================================
! THIS MODULE : OZONE
! PURPOSE : Module containing variables & parameters used in 
!           ozone & emissions calculation (EMISSIONS & ATMOS submodels)
!=====================================================================

      IMPLICIT NONE
      SAVE
!---------------------------------------------------
! standard atmospheric & molar parameters

      REAL,PARAMETER :: matm = 5.148E+18  ! kg
      REAL,PARAMETER :: atm_moles = matm*1e3  &! grammes
                   / (0.72*2.*14.007 + 0.28*2.*15.999) ! / N2+O2 (g/mol)

! bounds for emissions / ozone calculations
      INTEGER,PARAMETER :: nods=20 ! # O3 Depleting Substances considered
      INTEGER,PARAMETER :: myr=300 ! max # of years considered

      INTEGER,PARAMETER :: mlat = 16 ! max latitude bands in ozone output
      INTEGER,PARAMETER :: mmo  = 12 ! months in a year

      REAL,PARAMETER :: mino3 = 100  ! minimum ozone allowed (DU)

! parameters & variables for TOMS regression
      INTEGER :: eescyr
      INTEGER,PARAMETER :: eescyr_default = 1980

      REAL :: bfac
      REAL :: eesc_1980, eesc_1990, eesc_ref

      REAL,DIMENSION(mmo,mlat) :: afac
      REAL,DIMENSION(mmo,mlat) :: O3bsl ! TOMS O3(DU) in 1980 in all lat bands

! filenames for emissions inputs and ODS parameters
      CHARACTER(len=30) :: scenfile   !
      CHARACTER(len=30) :: incrfile   !
      CHARACTER(len=30) :: combfile   !

      CHARACTER(len=30) :: odsfile    ! (ods data summary file)
      CHARACTER(len=30) :: glofile    ! 
      CHARACTER(len=30) :: frffile    ! 
      CHARACTER(len=30) :: taufile    ! 

!---------------------------------------------------
      END MODULE OZONE
!---------------------------------------------------

!=====================================================================
      MODULE GEODATA
!=====================================================================
! THIS MODULE : GEODATA
! PURPOSE : Module containing variables & parameters defining 
!           geographical scope of EXPOSURE and EFFECTS calculations
!=====================================================================
      IMPLICIT NONE
      SAVE
!---------------------------------------------------
! bounds for county and population data

! counties by region: USA-specific
      INTEGER :: icty,irgn,istate,r1,r2
      INTEGER,PARAMETER :: nrgns = 7
      INTEGER,PARAMETER :: maxcty  = 550  ! max counties per US region
      INTEGER,PARAMETER :: maxstate = 15  ! max states per US region

      INTEGER,DIMENSION(nrgns) :: numcty   ! total counties per US region
      INTEGER,DIMENSION(maxcty,nrgns) :: cty_fip  ! fip# for each US county
      REAL,   DIMENSION(maxcty,nrgns) :: cty_lat  ! latitude for each US county

      CHARACTER(len=25) :: ctyfipfile ! filename for county codes

!.....................................................................
! USA-specific geographical information:

! # of states per region
      INTEGER,DIMENSION(nrgns),PARAMETER :: nstate = (/15,5,7,6,9,7,2/) 

! counties in each US state in specified region
! Region 1: North-East
      CHARACTER(LEN=2),DIMENSION(nstate(1)),PARAMETER :: stnam1 = &
                       (/'CT','DE','DC','ME','MD' &
                        ,'MA','NH','NJ','NY','OH' &
                        ,'PA','RI','VT','VA','WV'/)
      INTEGER,DIMENSION(nstate(1)),PARAMETER :: st_id1 =          &
                       (/  9 , 10 , 11 , 23 , 24  &
                        , 25 , 33 , 34 , 36 , 39  &
                        , 42 , 44 , 50 , 51 , 54 /)
! Region 2: South-East
      CHARACTER(LEN=2),DIMENSION(nstate(2)),PARAMETER :: stnam2 = &
                       (/'GA','KY','NC','SC','TN'/)
      INTEGER,DIMENSION(nstate(2)),PARAMETER :: st_id2 =          &
                       (/ 13 , 21 , 37 , 45 , 47 /)
! Region 3: South 
      CHARACTER(LEN=2),DIMENSION(nstate(3)),PARAMETER :: stnam3 = &
                       (/'AL','AR','FL','LA','MS','MO','OK'/)
      INTEGER,DIMENSION(nstate(3)),PARAMETER :: st_id3 =          &
                       (/  1,   5,  12 , 22 , 28 , 29 , 40 /)
! Region 4: Midwest
      CHARACTER(LEN=2),DIMENSION(nstate(4)),PARAMETER :: stnam4 = &
                       (/'IL','IN','IA','MI','MN','WI'/)
      INTEGER,DIMENSION(nstate(4)),PARAMETER :: st_id4 =          &
                       (/ 17 , 18 , 19 , 26 , 27 , 55 /)
! Region 5: W Plains & Rockies
      CHARACTER(LEN=2),DIMENSION(nstate(5)),PARAMETER :: stnam5 = &
                       (/'CO','ID','KS','MT','NE' &
                        ,'ND','SD','UT','WY'/)
      INTEGER,DIMENSION(nstate(5)),PARAMETER :: st_id5 =          &
                       (/  8 , 16 , 20 , 30 , 31  &
                        , 38 , 46 , 49 , 56 /)
! Region 6: SW & W Coast
      CHARACTER(LEN=2),DIMENSION(nstate(6)),PARAMETER :: stnam6 = &
                       (/'AZ','CA','NV','NM','OR','TX','WA'/)
      INTEGER,DIMENSION(nstate(6)),PARAMETER :: st_id6 =          &
                       (/  4 ,  6 , 32 , 35 , 41 , 48 , 53 /)
! Region 7: AK & HI
      CHARACTER(LEN=2),DIMENSION(nstate(7)),PARAMETER :: stnam7 = &
                       (/'AK','HI'/)
      INTEGER,DIMENSION(nstate(7)),PARAMETER :: st_id7 =          &
                       (/  2 , 15 /)

!-------------------------------------------------------!
      END MODULE GEODATA
!-------------------------------------------------------!

!=====================================================================
      MODULE EXPEFF
!=====================================================================
! THIS MODULE : EXPEFF
! PURPOSE : Module containing variables, dimensions & parameters 
!           used in both EXPOSURE & EFFECTS submodels
!=====================================================================
      IMPLICIT NONE
      SAVE
!---------------------------------------------------

      INTEGER :: icoh, icohort, iage, iagey
      INTEGER :: cohi, colo_year, cohi_year

! bounds for age/cohort calculations
      INTEGER,PARAMETER :: colo = 1
      INTEGER,PARAMETER :: step = 5    ! width of cohort & effects age groups
      INTEGER,PARAMETER :: maxage = 85   ! bottom of max age group for effects
      INTEGER,PARAMETER :: maxcohorts = 45
      INTEGER,PARAMETER :: maxages = maxage/step+1 ! number of age groups
      INTEGER,PARAMETER :: topage  = maxage+step   ! top of max age group
      INTEGER,PARAMETER :: iageymax = topage+step-1 ! index for exposure arrays
                                                    ! includes cohort spread
!---------------------------------------------------
      END MODULE EXPEFF
!---------------------------------------------------

!=====================================================================
      MODULE EXPMOD
!=====================================================================
! THIS MODULE : EXPMOD 
! PURPOSE : Module containing variables & parameters used only in
!           EXPOSURE submodel
!=====================================================================

      USE EXPEFF,ONLY: iageymax,maxcohorts
      USE OZONE,ONLY: mlat
      USE GEODATA,ONLY: maxcty

      IMPLICIT NONE
      SAVE
!---------------------------------------------------
      INTEGER :: iday,imonth,itime
      INTEGER :: numlats_o3
      INTEGER :: ozloyr, ozhiyr

! bounds for irradiance calculation
      INTEGER,PARAMETER :: lcols = 21      ! UV lookup table columns
      INTEGER,PARAMETER :: lrows = 51      ! UV lookup table rows
      INTEGER,PARAMETER :: minmon = 1
      INTEGER,PARAMETER :: maxmon = 12
      INTEGER,PARAMETER :: minoyr = 1930
      INTEGER,PARAMETER :: maxoyr = 2150
      INTEGER,PARAMETER :: maxyrs = maxoyr-minoyr+1

      REAL,DIMENSION(mlat) :: latn,lats
      REAL,DIMENSION(maxyrs,maxcty) :: yearlyirrad
      REAL,DIMENSION(maxyrs,maxmon,mlat) :: dobson
      REAL,DIMENSION(lrows,lcols) :: lookup

! by county, by age in 1 yr increments 
! same name is used for both baseline and alternate scenario output
      REAL,DIMENSION(maxcohorts,iageymax,maxcty):: expos_age

!---------------------------------------------------
      END MODULE EXPMOD
!---------------------------------------------------

!=====================================================================
      MODULE EFFMOD
!=====================================================================
! ATMOSPHERE & HEALTH EFFECTS FRAMEWORK (AHEF) MODEL
!                  COPYRIGHT US-EPA, 2019 
!=====================================================================
! THIS MODULE : EFFMOD 
! PURPOSE : Module containing variables & parameters used only in 
!           EFFECTS submodel
!=====================================================================

      USE EXPEFF,ONLY: maxages,iageymax,topage,step,maxcohorts
      USE GEODATA,ONLY: maxcty

      IMPLICIT NONE
      SAVE

!---------------------------------------------------
      INTEGER ipop

!---------------------------------------------------
! bounds for sub-population data
      INTEGER,PARAMETER :: readpops = 6   ! WM,WF,BM,BF,NWM,NWF
      INTEGER,PARAMETER :: evalpops = 4   ! WM,WF,BM,BF 
!(NW -> W for MAXWHITE; NW -> B for MAXBLACK; NW+B->W for ALLWHITE)
      CHARACTER(len=2) :: subpop   ! {WM,WF,BM,BF} expanded text 
      !CHARACTER(len=8) :: pop_scenario="maxblack" 
      CHARACTER(len=8) :: pop_scenario="maxwhite" 
      !CHARACTER(len=8) :: pop_scenario="allwhite" 

!---------------------------------------------------
!--CENSUS DATA YEAR (VERSION) IS HARDWIRED HERE--
!------------------------------------------------
!--POPULATION DATA SECTION (A): FOR GENERAL USE--
!------------------------------------------------
! bounds for population data: 2019 files
      INTEGER,PARAMETER :: minpyr = 1985
      INTEGER,PARAMETER :: maxpyr = 2018
      INTEGER,PARAMETER :: popyrstep = 1  ! population data yr increment
      INTEGER,PARAMETER :: popagestep = 5  ! population age yr increment
      CHARACTER(len=4),PARAMETER :: cpyr = "2019" ! yr in pop filename
! bounds for population projection: 2019 version, for use with 2019 pop files
! 2018 data was added as a summation of 2018 data from regional files R1:R7
      INTEGER,PARAMETER :: minjyr = 2018
      INTEGER,PARAMETER :: maxjyr = 2060
      INTEGER,parameter :: projyrstep = 1  ! population projection yr increment
      CHARACTER(len=4),PARAMETER :: cjyr = "2019" ! yr in proj filename

!-------------------------------------------------------------------------
!--POPULATION DATA SECTION(B): FOR COMPARISON WITH 2015 AHEF REPORT ONLY--
!-------------------------------------------------------------------------
! bounds for population data: original 2010 files as used in 2015 AHEF report
!      INTEGER,PARAMETER :: minpyr = 1985
!      INTEGER,PARAMETER :: maxpyr = 2025
!      INTEGER,parameter :: popyrstep = 5  ! population data increment
!      INTEGER,PARAMETER :: popagestep = 5  ! population age yr increment
!      CHARACTER(len=4),PARAMETER :: cpyr = "2010" ! yr in pop filename
! bounds for population projection: 2008 version, for use with 2008 pop files
! (NB: this projection was NOT used in the 2015 report - values stayed
! constant after 2025 (2015?). This is accomplished via the "return" statement
! at line 237 in subroutine read_population.f90)
!!      INTEGER,PARAMETER :: minjyr = 2010  ! if projection data is available from 2010
!      INTEGER,PARAMETER :: minjyr = 2025  ! if projection data is only available from 2025
!      INTEGER,PARAMETER :: maxjyr = 2050
!      INTEGER,parameter :: projyrstep = 5  ! population projection increment
!      CHARACTER(len=4),PARAMETER :: cjyr = "2008" ! yr in proj filename
!!
!---------------------------------------------------
! POPULATION ARRAYS
      INTEGER,DIMENSION(minpyr:maxjyr,maxages,maxcty,evalpops) :: pop
      INTEGER,DIMENSION(minjyr:maxjyr,maxages,readpops) :: proj

! standard population age-adjustments
      REAL,DIMENSION(maxages) :: stdpop ! age-fraction of standard population
      REAL,DIMENSION(minpyr:maxjyr,maxages) :: popadj ! adjusted population

!---------------------------------------------------
! bounds for cohort calculations (each cohort spans 5 years)
! NB: mincyr quoted as 1887 in old code, but ncyr didn't add 1, so was
! effectively 1886
      INTEGER,PARAMETER :: mincyr = 1890-2
      INTEGER,PARAMETER :: maxcyr = 2100+2
      INTEGER,PARAMETER :: maxcyrs = maxcyr-mincyr+1+iageymax

      INTEGER,PARAMETER :: evalcohs = 4   ! evaluation cohorts
                        ! 1890-1980, 1985-2010, 2015-2050,2055-2100
      INTEGER,DIMENSION(evalcohs),PARAMETER :: &
              coh1 = (/1890,1985,2015,2055/), &
              coh2 = (/1980,2010,2050,2100/)

!---------------------------------------------------
! other bounds
      INTEGER,PARAMETER :: popfac_bi = 100000 ! baseline incidence quoted 
                                              ! per x people where
                                              ! x = popfac_bi
      INTEGER,PARAMETER :: numlats_bi = 3 ! # lats in baseline incidence file
                                          ! lats are in REVERSE order:
                                          ! 1>=40N, 40N>2>=30N, 3<=30N
      INTEGER :: poplo,pophi,outputlo,outputhi,opinterval
      INTEGER,PARAMETER :: opinterval_default = 5

!-------------------------------------------------------!
! names for input/output
      CHARACTER(LEN= 4) :: effsort    ! output sorting descriptor

!---------------------------------------------------
! Input arrays:
! exposure weights by age for current endpoint 
      REAL,DIMENSION(topage) :: expwgt
! Biological Amplification Factors for current endpoint, by population segment
      REAL,DIMENSION(evalpops) :: baf
! baseline incidence by cohort, by age group 
      REAL,DIMENSION(maxcohorts,maxages,numlats_bi,evalpops) :: incid_bl
! baseline incidence by cohort, by age in 1 yr increments
      REAL,DIMENSION(maxcohorts,topage,numlats_bi,evalpops) :: incage_bl

!----------------------------------------------------
! Unused input arrays:
! age coefficients by age group (not currently used)
!      REAL,DIMENSION(maxages) :: age_coeffs
! risk factors by birth cohort (not currently used)
!      REAL,DIMENSION(maxcohorts) :: cohort_risk
! unnamed coefficient by population and age group (not currently used)
!      REAL,DIMENSION(evalpops,step) :: coeff
!----------------------------------------------------

!-Input/output arrays:-------------------------------

! exposure by (birth cohort, age+cohortposition, county)
      REAL,DIMENSION(maxcohorts,iageymax,maxcty) :: expage,expagebl

!-Internal arrays:-----------------------------------

! cases by birth cohort, age-year, county, sub-population
! (baseline, scenario)
      REAL,DIMENSION(maxcohorts,topage,maxcty,evalpops) &
                                                 :: casecb,caseca
! cases by year, age, county & sub-population 
! (baseline, scenario)
      REAL,DIMENSION(maxcyrs,topage,maxcty,evalpops) &
                                                 :: caseyb,caseya
!- Output arrays (nationally aggregated):------------------------------
! cases by birth cohort, age, sub-population (baseline, scenario)
      REAL,DIMENSION(maxcohorts,topage,evalpops) :: totcap_b,totcap_a

! cases by birth cohort, age (baseline, scenario)
      REAL,DIMENSION(maxcohorts,topage) :: totca_b,totca_a

! cases by birth cohort, sub-population (baseline, scenario)
      REAL,DIMENSION(maxcohorts,evalpops) :: totcp_b,totcp_a

! cases by birth cohort (baseline, scenario)
      REAL,DIMENSION(maxcohorts) :: totc_b,totc_a

! cases by cohort group, age-index, sub-population (baseline, scenario)
      REAL,DIMENSION(evalcohs,topage,evalpops) :: totgap_b,totgap_a

! cases by cohort group, age-index (baseline, scenario)
      REAL,DIMENSION(evalcohs,topage) :: totga_b,totga_a

! cases by cohort group, sub-population (baseline, scenario)
      REAL,DIMENSION(evalcohs,evalpops) :: totgp_b,totgp_a

! cases by cohort group (baseline, scenario)
      REAL,DIMENSION(evalcohs) :: totg_b,totg_a

! cases by year, age-index, sub-population (baseline, scenario)
      REAL,DIMENSION(maxcyrs,topage,evalpops) :: totyap_b,totyap_a

! cases by year, age-index (baseline, scenario)
      REAL,DIMENSION(maxcyrs,topage) :: totya_b,totya_a

! cases by year, sub-population (baseline, scenario)
      REAL,DIMENSION(maxcyrs,evalpops) :: totyp_b,totyp_a

! cases by year (baseline, scenario)
      REAL,DIMENSION(maxcyrs) :: toty_b,toty_a

!-------------------------------------------------------!
      END MODULE EFFMOD
!=====================================================================
