************************************************************************
* MASTER MECHANISM - ROUTINE NAME : xcrieg                             *
*                                                                      *
*                                                                      *
* PURPOSE :                                                            *
*    perform the reactions (decomposition, stabilization) of hot       *
*    criegge produced by O3+alkene reactions. The routine returns the  *
*    list of products and associated stoi. coef. in tables p (short    *
*    name) and s, respectively. The dictionary, stack and related      *
*    tables are updated accordingly.                                   *
*                                                                      *
* INPUT:                                                               *
*  - xcri        : formula of the hot criegge                          *
*  - cut_off     : ratio below which a pathway is not considered       *
*  - dbrch       : NOT USED - MORE WORK ON THIS NEEDED                 *
*  - level       : number of level (stable + radicals) that were       *
*                  necessary to produce the parent of rdct             *
*  - stabl       : number of stable level (no radical) that were       *
*                  necessary to produce the parent of rdct             *
*  - nfn         : total nb. of species having a fixed name            *
*  - namfn(i)    : table of the fixed name (6 character)               *
*  - chemfn(i)   : formula corresponding the ith species having a      *
*                  fixed name                                          *
*                                                                      *
* INPUT/OUTPUT                                                         *
*  - dict(j)     : dictionary line (name + formula + functional        *
*                  group info) of species number j                     *
*  - namlst(j)   : name (lco=6 characters) of the species already      *
*                  used  at position number j                          *
*  - nhldvoc     : number of (stable) VOC in the stack                 *
*  - holdvoc(i)  : list of the VOC in the stack                        *
*  - nhldrad     : number of radical in the stack                      *
*  - holdrad(i)  : list of the radicals in the stack                   *
*                                                                      *
* OUTPUT:                                                              *
*  - s(i)        : stoichiometric coef. associated with product i      *
*  - p(i)        : product i                                           *
************************************************************************
      SUBROUTINE xcrieg(xcri,brch,s,p,
     &                  dbrch,dict,namlst,
     &                  cut_off,
     &                  nhldvoc,holdvoc,nhldrad,holdrad,level,stabl,
     &                  nfn,namfn,chemfn)
      IMPLICIT NONE
      INCLUDE 'general.h'
      INCLUDE 'common.h'
      INCLUDE 'organic.h'

* input:
      CHARACTER(LEN=lfo) xcri
      REAL            brch
      REAL            cut_off

* input/output
      CHARACTER(LEN=ldi) dict(mni)
      REAL            dbrch(mni)
      CHARACTER(LEN=lco) namlst(mni)
      INTEGER         level
      INTEGER         stabl
      CHARACTER(LEN=lst) holdvoc(mlv)
      INTEGER         nhldvoc
      CHARACTER(LEN=lst) holdrad(mra)
      INTEGER         nhldrad
      INTEGER          nfn
      CHARACTER(LEN=lco)  namfn(mfn)
      CHARACTER(LEN=lfo)  chemfn(mfn)

* output
      CHARACTER(LEN=lco) p(mnr)
      REAL            s(mnr)

* internal
      INTEGER         cnum,onum,l
      CHARACTER(LEN=lfo) pchem(mnr),pchem1(mnr),pchem2(mnr),prod,prod2
      CHARACTER(LEN=lgr) tgroup(mca),group(mca),tempgr,pold, pnew
      CHARACTER(LEN=lco) coprod(mca),coprod_del(mca)
      INTEGER         tbond(mca,mca),bond(mca,mca)
      INTEGER         dbflg,nring
      INTEGER         np,nc,nca,ig,j1,j2,i,j,nold,ngr
      INTEGER         posj1,posj2
      REAL            ftherm,fdec1,fmol,yld1,yld2
      REAL            brtio
      REAL            garrhc(3),c1,c2
      CHARACTER(LEN=lco) copchem
      REAL              rdtcopchem
      INTEGER         rngflg       ! 0 = 'no', 1 = 'yes'
      INTEGER         ring(mca)    ! =1 if node participates in current ring
      CHARACTER(LEN=lfo) rdckprod(mca)
      CHARACTER(LEN=lco) rdcktprod(mca,mca)
      INTEGER         nip
      REAL            sc(mca)
      CHARACTER(LEN=lfo) pchem_del

* ring-character-related variables
      INTEGER         rjg(mri,2)

      IF(wtflag.NE.0) WRITE(*,*) '*xcrieg*   ',xcri
* initialize:
      np = 0
      ig = 0
      rdtcopchem=0.

      DO i=1,mnr
         pchem(i) = ' '
         p(i) = ' '
         s(i) = 0.
      ENDDO

* calling function to define number of nodes (carbons + '-O-') in hot Criegee
      nc = INDEX(xcri,' ') - 1
      nca = cnum(xcri,nc)+onum(xcri,nc)
* ngr was previously used in call to "findring" but introduced errors.
      ngr=0
      DO i=1,mca
        IF (group(i).NE.' ') ngr=ngr+1
      ENDDO

* define functional groups and bond-matrix of hot Criegee and
* store the data into bond and group
      CALL grbond(xcri,nc,group,bond,dbflg,nring)
! DEBUG !
!      PRINT*,"nring=",nring
! DEBUG !
      DO i=1,mca
        tgroup(i)=group(i)
        DO j=1,mca
           tbond(i,j)=bond(i,j)
        ENDDO
      ENDDO

* find hot_criegge groups:
      DO i=1,mca
         IF (INDEX(tgroup(i),hot_criegee).NE.0) ig = i
      ENDDO
! DEBUG !
!      print*,"ig = ",ig
! DEBUG !
      IF (ig.EQ.0) THEN
         WRITE(6,'(a)') '--warning--(stop) in xcrieg'
         WRITE(6,'(a)') 'hot_criegge functional group not found in :'
         WRITE(6,'(a)') xcri
         WRITE(99,*) 'xcrieg',xcri !STOP
      ENDIF

* -----------------------------------------
*  1st section: single carbon hot criegees:
* ------------------------------------------

      IF (nca.EQ.1) THEN

* branching ratio for CH2.(OO.)* are taken in Atkinson, 1999, J. Chem. Ref.
* Data. The stabilized criegee radical is expected to produce a carboxylic
* acid and the "cold criegee" is no longer treated. If criegee radical is
* not specifically adressed below then the program stops.
         IF (xcri(1:10).EQ.'CH2.(OO.)*') THEN
           s(1) = 0.37
           pchem(1) = 'CHO(OH)'
           brtio = brch * s(1)
*           CALL bratio(pchem(1),brtio,p(1),copchem,rdtcopchem,
           CALL bratio(pchem(1),brtio,p(1),
     &               dbrch,dict,namlst,
     &               nhldvoc,holdvoc,nhldrad,holdrad,level,stabl,
     &               nfn,namfn,chemfn)
           s(2) = 0.13
           p(2) = 'CO2  '
           s(3) = 0.13
           p(3) = 'H2   '
           s(4) = 0.50
           p(4) = 'CO   '
           s(5) = 0.38
           p(5) = 'H2O  '
           s(6) = 0.12
           p(6) = 'HO2  '
           s(7) = 0.12
           p(7) = 'HO   '
           RETURN
! escape routes for minor products of substituted ring-opening
! if criegee is "CH(OH).(OO.)" then assume NO conversion to acid, 100% yield.
         ELSE IF (xcri(1:13).EQ.'CH(OH).(OO.)*') THEN
           s(1) = 1.0
           p(1) = 'CHO(OH)  '
           s(2) = 1.0
           p(2) = 'NO2  '
           RETURN
! JMLT June 2020
! if criegee is "CH(OONO2).(OO.)" then assume NO3 elimination, 100% yield.
         ELSE IF (xcri(1:16).EQ.'CH(OONO2).(OO.)*') THEN
           pchem(1) = 'CHO(OO.) '
           CALL bratio(pchem(1),brtio,p(1),
     &               dbrch,dict,namlst,
     &               nhldvoc,holdvoc,nhldrad,holdrad,level,stabl,
     &               nfn,namfn,chemfn)

           s(1) = 1.0
           s(2) = 1.0
           p(2) = 'NO3  '
           RETURN
! if criegee is "CH(ONO2).(OO.)" then assume NO2 elimination, 100% yield.
         ELSE IF (xcri(1:15).EQ.'CH(ONO2).(OO.)*') THEN
           pchem(1) = 'CHO(OO.) '
           CALL bratio(pchem(1),brtio,p(1),
     &               dbrch,dict,namlst,
     &               nhldvoc,holdvoc,nhldrad,holdrad,level,stabl,
     &               nfn,namfn,chemfn)

           s(1) = 1.0
           s(2) = 1.0
           p(2) = 'NO2  '
           RETURN
! if criegee is "CH(OOH).(OO.)" then assume OH elimination, 100% yield.
         ELSE IF (xcri(1:14).EQ.'CH(OOH).(OO.)*') THEN
           pchem(1) = 'CHO(OO.) '
           CALL bratio(pchem(1),brtio,p(1),
     &               dbrch,dict,namlst,
     &               nhldvoc,holdvoc,nhldrad,holdrad,level,stabl,
     &               nfn,namfn,chemfn)
           s(1) = 1.0
           s(2) = 1.0
           p(2) = 'HO   '
           RETURN
! if criegee is "CO.(OO.)" then self-destruct, 100% yield.
         ELSE IF (xcri(1:9).EQ.'CO.(OO.)*') THEN
           s(1) = 1.0
           p(1) = 'CO   '
           s(2) = 1.0
           p(2) = 'O2   '
           RETURN

         ELSE
           WRITE(6,'(a)') '--warning--(stop) in xcrieg'
           WRITE(6,'(a)') 'following C1 hot_criegge not found'
           WRITE(6,'(a)') 'in the routine. Change the program'
           WRITE(6,'(a)') 'accordingly.'
           WRITE(6,'(a)') xcri
           WRITE(99,*) 'xcrieg',xcri !STOP
         ENDIF
       ENDIF

* -------------------------------------------
* 2nd section:  multi-carbon hot criegees:
* -------------------------------------------

       j1 = 0
       j2 = 0

* see if external or internal Criegee (internal:J2>0, external:J2=0):
       DO 210 i=1,mca
          IF (tbond(i,ig).EQ.0) GOTO 210 !(CYCLE)
          IF (j1.EQ.0) THEN
             j1 = i
          ELSE
             j2 = i
          ENDIF
 210   CONTINUE

* --------------------------------
*  open chapter "external criegee"
* --------------------------------

* First the program checks if the criegee radical is an alpha
* carbonyl or alpha unsaturated carbon. If not, then perform
* the reaction for the "regular" R-CH.(OO.)* criegee.

       IF (j2.EQ.0) THEN
         IF (tgroup(ig)(1:9).EQ.'CH.(OO.)*') THEN

* if criegee is "CHO-CH.(OO.)" then assume decomposition with 100% yield.
* The 2 channels (CHO+CO2+H, HCO+CO+OH) are arbitrarly set with equal
* probability. This scheme is borrowed from the SAPRC99 mechanism and
* is based on the fact that HCO-C.(OO.) has weaker bond energy than the
* corresponding CH3-CH.(OO.) radical
           IF (tgroup(j1)(1:3).eq.aldehyde) THEN
             s(1) = 1.5
             p(1) = 'CO   '
             s(2) = 1.5
             p(2) = 'HO2  '
             s(3) = 0.5
             p(3) = 'HO   '
             s(4) = 0.50
             p(4) = 'CO2  '
             RETURN
           ENDIF

* if criegee is "R-CO-CH.(OO.)" then assume O shift to -CO- group with
* 100% yield. This scheme is borrowed from the SAPRC99 mechanism and
* is based on the fact that the resulting criegee is more stable than
* the first one. This reaction changes the external criegee into an
* internal which is treated in the next chapter
           IF (tgroup(j1)(1:3).eq.'CO ') THEN

* swap criegee into aldehyde
             pold='CH.(OO.)*'
             pnew=aldehyde
             tempgr=tgroup(ig)
             CALL swap(tempgr,pold,tgroup(ig),pnew)

* swap carbonyl into criegee
             pold=carbonyl
             pnew='C.(OO.)*'
             tempgr=tgroup(j1)
             CALL swap(tempgr,pold,tgroup(j1),pnew)

* store new position and new group matrix
             ig=j1
             DO i=1,mca
               group(i)=tgroup(i)
             ENDDO
             j1 = 0
             j2 = 0
             DO 220 i=1,mca
               IF (tbond(i,ig).EQ.0) GOTO 220
               IF (j1.EQ.0) THEN
                 j1 = i
               ELSE
                 j2 = i
               ENDIF
220          CONTINUE

* jump to next chapter
             GOTO 456
           ENDIF

* if criegee is "C=C-CH.(OO.)" then assume H shift to give the
* corresponding alkene with 25 % yield. The remaining fraction is
* expected to be stabilized (and hence yield the corresponding
* carboxylic acid). This scheme is borrowed from the SAPRC99 mechanism
* and is based on detailed isoprene chemistry.
           IF (tgroup(j1)(1:2).eq.'Cd') THEN

* H shift
* -------

* add CO2
             np=np+1
             IF (np.GT.mnr) THEN
                WRITE(6,'(a)') '--error-- in xcrieg'
                WRITE(6,'(a)') 'np is greater than mnr'
                WRITE(99,*) 'xcrieg',xcri !STOP
             ENDIF
             s(np) = 0.25
             p(np) = 'CO2  '

* add corresponding alkene
             np=np+1
             IF (np.GT.mnr) THEN
                WRITE(6,'(a)') '--error-- in xcrieg'
                WRITE(6,'(a)') 'np is greater than mnr'
                WRITE(99,*) 'xcrieg',xcri !STOP
             ENDIF
             s(np) = 0.25

             tbond(ig,j1)=0
             tbond(j1,ig)=0

             IF (tgroup(j1)(1:3).eq.'CdH') THEN
               pold='CdH'
               pnew='CdH2'
             ELSE
               pold='Cd'
               pnew='CdH'
             ENDIF
             tempgr=tgroup(j1)
             CALL swap(tempgr,pold,tgroup(j1),pnew)
             CALL rebond(tbond,tgroup,pchem(np),nring)

* reset
             tgroup(:)=group(:)
             tbond(:,:)=bond(:,:)

* stabilization
* -------------

             np=np+1
             IF (np.GT.mnr) THEN
               WRITE(6,'(a)') '--error-- in xcrieg'
               WRITE(6,'(a)') 'np is greater than mnr'
               WRITE(99,*) 'xcrieg',xcri !STOP
             ENDIF
             s(np) = 0.75

!             pold='CH.(OO.)*'
!             pnew=carboxylic_acid
!             tempgr=tgroup(ig)
!             CALL swap(tempgr,pold,tgroup(ig),pnew)
             CALL rebond(tbond,tgroup,pchem(np),nring)
             CALL stdchm(pchem(np))
             brtio = brch * s(np)
             CALL bratio(pchem(np),brtio,p(np),
     &               dbrch,dict,namlst,
     &               nhldvoc,holdvoc,nhldrad,holdrad,level,stabl,
     &               nfn,namfn,chemfn)

* reset
             tgroup(:)=group(:)
             tbond(:,:)=bond(:,:)

* jump to end criegee reaction
             GOTO 920
           ENDIF

*  R-CH.(OO.)
* If the program has reached this point, then the criegee radical react
* has a regular R-CH.(OO.)
* Atkinson, 1999, J. Phys. Chem. Ref. Data recommanded for CH3-CH.(OO.):
* -> CH3CH.(OO.) (stab.) : 15%
* -> CH3+CO+OH           : 54%
* -> CH3+CO2+H           : 17%
* -> CH4+CO              : 14%
* The above scheme was found to produce a too large amount
* of radical to match observation in smog chambers (see Carter,1999,
* SAPRC99 scheme). Measurements of Paulson et al 99 confirm the decrease
*  of OH yield with increasing length of radical.
* Therefore, we choose to affect for CH3-CH.(OO.) (Carter SAPRC99 in
* agreement with Rickard 99 OH measurements) :
* -> CH3CH.(OO.) (stab.) : 34%   (A)
* -> CH3+CO+OH           : 52%   (B)
* -> CH4+CO              : 14%   (C)
* for RCH.(OO.), b ratios fixed to fit OH measurement from Paulson 99
* if R = 2C => (A) 0.54 ;(B) 0.46 ;(C) 0 ;
* if R = 3C => (A) 0.64 ;(B) 0.36 ;(C) 0 ;
* if R = 4C => (A) 0.76 ;(B) 0.24 ;(C) 0 ;
* if R = 5C => (A) 0.84 ;(B) 0.16 ;(C) 0 ;
* if R = 6C => (A) 0.92 ;(B) 0.08 ;(C) 0 ;
* if R > 6C => (A) 1.   ;(B) 0    ;(C) 0 ;

* Stabilization
* -------------
           IF (nca.eq.2) ftherm = 0.34
           IF (nca.eq.3) ftherm = 0.54
           IF (nca.eq.4) ftherm = 0.64
           IF (nca.eq.5) ftherm = 0.76
           IF (nca.eq.6) ftherm = 0.84
           IF (nca.eq.7) ftherm = 0.92
           IF (nca.gt.7) ftherm = 1
           np=np+1
           IF (np.GT.mnr) THEN
              WRITE(6,'(a)') '--error-- in xcrieg'
              WRITE(6,'(a)') 'np is greater than mnr'
              WRITE(99,*) 'xcrieg',xcri !STOP
           ENDIF
           s(np) = ftherm
!           pold='CH.(OO.)*'
!           pnew=carboxylic_acid
!           tempgr=tgroup(ig)
!           CALL swap(tempgr,pold,tgroup(ig),pnew)
           CALL rebond(tbond,tgroup,pchem(np),nring)
           CALL stdchm(pchem(np))
           CALL bratio(pchem(np),brtio,p(np),
     &               dbrch,dict,namlst,
     &                  nhldvoc,holdvoc,nhldrad,holdrad,level,stabl,
     &                  nfn,namfn,chemfn)
* reset
           tgroup(:)=group(:)
           tbond(:,:)=bond(:,:)

* H shift (molecular channel):
* ----------------------------
* This channel occurs only for C2 species.
* Old version was setting reaction products in the stack even
* if not produced.
           IF (nca.eq.2) THEN
c            IF (nca.eq.2) fmol = 0.14
c            IF (nca.ge.3) fmol = 0.
             fmol=0.14
* add CO2
             np=np+1
             IF (np.GT.mnr) THEN
                WRITE(6,'(a)') '--error-- in xcrieg'
                WRITE(6,'(a)') 'np is greater than mnr'
                WRITE(99,*) 'xcrieg',xcri !STOP
             ENDIF
             s(np) = fmol
             p(np) = 'CO2  '

* add corresponding molecule
             np    = np + 1
             IF (np.GT.mnr) THEN
                WRITE(6,'(a)') '--error-- in xcrieg'
                WRITE(6,'(a)') 'np is greater than mnr'
                WRITE(99,*) 'xcrieg',xcri !STOP
             ENDIF
             s(np) = fmol

             tgroup(ig) = ' '

             tbond(ig,j1) = 0
             tbond(j1,ig) = 0

             IF (tgroup(j1)(1:3).EQ.methyl) THEN
                p(np) = 'CH4  '
             ELSE
               IF (tgroup(j1)(1:3).EQ.primary) THEN
                  pold = primary
                  pnew = 'CH3'
               ELSE IF (tgroup(j1)(1:2).EQ.secondary) THEN
! JMLT Nov'15: decomposition of multi-substituted C1 coproducts
!              similar to hvdiss2.f
! NB: xcrieg 0.28 chan transfers H from adjacent =CdH, eliminates CO2
!     We might prefer to eliminate CO + OH (but do not do so here)
                  IF (INDEX(tgroup(j1),'(OOH)(ONO2)').NE.0)THEN
                    pold='CH(OOH)(ONO2)'
                    pnew='CH2(OH)(OOH)'
                    p(np) = 'NO2  '
                    np = np+1
                    s(np) = fmol
                  ELSE IF (INDEX(tgroup(j1),'(ONO2)(OOH)').NE.0)THEN
                    pold='CH(ONO2)(OOH)'
                    pnew='CH2(OH)(OOH)'
                    p(np) = 'NO2  '
                    np = np+1
                    s(np) = fmol
                  ELSE IF (INDEX(tgroup(j1),'(ONO2)(ONO2)').NE.0)THEN
                    pold='CH(ONO2)(ONO2)'
                    pnew='CH2(OH)(ONO2)'
                    p(np) = 'NO2  '
                    np = np+1
                    s(np) = fmol
                  ELSE IF (INDEX(tgroup(j1),'(OOH)(OOH)').NE.0)THEN
                    pold='CH(OOH)(OOH)'
                    pnew='CH2(OH)(OOH)'
                    p(np) = 'OH   '
                    np = np+1
                    s(np) = fmol
                  ELSE
                    pold = secondary
                    pnew = 'CH2'
                  ENDIF
               ELSE IF (tgroup(j1)(1:1).EQ.'C') THEN
                  pold = 'C'
                  pnew = secondary
               ELSE
                 WRITE(6,'(a)') '--error-- in xcrieg'
                 WRITE(6,'(a)') 'group not found to perform'
                 WRITE(6,'(a)') 'the molecular channel for :'
                 WRITE(6,'(a)') xcri
                 WRITE(99,*) 'xcrieg',xcri !STOP
               ENDIF

               tempgr=tgroup(j1)
               CALL swap(tempgr,pold,tgroup(j1),pnew)
               CALL rebond(tbond,tgroup,pchem(np),nring)
             ENDIF
* reset
             DO i=1,mca
               tgroup(i)=group(i)
               DO j=1,mca
                 tbond(i,j)=bond(i,j)
               ENDDO
             ENDDO

           ENDIF

* decomposition 1 channel (R. + CO + HO)
* --------------------------------------
* This channel occurs only for C<8 species.
* Old version was setting reaction products in the stack even
* if not produced.
           IF (nca.gt.7) GOTO 42
           IF (nca.eq.2) fdec1 = 0.52
           IF (nca.eq.3) fdec1 = 0.46
           IF (nca.eq.4) fdec1 = 0.36
           IF (nca.eq.5) fdec1 = 0.24
           IF (nca.eq.6) fdec1 = 0.16
           IF (nca.eq.7) fdec1 = 0.08

* add CO
           np=np+1
           IF (np.GT.mnr) THEN
              WRITE(6,'(a)') '--error-- in xcrieg'
              WRITE(6,'(a)') 'np is greater than mnr'
              WRITE(99,*) 'xcrieg',xcri !STOP
           ENDIF
           s(np) = fdec1
           p(np) = 'CO   '

* add HO
           np=np+1
           IF (np.GT.mnr) THEN
              WRITE(6,'(a)') '--error-- in xcrieg'
              WRITE(6,'(a)') 'np is greater than mnr'
              WRITE(99,*) 'xcrieg',xcri !STOP
           ENDIF
           s(np) = fdec1
           p(np) = 'HO   '

* add corresponding radical
           np    = np + 1
           IF (np.GT.mnr) THEN
              WRITE(6,'(a)') '--error-- in xcrieg'
              WRITE(6,'(a)') 'np is greater than mnr'
              WRITE(99,*) 'xcrieg',xcri !STOP
           ENDIF
           s(np) = fdec1

           tgroup(ig) = ' '
           tbond(ig,j1) = 0
           tbond(j1,ig) = 0

* if the group next to the criegee radical is an ether,
* it must be converted into an alkoxy radical (ric nov 2008)
           IF (tgroup(j1).EQ.'-O-') THEN
	     DO i=1,mca
	       IF ((tbond(i,j1).NE.0).AND.(i.NE.ig)) THEN
                 tbond(i,j1) = 0
                 tbond(j1,i) = 0
                 tgroup(j1) = ' '
                 nc = INDEX(tgroup(i),' ')
		 tgroup(i)(nc:nc+3)='(O.)'
	       ENDIF
	     ENDDO
	   ELSE
             nc = INDEX(tgroup(j1),' ')
             tgroup(j1)(nc:nc) = '.'
	   ENDIF

           CALL rebond(tbond,tgroup,pchem(np),nring)
* reset
           DO i=1,mca
             tgroup(i)=group(i)
             DO j=1,mca
               tbond(i,j)=bond(i,j)
             ENDDO
           ENDDO

42         CONTINUE

* decomposition 2 channel (R. + CO2 + H)
* --------------------------------------
c           fdec2=0.17
* add CO2
c           np=np+1
c           IF (np.GT.mnr) THEN
c              WRITE(6,'(a)') '--error-- in xcrieg'
c              WRITE(6,'(a)') 'np is greater than mnr'
c              WRITE(99,*) 'xcrieg',xcri !STOP
c           ENDIF
c           s(np) = fdec2
c           p(np) = 'CO2  '

* add HO2
c           np=np+1
c           IF (np.GT.mnr) THEN
c              WRITE(6,'(a)') '--error-- in xcrieg'
c              WRITE(6,'(a)') 'np is greater than mnr'
c              WRITE(99,*) 'xcrieg',xcri !STOP
c           ENDIF
c           s(np) = fdec2
c           p(np) = 'HO2  '

* add corresponding radical
c           np    = np + 1
c           IF (np.GT.mnr) THEN
c              WRITE(6,'(a)') '--error-- in xcrieg'
c              WRITE(6,'(a)') 'np is greater than mnr'
c              WRITE(99,*) 'xcrieg',xcri !STOP
c           ENDIF
c           s(np) = fdec2

c           tgroup(ig) = ' '
c           tbond(ig,j1) = 0
c           tbond(j1,ig) = 0

c           nc = INDEX(tgroup(j1),' ')
c           tgroup(j1)(nc:nc) = '.'
c           CALL rebond(tbond,tgroup,pchem(np),nring)

* reset
c           DO i=1,mca
c             tgroup(i)=group(i)
c             DO j=1,mca
c               tbond(i,j)=bond(i,j)
c             ENDDO
c           ENDDO

* go to assignment 920, where the products are checked to either be
* or not be loaded in the stack.
           GO TO 920

* escape routes for products of ring-opening (John Orlando, no reference)
* if criegee is "R-C(OH).(OO.)*" then assume conversion to acid with NO, 100% yield.
         ELSE IF (tgroup(ig)(1:12).EQ.'C(OH).(OO.)*') THEN
           s(2) = 1.0
           p(2) = 'NO2  '
           s(1) = 1.0
           pold='C(OH).(OO.)*'
           pnew=carboxylic_acid
           tempgr=tgroup(ig)
           CALL swap(tempgr,pold,tgroup(ig),pnew)
           CALL rebond(tbond,tgroup,pchem(1),nring)
           GOTO 920
* JMLT June 2020
* if criegee is "R-C(OONO2).(OO.)*" then assume NO3 elimination, 100% yield.
         ELSE IF (tgroup(ig)(1:15).EQ.'C(OONO2).(OO.)*') THEN
           s(2) = 1.0
           p(2) = 'NO3  '
           s(1) = 1.0
           pold='C(OONO2).(OO.)*'
           pnew='CO(OO.)'
           tempgr=tgroup(ig)
           CALL swap(tempgr,pold,tgroup(ig),pnew)
           CALL rebond(tbond,tgroup,pchem(1),nring)
           GOTO 920
* if criegee is "R-C(ONO2).(OO.)*" then assume NO2 elimination, 100% yield.
         ELSE IF (tgroup(ig)(1:14).EQ.'C(ONO2).(OO.)*') THEN
           s(2) = 1.0
           p(2) = 'NO2  '
           s(1) = 1.0
           pold='C(ONO2).(OO.)*'
           pnew='CO(OO.)'
           tempgr=tgroup(ig)
           CALL swap(tempgr,pold,tgroup(ig),pnew)
           CALL rebond(tbond,tgroup,pchem(1),nring)
           GOTO 920
* if criegee is "R-C(OOH).(OO.)*" then assume OH elimination, 100% yield.
         ELSE IF (tgroup(ig)(1:13).EQ.'C(OOH).(OO.)*') THEN
           s(2) = 1.0
           p(2) = 'HO   '
           s(1) = 1.0
           pold='C(OOH).(OO.)*'
           pnew='CO(OO.)'
           tempgr=tgroup(ig)
           CALL swap(tempgr,pold,tgroup(ig),pnew)
           CALL rebond(tbond,tgroup,pchem(1),nring)
           GOTO 920

         ELSE
* If the program continues up to this point for external criegee then
* the criegee is not a -CH.(OO.) criegee. The program will stop. Add
* some additional program lines if required.
           WRITE(6,'(a)') '--warning--(stop) in xcrieg'
           WRITE(6,'(a)') 'following external hot_criegge cannot be'
           WRITE(6,'(a)') 'computed. Please add some'
           WRITE(6,'(a)') 'additional fortran lines'
           WRITE(6,'(a)') xcri
           WRITE(99,*) 'xcrieg',xcri
           STOP
         ENDIF

* end external criegee
       ENDIF


* -----------------------------------------------
* else open chapter "internal criegees"
* -----------------------------------------------

456    CONTINUE

* For -CH-C.(OO.)-C criegee radical, major evolution pathway is
* expected to be the hydroperoxide channel (e.g. see Atkinson, 1999,
* J. Chem. Ref. Data) :
* -CH-C.(OO.)-C => -C=C(OOH)-C => -C.-CO-C + OH
* This pathway is set with 100% yield. This mechanism requires an
* H for the group in alpha position. Note that for H in -CHO, the
* above mechanism is expected to not occur, because of the formation
* of a strained transition state (see Carter, 1999, SAPRC99 mechanism).
* Therefore, the program first checks the two group at alpha position

       IF (j2.NE.0) THEN

* check that that criegee make sense
         IF (tgroup(ig)(1:8).NE.'C.(OO.)*') THEN
           WRITE(6,'(a)') '--warning--(stop) in xcrieg'
           WRITE(6,'(a)') 'something wrong for the following'
           WRITE(6,'(a)') 'criegee radical (was first expected'
           WRITE(6,'(a)') 'to be C.(OO.)*'
           WRITE(6,'(a)') xcri
           WRITE(99,*) 'xcrieg',xcri !STOP
         ENDIF

! JMLT April '17 !
* remove any ring flags from tgroups
* CAUTION!! Note that this code allows formation of
* CH3C1(CH3)C.(OO.)*C1CHO (carene mechanism) which is a
* 3-membered ring Criegge and therefore highly strained.

         IF(nring.NE.0)THEN
! DEBUG !
!         PRINT*,"removing ring characters"
!         print*,j1,tgroup(j1),j2,tgroup(j2)
! DEBUG !
           CALL rjgrm(nring,tgroup(j1),rjg)
           CALL rjgrm(nring,tgroup(j2),rjg)
         ENDIF
! DEBUG !
!         print*,j1,tgroup(j1),j2,tgroup(j2)
! DEBUG !
! JMLT !

* check for CH groups
         posj1=1
         posj2=1
         IF (INDEX(tgroup(j1),'CH').EQ.0) posj1=0
         IF (INDEX(tgroup(j2),'CH').EQ.0) posj2=0
! DEBUG !
!         print*,"posj1,2",posj1,posj2
! DEBUG !

!* check for CH groups (but not CHO)
! I'd put this in for cyclic compounds: removed later for test
!         posj1=1
!         posj2=1
!         IF (INDEX(tgroup(j1),'CH').EQ.0 .OR.
!     &       INDEX(tgroup(j1),'CHO').NE.0) posj1=0
!         IF (INDEX(tgroup(j2),'CH').EQ.0 .OR.
!     &       INDEX(tgroup(j2),'CHO').NE.0) posj2=0

* no CH group.
* ------------

* Carter (SAPRC99) arbitrarily assume that 90% of this type of Criegee
* is stabilized and 10% decomposes to CO2 and 2 R.
* We consider that the thermalised radical only reacts with H2O to
* product H2O2 + ketone (Baker 2002 assumed it is the only way for
* R1R2C(OOH)(OH) )

         IF ( (posj1.EQ.0).AND.(posj2.EQ.0) ) THEN

* 10% decomposition
* if the carbon bearing the criegee belongs to a ring, this channel
* lead to a di-radical species, which is not currently treated in the
* generator. need update. (ric 2008)
           CALL findring(ig,j1,nca,tbond,rngflg,ring)
	   IF (ring(ig).EQ.1) GOTO 200

           tbond(ig,j1) = 0
           tbond(j1,ig) = 0
           tbond(ig,j2) = 0
           tbond(j2,ig) = 0
           tgroup(ig) = ' '
           nc = INDEX(tgroup(j1),' ')
           tgroup(j1)(nc:nc) = '.'
           nc = INDEX(tgroup(j2),' ')
           tgroup(j2)(nc:nc) = '.'
* fragment and write in correct format:
           CALL fragm(tbond,tgroup,pchem1(np),pchem2(np))
           CALL stdchm(pchem1(np))
           CALL stdchm(pchem2(np))
           prod2 = pchem2(np)
* check pchem1, write in standard format:
c           CALL radchk(pchem1(np),prod,coprod)
           CALL radchk(pchem1(np),rdckprod,rdcktprod,nip,sc)
           prod = rdckprod(1)
	   IF (nip.NE.1) WRITE(6,*) 'xcrieg.f coprod a revoir'
	   IF (nip.NE.1) WRITE(6,*) prod
	   IF (nip.NE.1) STOP
	   DO l = 1,mca
	     coprod(l) = rdcktprod(1,l)
	   ENDDO
           CALL stdchm(prod)
           DO l=1,mca
             IF (coprod(l).NE.' ') THEN
               np = np + 1
               s(np) = 0.1
               p(np) = coprod(l)
             ENDIF
           ENDDO
           IF (np+1.GT.mnr) THEN
             WRITE(6,'(a)') '--error-- in xcrieg'
             WRITE(6,'(a)') 'np is greater than mnr'
             WRITE(99,*) 'xcrieg',xcri !STOP
           ENDIF

           np = np + 1
           s(np) = 0.1
           brtio = brch * s(np)

           CALL bratio(prod,brtio,p(np),
     &               dbrch,dict,namlst,
     &                  nhldvoc,holdvoc,nhldrad,holdrad,level,stabl,
     &                  nfn,namfn,chemfn)
           IF (rdtcopchem.GT.0.) THEN
             np = np + 1
             IF (np.GT.mnp) then
               WRITE(6,'(a)') '--error-- in xcrieg'
               WRITE(6,'(a)') 'np is greater than mnp'
               WRITE(6,'(a)') '(too much product in the reaction)'
               WRITE(99,*) 'xcrieg',xcri !STOP
             ENDIF
             s(np) = rdtcopchem
             p(np) = copchem
           ENDIF
           IF (np+2.GT.mnr) THEN
              WRITE(6,'(a)') '--error-- in xcrieg'
              WRITE(6,'(a)') 'np is greater than mnr'
              WRITE(99,*) 'xcrieg',xcri !STOP
           ENDIF


* check prod2 = pchem2, write in standard format:

           CALL radchk(prod2,rdckprod,rdcktprod,nip,sc)
           prod = rdckprod(1)
           IF (nip.NE.1) WRITE(6,*) 'xcrieg.f coprod \E0 revoir 2'
           IF (nip.NE.1) WRITE(6,*) prod
           IF (nip.NE.1) STOP

           coprod(:) = rdcktprod(1,:)

           CALL stdchm(prod)
           DO l=1,mca
             IF (coprod(l).NE.' ') THEN
             np = np + 1
             s(np) = 0.1
             p(np) = coprod(l)
             ENDIF
           ENDDO

           np = np + 1
           s(np) = 0.1
           brtio = brch * s(np)

           CALL bratio(prod,brtio,p(np),
     &               dbrch,dict,namlst,
     &                  nhldvoc,holdvoc,nhldrad,holdrad,level,stabl,
     &                  nfn,namfn,chemfn)
           s(np+1) = 0.1
           p(np+1) = 'CO2'
           IF (rdtcopchem.GT.0.) THEN
             np = np + 1
             IF (np.GT.mnp) then
               WRITE(6,'(a)') '--error-- in xcrieg'
               WRITE(6,'(a)') 'np is greater than mnp'
               WRITE(6,'(a)') '(too much product in the reaction)'
               WRITE(99,*) 'xcrieg',xcri !STOP
             ENDIF
             s(np) = rdtcopchem
             p(np) = copchem
           ENDIF
* reset
           DO i=1,mca
             tgroup(i)=group(i)
             DO j=1,mca
               tbond(i,j)=bond(i,j)
             ENDDO
           ENDDO

200        CONTINUE
! DEBUG !
!           print*,"reached 200"
!           print*,"cyclic criegge with no CH group"
!           STOP
! DEBUG !

* 90% product H2O2 + ketone
           np = np + 1
           s(np) = 0.9
           IF (ring(ig).EQ.1) s(np) = 1.
!           p(np) = 'H2O2'
!           pold = 'C.(OO.)*'
!           pnew = 'CO'
!           CALL swap(group(ig),pold,tgroup(ig),pnew)
!           CALL rebond(tbond,tgroup,prod2,nring)
!           CALL stdchm(prod2)
!           s(np) = 0.9
!           IF (ring(ig).EQ.1) s(np) = 1.
           brtio = brch * s(np)
           CALL bratio(xcri,brtio,p(np),
     &               dbrch,dict,namlst,
     &                  nhldvoc,holdvoc,nhldrad,holdrad,level,stabl,
     &                  nfn,namfn,chemfn)
!           IF (rdtcopchem.GT.0.) THEN
!             np = np + 1
!             IF (np.GT.mnp) then
!               WRITE(6,'(a)') '--error-- in xcrieg'
!               WRITE(6,'(a)') 'np is greater than mnp'
!               WRITE(6,'(a)') '(too much product in the reaction)'
!               WRITE(99,*) 'xcrieg',xcri !STOP
!             ENDIF
!             s(np) = rdtcopchem
!             p(np) = copchem
!           ENDIF

* reset
           tgroup(:)=group(:)
           tbond(:,:)=bond(:,:)
           RETURN
         ENDIF

* CH in alpha-position
* --------------------
* put yield for the various channels. If the reaction can
* occur at the two positions, then ratio is arbitrarily set
* using rate constant for OH reaction at the given site. A
* cut off of 5% is used

! CAUTION !
* If the carbon bearing the criegee belongs to a ring, this channel
* leads to a very strained radical species, of dubious stability.
* Allowed for now. Needs update. (JMLT, 2017)

         CALL findring(ig,j1,nca,tbond,rngflg,ring)
!         IF (ring(ig).EQ.1) GOTO 300

         IF ( (posj1.EQ.1).AND.(posj2.EQ.0) ) THEN
           yld1=1.
           yld2=0.
         ENDIF
         IF ( (posj1.EQ.0).AND.(posj2.EQ.1) ) THEN
           yld1=0.
           yld2=1.
         ENDIF
         IF ( (posj1.EQ.1).AND.(posj2.EQ.1) ) THEN
           CALL rabsoh(tbond,tgroup,j1,garrhc,nring)
           c1 = garrhc(1) * 298.**garrhc(2) * exp(-garrhc(3)/298.)
           c1 = AMIN1(c1,2.0E-10)
           CALL rabsoh(tbond,tgroup,j2,garrhc,nring)
           c2 = garrhc(1) * 298.**garrhc(2) * exp(-garrhc(3)/298.)
           c2 = AMIN1(c2,2.0E-10)

           yld1=c1/(c1+c2)
           yld2=c2/(c1+c2)
           IF (yld1.LT.0.05) THEN
             yld1=0.
             yld2=1.
           ENDIF
           IF (yld2.LT.0.05) THEN
             yld1=1.
             yld2=0.
           ENDIF
         ENDIF

* CH at position 1
* ----------------
         IF (yld1.GT.0.) THEN

* add OH
           np=np+1
           IF (np.GT.mnr) THEN
              WRITE(6,'(a)') '--error-- in xcrieg'
              WRITE(6,'(a)') 'np is greater than mnr'
              WRITE(99,*) 'xcrieg',xcri !STOP
           ENDIF
           s(np) = yld1
           p(np) = 'HO   '

* add corresponding molecule
           np    = np + 1
           IF (np.GT.mnr) THEN
              WRITE(6,'(a)') '--error-- in xcrieg'
              WRITE(6,'(a)') 'np is greater than mnr'
              WRITE(99,*) 'xcrieg',xcri !STOP
           ENDIF
           s(np) = yld1

* swap C.(OO).* into CO and add radical dot at j1
           tgroup(ig) = 'CO'
           IF (tgroup(j1)(1:3).EQ.methyl) THEN
                pold = methyl
                pnew = primary
           ELSE IF (tgroup(j1)(1:3).EQ.primary) THEN
                pold = primary
                pnew = secondary
           ELSE IF (tgroup(j1)(1:2).EQ.secondary) THEN
                pold = secondary
                pnew = 'C'
           ELSE
               WRITE(6,'(a)') '--error-- in xcrieg'
               WRITE(6,'(a)') 'group not found to perform'
               WRITE(6,'(a)') 'the molecular channel for :'
               WRITE(6,'(a)') xcri
               WRITE(99,*) 'xcrieg',xcri !STOP
           ENDIF

           tempgr=tgroup(j1)
           CALL swap(tempgr,pold,tgroup(j1),pnew)
           nc = INDEX(tgroup(j1),' ')
           tgroup(j1)(nc:nc) = '.'
           CALL rebond(tbond,tgroup,pchem(np),nring)

* reset
           DO i=1,mca
             tgroup(i)=group(i)
             DO j=1,mca
               tbond(i,j)=bond(i,j)
             ENDDO
           ENDDO
         ENDIF

* CH at position 2
* ----------------
         IF (yld2.GT.0.) THEN

* add OH
           np=np+1
           IF (np.GT.mnr) THEN
              WRITE(6,'(a)') '--error-- in xcrieg'
              WRITE(6,'(a)') 'np is greater than mnr'
              WRITE(99,*) 'xcrieg',xcri !STOP
           ENDIF
           s(np) = yld2
           p(np) = 'HO   '

* add corresponding molecule
           np    = np + 1
           IF (np.GT.mnr) THEN
              WRITE(6,'(a)') '--error-- in xcrieg'
              WRITE(6,'(a)') 'np is greater than mnr'
              WRITE(99,*) 'xcrieg',xcri !STOP
           ENDIF
           s(np) = yld2

* swap C.(OO).* into CO and add radical dot at j1
           tgroup(ig) = 'CO'
           CALL rjgrm(nring, tgroup, rjg)
           IF (tgroup(j2)(1:3).EQ.methyl) THEN
                pold = methyl
                pnew = primary
           ELSE IF (tgroup(j2)(1:3).EQ.primary) THEN
                pold = primary
                pnew = secondary
           ELSE IF (tgroup(j2)(1:2).EQ.secondary) THEN
                pold = secondary
                pnew = 'C'
           ELSE
               WRITE(6,'(a)') '--error-- in xcrieg'
               WRITE(6,'(a)') 'group not found to perform'
               WRITE(6,'(a)') 'the molecular channel for :'
               WRITE(6,'(a)') xcri
               WRITE(99,*) 'xcrieg',xcri !STOP
           ENDIF

           tempgr=tgroup(j2)
           CALL swap(tempgr,pold,tgroup(j2),pnew)
           nc = INDEX(tgroup(j2),' ')
           tgroup(j2)(nc:nc) = '.'
           CALL rjgadd(nring, tgroup, rjg)
           CALL rebond(tbond,tgroup,pchem(np),nring)

* reset
           DO i=1,mca
             tgroup(i)=group(i)
             DO j=1,mca
               tbond(i,j)=bond(i,j)
             ENDDO
           ENDDO
         ENDIF

* end internal criegee
       ENDIF

 300     CONTINUE
! DEBUG !
!         print*,ig,j1,ring(ig)," alpha-CH ->  cyclic criegge"
!         STOP
! DEBUG !
* ------------------------------------------------------
* check the various product and load species in the
* stack (if required).
* ------------------------------------------------------
920   nold = np

      DO 950 i=1,nold
         IF (pchem(i)(1:1).EQ.' ') GO TO 950
         prod = pchem(i)

         pchem_del=' '
         DO j=1,mca
           coprod_del(j) = ' '
         ENDDO

         IF (INDEX(pchem(i),'.').NE.0) THEN
c            CALL radchk(prod,pchem(i),coprod)
            CALL radchk(prod,rdckprod,rdcktprod,nip,sc)
            pchem(i) = rdckprod(1)
            IF (nip.EQ.2) THEN
              pchem_del = rdckprod(2)
              DO j=1,mca
                coprod_del(j) = rdcktprod(2,j)
              ENDDO
            ENDIF
	    DO l = 1,mca
	      coprod(l) = rdcktprod(1,l)
	    ENDDO
            DO j=1,mca
               IF (coprod(j)(1:1).NE.' ') THEN
                  np = np + 1
                  IF (np.GT.mnr) THEN
                     WRITE(6,'(a)') '--error-- in xcrieg'
                     WRITE(6,'(a)') 'np is greater than mnr'
                     WRITE(99,*) 'xcrieg',xcri !STOP
                  ENDIF
                  s(np)=s(i)
                  p(np) = coprod(j)
               ENDIF
            ENDDO
            DO j=1,mca
               IF (coprod_del(j)(1:1).NE.' ') THEN
                   WRITE(6,*) pchem_del
                   WRITE(6,*) coprod_del(j)
                  np = np + 1
                  IF (np.GT.mnr) THEN
                     WRITE(6,'(a)') '--error-- in xcrieg'
                     WRITE(6,'(a)') 'np is greater than mnr'
                     WRITE(99,*) 'xcrieg',xcri !STOP
                  ENDIF
                  s(np)=sc(2)
                  p(np) = coprod_del(j)
               ENDIF
            ENDDO
         ENDIF

         CALL stdchm(pchem(i))
         IF (nip.EQ.2) s(i)=s(i)*sc(1)
         brtio = brch * s(i)
*         CALL bratio(pchem(i),brtio,p(i),copchem,rdtcopchem,
         CALL bratio(pchem(i),brtio,p(i),
     &               dbrch,dict,namlst,
     &                  nhldvoc,holdvoc,nhldrad,holdrad,level,stabl,
     &                  nfn,namfn,chemfn)
         IF (nip.EQ.2) THEN
           np=np+1
           s(np)=(s(i)*sc(2))/sc(1)
           brtio = brch * s(np)
           CALL bratio(pchem_del,brtio,p(np),
     &               dbrch,dict,namlst,
     &                  nhldvoc,holdvoc,nhldrad,holdrad,level,stabl,
     &                  nfn,namfn,chemfn)
         ENDIF
         IF (rdtcopchem.GT.0.) THEN
           np = np + 1
           IF (np.GT.mnp) then
             WRITE(6,'(a)') '--error-- in ho_rad'
             WRITE(6,'(a)') 'np is greater than mnp'
             WRITE(6,'(a)') '(too much product in the reaction)'
             WRITE(99,*) 'xcrieg',xcri !STOP
           ENDIF
           s(np) = rdtcopchem
           p(np) = copchem
         ENDIF

      IF(wtflag.NE.0) WRITE(*,*) '*xcrieg* : product : ',i,pchem(i)
950   CONTINUE

      RETURN
      END

******************************************************************


