************************************************************************
* MASTER MECHANISM - ROUTINE NAME : rabsno3                            *
*                                                                      *
*                                                                      *
* PURPOSE :                                                            *
*  Find rate constants for H-atom abstraction from VOC by NO3 based on *
*  Kerdouci et al., 2010.                                              *
*                                                                      *
* INPUT:                                                               *
* - tgroup(i)     : groups at position (carbon) i                      *
* - tbond(i,j)    : carbon-carbon bond matrix of chem                  *
* - ig            : group bearing the leaving H                        *
*                                                                      *
* OUTPUT                                                               *
* - arrhc(3)     : arrhenius coefficient for H abstraction             *
*                                                                      *
*************************************************************************
      SUBROUTINE rabsno3(tbond,tgroup,ig,arrhc,nring)
      IMPLICIT NONE
      INCLUDE 'general.h'
      INCLUDE 'organic.h'
      INCLUDE 'common.h'

* input
      INTEGER         tbond(mca,mca)
      CHARACTER(LEN=lgr) tgroup(mca)
      INTEGER         ig
      INTEGER         nring

* output
      REAL            arrhc(3)

* internal:
      INTEGER         i,j,k
      REAL            mult
      INTEGER         nether,nca,rngflg,rgord,ring(mca)

      IF (wtflag.NE.0) WRITE(*,*) 
     & "*rabsno3*------------------------- ig = ",ig,' ',tgroup(ig)
* -----------
* initialize
* -----------
      DO i=1,3
        arrhc(i)=0.
      ENDDO

      nca=0
      DO i=1,mca 
        IF (tgroup(i)(1:1).EQ.'C') nca=nca+1
      ENDDO

* check ig value is ok
      IF (ig.GT.mca) THEN
         WRITE(6,*) '--error--, in rabsno3'
         WRITE(6,*) ' => ig is greater than mca'
         STOP
      ENDIF

* ------------------
* FIND K(0) VALUE
* ------------------

* Values are from Kerdouci et al., 2011
      IF (tgroup(ig)(1:3).EQ.'CH3') THEN
        arrhc(1) = 1.00E-18
        arrhc(2) = 0
        arrhc(3) = 0.
      ELSE IF(tgroup(ig)(1:3).EQ.'CH2') THEN
        arrhc(1) = 2.56E-17
        arrhc(2) = 0
        arrhc(3) = 0.
      ELSE IF(tgroup(ig)(1:3).EQ.'CHO') THEN
        arrhc(1) = 2.415E-15
        arrhc(2) = 0
        arrhc(3) = 0.
      ELSE IF(tgroup(ig)(1:2).EQ.'CH') THEN
        arrhc(1) = 1.05E-16
        arrhc(2) = 0
        arrhc(3) = 0.
        
      ELSE
        WRITE(6,'(a)') '--error--'
        WRITE(6,'(a)') 'from MASTER MECHANISM ROUTINE : rabsno3'
        WRITE(6,'(a)') 'group not allowed to react with NO3'
        WRITE(6,'(a)') tgroup(ig)
        STOP
      ENDIF
30    CONTINUE          
       
* ---------------------------------------
* FIND MULTIPLIERS BASED ON SUBSTITUENTS
* ---------------------------------------
* and correct the activation energy only

* on same carbon:
      mult = 1.
      IF (INDEX(tgroup(ig),hydroxy).NE.0) THEN
        mult = 18.
        arrhc(1)=arrhc(1)*mult
      ENDIF
      IF ((INDEX(tgroup(ig),aldehyde).NE.0).AND.(nca.GT.3)) THEN
        mult =  -14.5 + 21.4 *(1-exp(-0.43*nca))
        arrhc(1)=arrhc(1)*mult
      ENDIF

* on alpha carbons:
      nether = 0
      DO 10 i=1,mca
        mult = 1.
        IF (tbond(ig,i).NE.0) THEN
* simple alkyl:
          IF (tgroup(i)(1:3).EQ.methyl) THEN
            mult =  1.
            arrhc(1)=arrhc(1)*mult
          ENDIF
          IF(tgroup(i)(1:4) .EQ. 'CH2 ') THEN
            DO j=1,mca
              IF ((tbond(i,j).EQ.3).AND.(j.ne.ig)) THEN
                mult =  1.
              ENDIF
            ENDDO
            arrhc(1)=arrhc(1)*mult
          ENDIF
          IF(tgroup(i)(1:4) .EQ. 'CH2(') THEN
            mult =  1.02
            arrhc(1)=arrhc(1)*mult
          ENDIF

          IF(tgroup(i)(1:3) .EQ. 'CH ' ) THEN
            DO j=1,mca
              IF ((tbond(i,j).EQ.3).AND.(j.ne.ig)) THEN
                mult =  10.
              ELSE 
                mult =  1.61
              ENDIF
            ENDDO
            arrhc(1)=arrhc(1)*mult
          ENDIF
          IF(tgroup(i)(1:3) .EQ. 'CH(' ) THEN
            DO j=1,mca
              IF ((tbond(i,j).EQ.3).AND.(j.ne.ig)) THEN
                mult =  10.
              ELSE 
                mult =  1.61
              ENDIF
            ENDDO
            arrhc(1)=arrhc(1)*mult
          ENDIF

          IF(tgroup(i)(1:2) .EQ. 'C('  ) THEN
            DO j=1,mca
              IF ((tbond(i,j).EQ.3).AND.(j.ne.ig)) THEN
                mult =  48.
              ELSE 
                mult =  2.03
              ENDIF
            ENDDO
            arrhc(1)=arrhc(1)*mult
          ENDIF
          IF(tgroup(i)(1:2) .EQ. 'C '  ) THEN
            DO j=1,mca
              IF ((tbond(i,j).EQ.3).AND.(j.ne.ig)) THEN
                mult =  48.
              ELSE 
                mult =  2.03
              ENDIF
            ENDDO
            arrhc(1)=arrhc(1)*mult
          ENDIF

* overwrite for carbonyls, alcohols and double bonds values 
          IF (tgroup(i)(1:2).EQ.carbonyl) THEN
            mult = 0.64
            arrhc(1)=arrhc(1)*mult
          ENDIF
          IF (tgroup(i)(1:3).EQ.aldehyde) THEN
            mult =  143
            arrhc(1)=arrhc(1)*mult
          ENDIF
          IF (tgroup(i)(1:2).EQ.'Cd') THEN
            mult =  1.
            arrhc(1)=arrhc(1)*mult
          ENDIF

* Ethers and esters
* For acetal, consider only one -O- influence
          IF (INDEX(tgroup(i),ether).NE.0) THEN
            DO j=1,mca
              IF ((tbond(i,j).EQ.3).AND.(j.ne.ig)) THEN
!! ether in alpha of ig
         	IF (tgroup(j)(1:3).EQ.methyl) THEN
         	  mult =  130.
         	  arrhc(1)=arrhc(1)*mult
         	ENDIF
         	IF(tgroup(j)(1:4) .EQ. 'CH2 ') THEN
         	  mult =  58.
         	  arrhc(1)=arrhc(1)*mult
         	ENDIF
         	IF(tgroup(j)(1:4) .EQ. 'CH2(') THEN
         	  mult =  58.
         	  arrhc(1)=arrhc(1)*mult
         	ENDIF
         	IF(tgroup(j)(1:3) .EQ. 'CH ' ) THEN
         	  mult =  23.
         	  arrhc(1)=arrhc(1)*mult
         	ENDIF
         	IF(tgroup(j)(1:3) .EQ. 'CH(' ) THEN
         	  mult =  23.
         	  arrhc(1)=arrhc(1)*mult
         	ENDIF
         	IF(tgroup(j)(1:2) .EQ. 'C('  ) THEN
         	  mult =  495.
         	  arrhc(1)=arrhc(1)*mult
         	ENDIF
         	IF(tgroup(j)(1:2) .EQ. 'C '  ) THEN
         	  mult =  495.
         	  arrhc(1)=arrhc(1)*mult
         	ENDIF
              ENDIF                 
            ENDDO
           ENDIF
        ENDIF
10    CONTINUE

* exit point
20    CONTINUE
* end of rabsno3

      RETURN
      END

