!***********************************************************************
! MASTER MECHANISM - ROUTINE NAME : abcde_map                          *
!                                                                      *
! PURPOSE :                                                            *
!  Browse the bond matrix to find all possible node at position 1 to 6 *
!  (i.e. alpha, beta, gamma ...), starting from node 'top'. Ring are   *
!  allowed (track is stopped when a full loop along the circle is made).*
!                                                                      *
! The subroutine call "gettrack" which give all the possible track,    *
! starting from top.                                                   *
!  track(*,2) give all nodes in alpha position regarding node top      *
!  track(*,3) give all nodes in beta position regarding node top       *
!  track(*,4) give all nodes in gamma position regarding node top      *
!  etc...                                                              *
!                                                                      *
!                                                                      *
! INPUT:                                                               *
! - bond(i,j)    : carbon-carbon bond matrix                           *
! - nca          : number of group                                     *
! - top          : starting node number                                *
!                                                                      *
! OUTPUT:                                                              *
! - nabcde(k)    : number of distinct pathways that end up at a        *
!                  position k relative to top (e.g. nabcde(3) gives    *
!                  the number of distinct pathways finishing in a      *
!                  beta position relative to top                       * 
! - tabcde(k,i,j) : give the pathways (node j), for the track number i * 
!                   to reach the position k (k=2 is beta position ...).*
!                   For example, tabcde(4,1,j) give the first track to *
!                   reach a gamma position (node given by              *    
!                   tabcde(4,1,4), using the track given by            *
!                   tabcde(4,1,*)                                      *
!***********************************************************************
      SUBROUTINE abcde_map(bond,top,nca,nabcde,tabcde)
      IMPLICIT NONE
      INCLUDE 'general.h'

! input
      INTEGER, INTENT(in)  :: top
      INTEGER, INTENT(in)  :: nca
      INTEGER, INTENT(in)  :: bond(mca,mca)

! output:
      INTEGER, INTENT(out)  :: nabcde(9), tabcde(9,mco,mca)

* internal:
      INTEGER  :: track(mco,mca)
      INTEGER  :: trlen(mco)
      INTEGER  :: ntr
      INTEGER  :: ltabc(9,mco,mca)

      INTEGER  :: i,j,k,ii

! initialize
      nabcde(:)=0
      tabcde(:,:,:)=0
      ltabc(:,:,:)=0

! get all tracks starting at top
      CALL gettrack(bond,top,nca,ntr,track,trlen)

! range the track for each length (up to 9)
! Remember that 2=alpha position, 3=beta ... 
      IF (ntr.gt.mco) STOP 'in abcde_map, ntr > mco'
      DO k=2,9
        DO i=1,ntr
          IF (trlen(i).ge.k) THEN
	    DO j=1,k
	      ltabc(k,i,j)=track(i,j)
	    ENDDO
          ENDIF        
	ENDDO        
      ENDDO      
 
! avoid duplicate - set 0 to duplicate nodes tracks
      DO k=2,9
        DO 100 i=1,mca-1
	  IF (ltabc(k,i,k).eq.0) CYCLE
	  DO ii=i+1,mca
	    DO j=1,k
	      IF (ltabc(k,i,j).ne.ltabc(k,ii,j)) GOTO 100
	    ENDDO
	    DO j=1,k             ! If that point is reached, track i= track ii
	      ltabc(k,ii,j)=0
	    ENDDO
	  ENDDO
100	CONTINUE
      ENDDO

       
! get nodes
      DO k=2,9
        DO i=1,ntr
          IF (ltabc(k,i,k).ne.0) THEN
            nabcde(k)=nabcde(k)+1     ! add one more pathway that end up at a k position
	    DO j=1,k
	      ii=nabcde(k)
	      tabcde(k,ii,j)=ltabc(k,i,j)
	    ENDDO
          ENDIF
        ENDDO
      ENDDO

      END
