FORWARD_FUNCTION ir_dims, ir_all, ir_data, ir_attribute

FUNCTION get_irwg, file, varname, data, nobs, nalt, attrname=attrname, dump=dump, all=all, vrbs=vrbs
;+
; NAME:
;  GET_IRWG
;
; PURPOSE:
;       extract data from IRWG HDF GEOMS compatible file
;
; CATEGORY:
;        HDF GEOMS file reader
;
; CALLING SEQUENCE:
;       status = GET_IRWG(file, varname, data)
;
; INPUTS:
;       file is the file name of the HDF file to read
;       varname is the name of specific IRWG HDF object to find 
;
; KEYWORD PARAMETERS:
;       attribute must be set if varname is an attribute object (GET_IRWG,/attribute, ...)
;
; OUTPUTS:
;  data is the returned IDL data variable name
;       returns 0 for success and -1 for failure
;
; EXAMPLES:
;
; 1) extract datsets
;
; extract the 'HCl.MIXING.RATIO_ABSORPTION.SOLAR' dataset and put into variable vmr
; status = get_irwg( 'irwg_geoms.hdf','HCl.MIXING.RATIO_ABSORPTION.SOLAR',vmr)
;
; 2) extract attributes associated with a particular dataset
;
;
; 3) extract attributes associated with groups
;
;
; 4) dump contents of file
; status = get_irwg( 'irwg_geoms.hdf', /dump )
;
; MODIFICATION HISTORY:
;     PSM 11 APR 2014 Corrected AVK indexing to be [i,*] since the indexes are kernel, layer
;                     Changed to use new plot function not old plot subroutine
;                     Added plot of Uncertainty Random and Systematic 
;                     Corrected dataset names and attribute names (does not change operation)
;     JWH 30 NOV 2010, Initial writing
;     Version 0.5
;-

   forward_function fillnames, ir_dump, ir_data, ir_attribute, ir_all, $
                    ir_mkstruc, ir_dataname, ir_fileattr

;print, varname
;print, attrname
;print, n_elements(attrname)

   ; Start the SD interface:
   SDinterface_id = HDF_SD_START( file, /READ)

   if( SDinterface_id eq -1 )then message, "file does not exist."

   status = ir_dims( SDinterface_id, nobs, nalt )

   if( keyword_set( dump ))then begin

      status = ir_dump( SDinterface_id )

   endif else if( keyword_set( all ))then begin

      status = ir_all( SDinterface_id, data )

   endif else begin

      if( n_elements( attrname ) eq 0 )then begin

         status = ir_data( SDinterface_id, varname, data, nobs, nalt, vrbs=vrbs )

      endif else begin

          status = ir_attribute( SDinterface_id, varname, data, attrname )

      endelse

   endelse

   ; End the SD interface:
   HDF_SD_END, SDinterface_id

   return, status

end

;-------------------------------------------------------------------------------
function ir_dims, SDinterface_id, nobs, nalt

   HDF_SD_FILEINFO, SDinterface_id, datasets, attributes

   ; Get number of altitudes
   index        = HDF_SD_NAMETOINDEX( SDinterface_id, 'ALTITUDE' )
   SDdataset_id = HDF_SD_SELECT( SDinterface_id, index )
   dim_id       = HDF_SD_DIMGETID( SDdataset_id, 0 )
   HDF_SD_DIMGET, dim_id, NAME=name, COUNT=nalt
   ;print, 'Altitudes  : ', nalt

   ; Get number of blocks of data
   index        = HDF_SD_NAMETOINDEX( SDinterface_id, 'DATETIME' )
   SDdataset_id = HDF_SD_SELECT( SDinterface_id, index )
   dim_id       = HDF_SD_DIMGETID( SDdataset_id, 0 )
   HDF_SD_DIMGET, dim_id, NAME=name, COUNT=nobs
   ;print, 'Datablocks : ', nobs

return, 0
end


;-------------------------------------------------------------------------------
function ir_data, SDinterface_id, varname, thisdata, nobs, nalt, vrbs=vrbs


   ; Get number of altitudes
   ; Get number of blocks of data
   rc = ir_dims( SDinterface_id, nobs, nalt )

   ; Access index of data in the HDF file by dataset name
   index = HDF_SD_NAMETOINDEX( SDinterface_id, varname )
   if( index eq -1 ) then begin
      print, 'ERROR: irwg_data: Invalid variable name : "', varname, '" exiting...'
      return, index
   endif

   ; Get dataset id for this index
   SDdataset_id = HDF_SD_SELECT( SDinterface_id, index )
   ;print, SDdataset_id

   ; Get data for this dataset / index / varname
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE ;COUNT=count, START=vector] [, STRIDE=vector]

   if(keyword_set(vrbs))then help, thisdata
   ;print , 'ir_data'

   HDF_SD_GETINFO, SDdataset_id, DIMS=dims, FILL=fill, FORMAT=format, HDF_TYPE=hdftype, LABEL=label, NAME=name, NATTS=natts, NDIMS=ndims, RANGE=range, TYPE=type, UNIT=unit ; /NOREVERSE
   if(keyword_set(vrbs))then help, name, natts, ndims, dims, unit

   if(keyword_set(vrbs))then begin
      for j=0, natts-1 do begin
         print, '  Attribute : ', j
         HDF_SD_ATTRINFO, SDdataset_id, j, COUNT=count, DATA=data, HDF_TYPE=hdftype, NAME=name, TYPE=type
         ;help, count, data, hdftype, name, type
         help, name, data
      endfor
   endif

   return, 0

   if(keyword_set(vrbs))then begin
      for j=0, ndims-1 do begin
         print, '  Dimension : ', j
         dim_id = HDF_SD_DIMGETID( SDdataset_id, j )
         ;HDF_SD_DIMGET, dim_id, NAME=name, LABEL=label, FORMAT=format, UNIT=unit, SCALE=scale, COUNT=count, NATTR=natt, TYPE=type
         ;help, name, label, format, unit, scale, count, natt, type
         HDF_SD_DIMGET, dim_id, NAME=name, LABEL=label, FORMAT=frmt, UNIT=unit, COUNT=count, NATTR=natt, TYPE=type
         help, name, label, frmt, unit, count, natt, type
      endfor
   endif

   return, 0

end


;-------------------------------------------------------------------------------
function ir_all, SDinterface_id, data

   HDF_SD_FILEINFO, SDinterface_id, datasets, attributes

   ; Get number of altitudes
   ; Get number of blocks of data
   rc = ir_dims( SDinterface_id, n, m )
   rc = ir_mkstruc( m, n, data, dex )
   rc = ir_dataname( name )

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[0] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].datetime = thisdata
   ;help, thisdata
   ;print, data[*].datetime, jdf_2_datetime( data[*].datetime, /mjd2000 )
   ;stop

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[1] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].surface_pressure = thisdata

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[2] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].surface_temperature = thisdata

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[3] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].pressure[*] = thisdata ;transpose(thisdata)

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[4] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].temperature = thisdata ;transpose(thisdata)

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[5] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].mixing_ratio = thisdata ;transpose(thisdata)

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[6] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].mixing_ratio_apriori = thisdata ;transpose(thisdata)

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[7] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].mixing_ratio_avk = thisdata

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[8] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   ;help, thisdata
   ;help, data
   ;help, hdfdata
   data[*].integration_time = thisdata

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[9] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].mixing_ratio_uncert_ran  = thisdata

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[10] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].mixing_ratio_uncert_sys = thisdata

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[11] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].column_partial = thisdata

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[12] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].column_partial_apriori = thisdata

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[13] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].column = thisdata

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[14] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].column_apriori = thisdata

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[15] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].column_avk = thisdata

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[16] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].column_uncert_ran = thisdata

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[17] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].column_uncert_sys = thisdata

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[18] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].angle_solar_zenith = thisdata

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[19] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].angle_solar_azimuth = thisdata

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[20] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].h2o_mixing_ratio = thisdata

   SDdataset_id = HDF_SD_SELECT( SDinterface_id, dex[21] )
   HDF_SD_GETDATA, SDdataset_id, thisdata ; /NOREVERSE
   data[*].h2o_column = thisdata

   ;help, data

   return, 0

end


;-------------------------------------------------------------------------------
function ir_attribute, SDinterface_id, varname, data, attrname

   ; print, varname, ' ', attrname

   ; file attribute?
   HDF_SD_FILEINFO, SDinterface_id, datasets, attributes
   ;print, 'File attributes : ', attributes
   for j=0, attributes-1 do begin

      HDF_SD_ATTRINFO, SDinterface_id, j, NAME=name, COUNT=count, DATA=data, HDF_TYPE=hdftype, TYPE=type

      if( name eq varname )then begin
         ;print, j, name, varname
         help, name, count, data, hdftype, type
         return, 0
      endif

   endfor

   ; dataset variable attribute
   index = HDF_SD_NAMETOINDEX( SDinterface_id, varname )
   ;print, 'varname index ', index

   ;SDdataset_id = HDF_SD_SELECT( SDinterface_id, index )

   if( index ne -1 ) then begin

      SDdataset_id = HDF_SD_SELECT( SDinterface_id, index )
      ;print, 'SDdataset_id : ', SDdataset_id

      aindex = HDF_SD_ATTRFIND(SDdataset_id, attrname)
      ;print, 'Attribute index : ', aindex

      if( aindex eq -1 )then begin

         HDF_SD_GETINFO, SDdataset_id, CALDATA=caldata, COORDSYS=coordsys, DIMS=dims, FILL=fill, FORMAT=format, HDF_TYPE=hdftype, LABEL=label, NAME=name, NATTS=natts, NDIMS=ndims, RANGE=range, TYPE=type, UNIT=unit ; /NOREVERSE
         ;help, caldata, coordsys, dims, fill, format, hdftype, label, name, natts, ndims, range, type, unit

         for j=0, natts-1 do begin

            HDF_SD_ATTRINFO, SDdataset_id, j, COUNT=count, DATA=data, HDF_TYPE=hdftype, NAME=name, TYPE=type
            ;help, count, data, hdftype, name, type
            ;help, name, data

         endfor

      endif else begin
;print, 'here'
         HDF_SD_ATTRINFO, SDdataset_id, aindex, name=name, COUNT=count, DATA=data, HDF_TYPE=hdftype, TYPE=type
         ;help, name, count, data, hdftype, type

      endelse

   endif

   return, 0

end



;-------------------------------------------------------------------------------
function ir_dump, SDinterface_id

   ; Retrieve info about the dataset:
   HDF_SD_FILEINFO, SDinterface_id, datasets, attributes
   print, 'Interface_id : ', SDinterface_id

   ; Print information about the returned variables:
   HELP, datasets, attributes


   print,''
   print, '***File attributes***'
   print,''


   for j=0, attributes-1 do begin

      HDF_SD_ATTRINFO, SDinterface_id, j, name=name, COUNT=count, DATA=data, HDF_TYPE=hdftype, TYPE=type

      print, 'Attribute : ', j
      ;help, name, count, data, hdftype, type
      help, name, data

   endfor


   print,''
   print, '***Datasets***'
   print,''


   for i=0, datasets-1 do begin

      ; Access the first SD in the HDF file:
      SDdataset_id = HDF_SD_SELECT(SDinterface_id, i)
      print,''
      print,'Dataset Index  : ', i, SDdataset_id
      print,''

      ; Get info for this dataset
      ;HDF_SD_GETINFO, SDdataset_id, CALDATA=caldata, COORDSYS=coordsys, DIMS=dims, FILL=fill, FORMAT=format, HDF_TYPE=hdftype, LABEL=label, NAME=name, NATTS=natts, NDIMS=ndims, RANGE=range, TYPE=type, UNIT=unit ; /NOREVERSE
      ;help, caldata, coordsys, dims, fill, format, hdftype, label, name, natts, ndims, range, type, unit

      HDF_SD_GETINFO, SDdataset_id, DIMS=dims, FILL=fill, FORMAT=format, HDF_TYPE=hdftype, LABEL=label, NAME=name, NATTS=natts, NDIMS=ndims, RANGE=range, TYPE=type, UNIT=unit ; /NOREVERSE
      help, name, natts, ndims, dims, unit

      for j=0, natts-1 do begin

         print, '  Attribute : ', j
         HDF_SD_ATTRINFO, SDdataset_id, j, COUNT=count, DATA=data, HDF_TYPE=hdftype, NAME=name, TYPE=type

         ;help, count, data, hdftype, name, type
         help, name, data

      endfor

      for j=0, ndims-1 do begin

         print, '  Dimension : ', j
         dim_id = HDF_SD_DIMGETID( SDdataset_id, j )

         ;HDF_SD_DIMGET, dim_id, NAME=name, LABEL=label, FORMAT=format, UNIT=unit, SCALE=scale, COUNT=count, NATTR=natt, TYPE=type
         ;help, name, label, format, unit, scale, count, natt, type

         HDF_SD_DIMGET, dim_id, NAME=name, LABEL=label, FORMAT=format, UNIT=unit, COUNT=count, NATTR=natt, TYPE=type
         help, name, label, format, unit, count, natt, type

      endfor

   endfor

   return, 0

end


;-------------------------------------------------------------------------------
function ir_mkstruc, m, n, data, dex

   dex = [ 0, 4, 5, indgen( 19 ) +8 ]

   print, ' Datasets in structure : ', n
   print, ' Altitudes in profiles : ', m

   a = { hdfdata, $
         datetime                 : 0.0D0, $
         surface_pressure         : 0.0d0, $
         surface_temperature      : 0.0d0, $
         pressure                 : dblarr(m), $
         temperature              : dblarr(m), $
         ;altitude                 : dblarr(m), $
         mixing_ratio             : dblarr(m), $
         mixing_ratio_apriori     : dblarr(m), $
         mixing_ratio_avk         : dblarr(m,m), $
         integration_time         : 0.0d0, $
         mixing_ratio_uncert_ran  : dblarr(m,m), $
         mixing_ratio_uncert_sys  : dblarr(m,m), $
         column_partial           : dblarr(m), $
         column_partial_apriori   : dblarr(m), $
         column                   : 0.0d0, $
         column_apriori           : 0.0d0, $
         column_avk               : dblarr(m), $
         column_uncert_ran        : 0.0d0, $
         column_uncert_sys        : 0.0d0, $
         angle_solar_zenith       : 0.0d0, $
         angle_solar_azimuth      : 0.0d0, $
         h2o_mixing_ratio         : dblarr(m), $
         h2o_column               : 0.0d0 }

   ;help, a

   data = replicate( {hdfdata}, n )
   return, 0

end


;-------------------------------------------------------------------------------
function ir_dataname, names

;   rc = fillnames( names )

   names = [ $
   'DATETIME', $
   'LATITUDE.INSTRUMENT', $
   'LONGITUDE.INSTRUMENT', $
   'ALTITUDE.INSTRUMENT', $
   'SURFACE.PRESSURE_INDEPENDENT', $
   'SURFACE.TEMPERATURE_INDEPENDENT', $
   'ALTITUDE', $
   'ALTITUDE.BOUNDARIES', $
   'PRESSURE_INDEPENDENT', $
   'TEMPERATURE_INDEPENDENT', $
   'X.MIXING.RATIO_ABSORPTION.SOLAR', $
   'X.MIXING.RATIO_ABSORPTION.SOLAR_APRIORI', $
   'X.MIXING.RATIO_ABSORPTION.SOLAR_AVK', $
   'INTEGRATION.TIME', $
   'X.MIXING.RATIO_ABSORPTION.SOLAR_UNCERTAINTY.RANDOM.COVARIANCE', $
   'X.MIXING.RATIO_ABSORPTION.SOLAR_UNCERTAINTY.SYSTEMATIC.COVARIANCE', $
   'X.COLUMN.PARTIAL_ABSORPTION.SOLAR', $
   'X.COLUMN.PARTIAL_ABSORPTION.SOLAR_APRIORI', $
   'X.COLUMN_ABSORPTION.SOLAR', $
   'X.COLUMN_ABSORPTION.SOLAR_APRIORI', $
   'X.COLUMN_ABSORPTION.SOLAR_AVK', $
   'X.COLUMN_ABSORPTION.SOLAR_UNCERTAINTY.RANDOM.STANDARD', $
   'X.COLUMN_ABSORPTION.SOLAR_UNCERTAINTY.SYSTEMATIC.STANDARD', $
   'ANGLE.SOLAR_ZENITH.ASTRONOMICAL', $
   'ANGLE.SOLAR_AZIMUTH', $
   'H2O.MIXING.RATIO.VOLUME_ABSORPTION.SOLAR', $
   'H2O.COLUMN_ABSORPTION.SOLAR' ]

   return, 0
end


;-------------------------------------------------------------------------------
function ir_fileattr, names

   names = [ $
   'PI_NAME', $
   'PI_AFFILIATION', $
   'PI_ADDRESS', $
   'PI_EMAIL', $
   'DO_NAME', $
   'DO_AFFILIATION', $
   'DO_ADDRESS', $
   'DO_EMAIL', $
   'DS_NAME', $
   'DS_AFFILIATION', $
   'DS_ADDRESS', $
   'DS_EMAIL', $
   'DATA_DESCRIPTION', $
   'DATA_DISCIPLINE', $
   'DATA_GROUP', $
   'DATA_LOCATION', $
   'DATA_SOURCE', $
   'DATA_VARIABLES', $
   'DATA_START_DATE', $
   'DATA_STOP_DATE', $
   'DATA_FILE_VERSION', $
   'DATA_MODIFICATIONS', $
   'DATA_TEMPLATE', $
   'DATA_QUALITY', $
   'DATA_CAVEATS', $
   'DATA_RULES_OF_USE', $
   'DATA_ACKNOWLEDGEMENT', $
   'FILE_DOI', $
   'FILE_NAME', $
   'FILE_GENERATION_DATE', $
   'FILE_ACCESS', $
   'FILE_PROJECT_ID', $
   'FILE_ASSOCIATION', $
   'FILE_META_VERSION' ]

   return, 0
end


;-------------------------------------------------------------------------------
pro testirwg, file=file, var=var, data=data, dump=dump, attrname=attrname

;print, " get_irwg( /dump)..."
;print, get_irwg( file, var, hdf, /dump )
;stop

   if( not keyword_set( file ))then $
<<<<<<< HEAD
   file = 'groundbased_ftir.hcn_ncar002_thule_20070306t170734z_20121013t163412z_001.hdf'
=======
    file = "/data1/ebaumer/tab/co/ErrTest/groundbased_ftir.co_ncar001_thule_19991010t143248z_19991010t181430z_001.hdf"
   ;file = "/Users/manning/Documents/HDF/testdata/groundbased_ftir.hcn_ncar002_thule_20070306t170734z_20071002t180229z_001.hdf"
   ;file = "groundbased_ftir.o3_ncar002_mauna.loa.hi_20080211t180913z_20080216t175139z_001.hdf"
>>>>>>> 05b1b9232ec1924e368bed2100138a02811effd8
   ;file = "groundbased_ftir.o3_ncar002_mauna.loa.hi_20080211t180913z_20080216t175139z_001.hdf"
   ;file = 'groundbased_ftir.o3_ncar002_mauna.loa.hi_20100103t194708z_20100228t193537z_001.hdf'
   ;file = 'groundbased_ftir.hcl_ncar001_thule_20090228t145607z_20091019t163419z_001.hdf'

   print, get_irwg( file, var, hdf, nobs, nalt, /all )
   help, hdf
   print, ' NOBS : ', nobs, nalt

   status = get_irwg( file, 'ALTITUDE', alt ) ; alt is the midpoints
   ;print, alt
   ;status = get_irwg( file, 'ALTITUDE.BOUNDARIES', altb )

   ;;!P.MULTI = [0,2,1,0,0]
   ;;set_plot, 'X'
   ;;device, decompose=0
   ;;tek_color
   top=60.

   xr = [0.,10.5e-6]
   for j = 0, nobs -1 do begin

      dt = jdf_2_datetime( hdf[j].datetime, /mjd2000 )
      datestr = string( dt, format="(i4,'/',i02,'/',i02,2x,i02,':',i02,':',f04.1)")

      p=plot( hdf[j].mixing_ratio, alt, title="Mix Ratio & Apriori "+datestr, yrange = [0,top], xrange=xr,layout=[4,1,1])
      p=plot( hdf[j].mixing_ratio_apriori, alt, color="g", /overplot)
   ;;   plot, hdf[j].mixing_ratio, alt, title=datestr, yrange = [0,top], xrange=xr
   ;;   oplot, hdf[j].mixing_ratio_apriori, alt, color=3

      minx = min(hdf[j].mixing_ratio_avk[*,*])
      maxx = max(hdf[j].mixing_ratio_avk[*,*])
      p=plot( hdf[j].mixing_ratio_avk[0,*], alt, title="AVK "+datestr, /nodata, $
            yrange = [0,top], xrange=[minx,maxx], xstyle=1,layout=[4,1,2],/current)
   ;;   plot, hdf[j].mixing_ratio_avk[0,*], alt, title=datestr, /nodata, $
   ;;         yrange = [0,top], xrange=[-1.,2.0], xstyle=1

      for i=0, nalt-1 do begin
         p=plot( hdf[j].mixing_ratio_avk[i,*], alt, vert_colors = c, RGB_TABLE = i mod 40, /overplot) ; reverse of original array orde
   ;;      oplot, hdf[j].mixing_ratio_avk[i,*], alt, color = i+1
      endfor
      
      minx = min(hdf[j].mixing_ratio_uncert_sys[*,*])
      maxx = max(hdf[j].mixing_ratio_uncert_sys[*,*])
      p=plot( hdf[j].mixing_ratio_uncert_sys[0,*], alt, title="Uncert Sys "+datestr, /nodata, $
            yrange = [0,top], xrange=[minx,maxx], xstyle=1,layout=[4,1,3],/current)
      for i=0, nalt-1 do begin
         p=plot( hdf[j].mixing_ratio_uncert_sys[i,*], alt, vert_colors = c, RGB_TABLE = i mod 40, /overplot) 
      endfor

      minx = min(hdf[j].mixing_ratio_uncert_ran[*,*])
      maxx = max(hdf[j].mixing_ratio_uncert_ran[*,*])
      p=plot( hdf[j].mixing_ratio_uncert_ran[0,*], alt, title="Uncert Rand "+datestr, /nodata, $
            yrange = [0,top], xrange=[minx,maxx], xstyle=1,layout=[4,1,4],/current)
      for i=0, nalt-1 do begin
         p=plot( hdf[j].mixing_ratio_uncert_ran[i,*], alt, vert_colors = c, RGB_TABLE = i mod 40, /overplot)
      endfor

      des = -1
      read, des, prompt=' 0=quit, 1=next, n=goto nth obs : '
      if( des eq 0 )then break;
      if( des gt 1 ) then j = des
   endfor

   ;stop

   ;for i=0, nobs-1 do begin
   ;   dt = jdf_2_datetime( hdf[i].datetime, /mjd2000 )
   ;   datestr = string( dt, format="(i4,'/',i02,'/',i02,2x,i02,':',i02,':',f04.1)")
   ;   print, i, hdf[i].datetime, '  ', datestr
   ;   for j=0, nalt-1 do print, alt[j],  hdf[i].mixing_ratio[j], hdf[i].temperature[j], hdf[i].pressure[j]
   ;endfor

;stop
end
