#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>

// Local functions
#include"constant.h"
#include"meeus.h"
#include"opus.h"
#include"bnr.h"

#define VERBOSE FALSE

// variables used in nearly all functions
FILE *opus;

char specblock[] ="NONE\0";

int fltr=0, datablocktype=0, paramblocktype=0;

float aperture=-1.;

double field_of_view=-1.;

struct utc_date collection_date, utc_coll_date;
struct utc_time collection_time, utc_coll_time;

struct coord res;

struct loc site;

struct AQPB_rcd AQPB;                     // Acquisition block
// Phase, Interferogram, Emission, Single Channel, Interferogram #2, Single Chan #2 block
struct SPPB_rcd IFPB, EMPB, SPPB, PHPB, SCPB, I2PB, S2PB, TRAN;
struct FTPB_rcd FTPB;                     // Fourier Transform block
struct OPPB_rcd OPPB;                     // Optic block
struct INPB_rcd INPB;                     // Instrument block


void skipComment( FILE *input) {
    char buffer[81];
    int i=0;
    memset( buffer, '\0', strlen( buffer ) );
    //fprintf( stderr, "%i  %s\n", -1, buffer );
    while( buffer[0] != '$' )  {
       //fprintf( stderr, "%i  %s\n", i, buffer );
       fscanf( input, "%80s", buffer );
       //fprintf( stderr, "%i  %s\n", i++, buffer );
   }
}

int write_bnr( char *blocktype, int SWAP, int ftype ) {

   FILE *bnr, *txt;

   char title[80], bnrname[12], cdum[ENUMSZ];

   int fnum_points, length, pointer;

   float *ydata;

   double fstartw, fstopw, spacing;

   strncpy( title, bnr_makeheader( utc_coll_date, utc_coll_time, res.zen, res.az,
                        INPB.DUR, AQPB.RES, FTPB.APF, field_of_view), HEADLEN );

	fprintf( stderr, "  BNR Header : '%s' %ld\n", title, strlen(title) );

   fstartw     = SPPB.FXV;
   fstopw      = SPPB.LXV;
   fnum_points = SPPB.NPT;
   spacing = (fstopw-fstartw) / (fnum_points-1);
   if( spacing < 0.0 ) spacing = (fstartw-fstopw) / (fnum_points-1);

   fprintf( stderr, "  %12.4lf to %12.4lf %8d points spaced at %12.8lf\n", fstartw, fstopw, fnum_points, spacing );
	fprintf( stderr, "  Reading Spectra data block : %Xx\n", datablocktype );

   if( ! opus_searchdir( opus, datablocktype, &length, &pointer, SWAP )) {
      fprintf( stderr," *Cannot find data block : %s %Xx\n", blocktype, datablocktype );
      fclose( opus );
      return(1);
   } else fprintf( stderr,"  %s Data block Found.\n", blocktype );

	ydata = (float *) malloc ( length*sizeof(float) );

	if( ydata == (float *) 0 ) {
		fprintf( stderr, " *Could not allocate array for y data.\n" );
		return(2);
	}

	if( opus_read_spectrum( opus, &fstartw, &fstopw, length, pointer, SPPB.CSF, SPPB.NPT, ydata, SWAP )) {
		fprintf( stderr," *Cannot read data.\n" );
		return(3);
	}

// 	fprintf( stderr, "\tFirst Data points\n");
// 	fprintf( stderr, "\t%11.6f %11.9e\n", fstartw+(0*spacing),ydata[0] ) ;
// 	fprintf( stderr, "\t%11.6f %11.9e\n", fstartw+(1*spacing),ydata[1] ) ;
// 	fprintf( stderr, "\t%11.6f %11.9e\n", fstartw+(2*spacing),ydata[2] ) ;
// 	fprintf( stderr, "\n\tLast Data points\n");
// 	fprintf( stderr, "\t%11.6f %11.9e\n", fstartw+((length-3)*spacing), ydata[length-3]) ;
// 	fprintf( stderr, "\t%11.6f %11.9e\n", fstartw+((length-2)*spacing), ydata[length-2]) ;
// 	fprintf( stderr, "\t%11.6f %11.9e\n", fstartw+((length-1)*spacing), ydata[length-1]) ;

	sprintf( bnrname, "%4s.bnr", blocktype );
	fprintf( stderr, "  Opening bnr file: %s\n", bnrname );

	if( bnr_write_init( &bnr, bnrname )) {
		fprintf( stderr, " *Cannot create %s bnr file.\n", bnrname );
		return(4);
	}

   if( ftype == 'F' ) {
      printf("   Writing FORTRAN file\n");
      if( bnr_write_header_F( bnr, title, fstartw, fstopw, spacing, length )) {
         fprintf( stderr, " *Cannot write header of %s bnr file.\n", bnrname );
         return(5);
      }

      if( bnr_write_data_F( bnr, ydata, length )) {
         fprintf( stderr, " *Cannot write data of %s bnr file.\n", bnrname );
         return(6);
      }

   } else {
      printf("   Writing C file\n");
      if( bnr_write_header( bnr, title, fstartw, fstopw, spacing, length )) {
         fprintf( stderr, " *Cannot write header of %s bnr file.\n", bnrname );
         return(5);
      }

      if( bnr_write_data( bnr, ydata, length )) {
         fprintf( stderr, " *Cannot write data of %s bnr file.\n", bnrname );
         return(6);
      }
   }

	bnr_write_done( bnr );
	fprintf( stderr, "  Closed bnr file: %s\n", bnrname );

	free(ydata);

   if(( txt = fopen( "ckopus.out", "wa" )) == (FILE *) NULL) return(7);

	fprintf( txt, "%09.4lf\n", res.zen );
	fprintf( txt, "%09.4lf\n", 0.9/AQPB.RES );
	fprintf( txt, "%05.3lf\n", field_of_view );
	fprintf( txt, "%s\n", FTPB.APF );
	fprintf( txt, "%0.lf\n", fstartw );
	fprintf( txt, "%0.lf\n", fstopw );
	fprintf( txt, "%04i %02i %02i %02i %02i %4.1lf\n",
	         utc_coll_date.year, utc_coll_date.month,  utc_coll_date.day,
				utc_coll_time.hour, utc_coll_time.minute, utc_coll_time.second );
	fprintf( txt, "%s\n", title );
	fprintf( txt, "%010.4lf\n", res.az );
        fprintf( txt, "%010.1lf\n", INPB.DUR );
	fprintf( txt, "%010.4lf\n", res.roe );

	fclose( txt );

return(0);

}

int showtime( char *filename, int SWAP ) {

   if( !opus_init( &opus, filename )) {
      fprintf( stderr,"Cannot open %s opus file \n",filename );
      exit(1);
   }

   if( read_SPPAR_block(  opus,IFGMPARBLK, &IFPB, SWAP ))
      printf( " IFGM: %s %s", IFPB.DAT, IFPB.TIM );

   if( read_SPPAR_block( opus, SNGCPARBLK, &SCPB, SWAP ))
      printf( " SNGC: %s %s", SCPB.DAT, SCPB.TIM );

   if( read_SPPAR_block( opus, EMISPARBLK, &EMPB, SWAP ))
      printf( " EMIS: %s %s", EMPB.DAT, EMPB.TIM );

   if( read_SPPAR_block( opus, PHASPARBLK, &PHPB, SWAP ))
      printf( " PHAS: %s %s", PHPB.DAT, PHPB.TIM );

   if( read_SPPAR_block( opus, SNG2PARBLK, &S2PB, SWAP ))
      printf( " SNG2: %s %s", S2PB.DAT, S2PB.TIM );

   if( read_SPPAR_block( opus, IFG2PARBLK, &I2PB, SWAP ))
      printf( " IFG2: %s %s", I2PB.DAT, I2PB.TIM );

   if( read_SPPAR_block( opus, TRANPARBLK, &TRAN, SWAP ))
      printf( " TRAN: %s %s", TRAN.DAT, TRAN.TIM );

   printf("\n");
   fclose( opus );

   return(0);
}


int initialize( char *filename, char *blocktype, int SWAP ) {

   int rc=0, flt=0;
   char filter[]="filter";

   fprintf( stderr, " File: <%s>\n", filename );

   if( !opus_init( &opus, filename )) {
      fprintf( stderr,"Cannot open %s opus file \n",filename );
      return(1);
   }

   if( VERBOSE ) {
      fprintf( stderr, "showinfo: file handle value       : %lx\n", (long) opus );
      fprintf( stderr, "showinfo: file handle address     : %lx\n", (long) &opus );
      //fprintf( stderr, "showinfo: file handle pointing to : %lx\n", (int) *opus );
      fprintf( stderr, "showinfo: opus filename           : %s\n", filename );
   }

   if( read_AQPAR_block( opus, &AQPB, SWAP )) {
      fprintf( stderr,"  Acquisition Param Block Read OK.\n" );
      fprintf( stderr,"     NSS, GSW, GSG          :  %16d %16d %16s\n", AQPB.NSS, AQPB.GSW, AQPB.GSG );
      fprintf( stderr,"     RES, LFW, HFW          :  %16.6lf %16.1lf %16.1lf\n", AQPB.RES, AQPB.LFW, AQPB.HFW );
   } else return(3);

   if( read_FTPAR_block( opus, &FTPB, SWAP )) {
      fprintf( stderr,"  FT Param Block Read OK.\n" );
      fprintf( stderr,"     LFQ, HFQ, APF          :  %16.4lf %16.4lf %16s\n", FTPB.LFQ, FTPB.HFQ, FTPB.APF );
      fprintf( stderr,"     PHR, ZFF, PHZ          :  %16lf %16s %16s\n", FTPB.PHR, FTPB.ZFF, FTPB.PHZ );
   } else return(4);

   if( read_OPPAR_block(opus, &OPPB, SWAP )) {
      fprintf( stderr,"  Optic Param Block Read OK.\n" );
      fprintf( stderr,"     APT, PGN, OPF          :  %16s %16s %16s\n", OPPB.APT, OPPB.PGN, OPPB.OPF );
      fprintf( stderr,"     BMS, DTC, SRC          :  %16s %16s %16s\n", OPPB.BMS, OPPB.DTC, OPPB.SRC );
      fprintf( stderr,"     VEL, LPF, HPF          :  %16s %16s %16s\n", OPPB.VEL, OPPB.LPF, OPPB.HPF );
      sscanf(OPPB.APT," %f ",&aperture); //,units); // in mm
      sscanf(OPPB.OPF," %s %i ", filter, &flt);  // "Filter n ...
      //printf( "%s\n", strpbrk( OPPB.APT, "0123456789" ));
      //printf( "filter # %i\n", flt);
   } else return(5);

   field_of_view = 0.0;
   if( read_INPAR_block( opus, &INPB, SWAP )) {
      fprintf( stderr,"  Instrument Param Block Read OK.\n" );
      fprintf( stderr,"     DUR, ASG, LWN          :  %16.4lf %16d %16lf\n", INPB.DUR, INPB.ASG, INPB.LWN );
      fprintf( stderr,"     GFW, FBW, INS          :  %16d %16d %16s\n", INPB.GFW, INPB.GBW, INPB.INS );
      fprintf( stderr,"     LFL, HFL, SSP          :  %16.4lf %16.4lf %16d\n", INPB.LFL, INPB.HFL, INPB.SSP );
      fprintf( stderr,"     LFF, HFF, DFR          :  %16.4lf %16.4lf %16d\n", INPB.LFF, INPB.HFF, INPB.DFR );
      if( INPB.FOC != 0.0 ) field_of_view = aperture/INPB.FOC * 1000.;
      fprintf( stderr,"     FOV, FOC, ABP          :  %16.4lf %16.4lf %16d\n", field_of_view, INPB.FOC, INPB.ABP );
   } else return(6);

   if( read_SPPAR_block(  opus, IFGMPARBLK, &IFPB, SWAP )) {
      fprintf( stderr,"  Interferogram Param Block Read OK.\n" );
      fprintf( stderr,"     Date and Time String   :  %16s %16s\n", IFPB.DAT, IFPB.TIM );
      fprintf( stderr,"     MNY, MXY, CSF          :  %16.4lf %16.4lf %16.4lf\n", IFPB.MNY, IFPB.MXY, IFPB.CSF );
      fprintf( stderr,"     NPT, DXU               :  %16d %16s\n", IFPB.NPT, IFPB.DXU );
   } else {
      fprintf( stderr,"  Interferogram Param Block Read NOT OK!\n" );
      //return(7);
   }

   if( ! strncmp( blocktype, "NONE", 4 )) {
      datablocktype = SNGCSPCBLK;
      paramblocktype = SNGCPARBLK;
      if( read_SPPAR_block( opus, paramblocktype, &SPPB, SWAP )) {
         fprintf( stderr,"  Single Channel Param Block Read OK.\n" );
         strncpy( specblock, "SNGC", 4 );
         strncpy( blocktype, "SNGC", 4 );
      } else {
         // no sample spectra so check for emission block
         fprintf( stderr, "*Could not read Single Channel Param Block...\n");
         datablocktype = EMISSPCBLK;
         paramblocktype = EMISPARBLK;
         if( read_SPPAR_block( opus, paramblocktype, &SPPB, SWAP )) {
            fprintf( stderr,"  Emission Param Block Read OK.\n" );
            strncpy( specblock, "EMIS", 4 );
            strncpy( blocktype, "EMIS", 4 );
         } else {
            // no emission spectra either so check for transmittance block
            fprintf( stderr, "*Could not read Single Chan or Emission Param Block...\n");
            datablocktype = TRANSPCBLK;
            paramblocktype = TRANPARBLK;
            if( read_SPPAR_block( opus, paramblocktype, &SPPB, SWAP )) {
               fprintf( stderr,"  Transmittance Param Block Read OK.\n" );
               strncpy( specblock, "TRAN", 4 );
               strncpy( blocktype, "TRAN", 4 );
            } else {
               report( "*Error No Emission or Single Channel or Transmittance Spectra block in file", opus_error, filename );
               fclose( opus );
               return(8);
            }
         }
      }
   } else {

      if( ! strncmp( blocktype, "IFGM", 4 )) {
         datablocktype = IFGMSPCBLK;
         paramblocktype = IFGMPARBLK;
         strncpy( specblock, "IFGM", 4 );
      }
      if( ! strncmp( blocktype, "PHAS", 4 )) {
         datablocktype = PHASSPCBLK;
         paramblocktype = PHASPARBLK;
         strncpy( specblock, "PHAS", 4 );
      }
      if( ! strncmp( blocktype, "EMIS", 4 )) {
         datablocktype = EMISSPCBLK;
         paramblocktype = EMISPARBLK;
         strncpy( specblock, "EMIS", 4 );
      }
      if( ! strncmp( blocktype, "SNGC", 4 )) {
         datablocktype = SNGCSPCBLK;
         paramblocktype = SNGCPARBLK;
         strncpy( specblock, "SNGC", 4 );
      }
      if( ! strncmp( blocktype, "TRAN", 4 )) {
         datablocktype = TRANSPCBLK;
         paramblocktype = TRANPARBLK;
         strncpy( specblock, "TRAN", 4 );
      }
      if( ! strncmp( blocktype, "SNG2", 4 )) {
         datablocktype = SNG2SPCBLK;
         paramblocktype = SNG2PARBLK;
         strncpy( specblock, "SNG2", 4 );
      }
      if( ! strncmp( blocktype, "IFG2", 4 )) {
         datablocktype = IFG2SPCBLK;
         paramblocktype = IFG2PARBLK;
         strncpy( specblock, "IFG2", 4 );
      }
      if( paramblocktype != 0 ) {
          if( read_SPPAR_block( opus, paramblocktype, &SPPB, SWAP )) {
            fprintf( stderr,"  %s Param Block Read OK.\n", blocktype );
         } else {
            fprintf( stderr,"  *%s Param Block Read NOT OK!!\n", blocktype );
            fclose( opus );
            return(9);
         }
      } else {
         fprintf( stderr,"** %s Block type invalid.\n", blocktype );
         fclose( opus );
         return(10);
      }
   }

//    printf(" blocktype        %10s\n", blocktype );
//    printf(" paramblocktype   %10Xx\n", paramblocktype );
//    printf(" datablocktype    %10Xx\n", datablocktype );

   fprintf( stderr,"     Date and Time String   :  %16s %16s\n", SPPB.DAT, SPPB.TIM );
   fprintf( stderr,"     FXV, LXV, NPT          :  %16.4lf %16.4lf %16d\n", SPPB.FXV, SPPB.LXV, SPPB.NPT );
   fprintf( stderr,"     MNY, MXY, CSF          :  %16.4lf %16.6lf %16.1lf\n", SPPB.MNY, SPPB.MXY, SPPB.CSF );
   fprintf( stderr,"     NF1, NF2, SN1          :  %16.4lf %16.4lf %16.1lf\n", SPPB.NF1, SPPB.NF2, SPPB.SN1 );

   if( read_SPPAR_block( opus, PHASPARBLK, &PHPB, SWAP )) {
      fprintf( stderr,"  Phase Param Block Read OK.\n" );
      fprintf( stderr,"     Date and Time String   :  %16s %16s\n", PHPB.DAT, PHPB.TIM );
      // get date and time into structures, from strings
      sscanf( PHPB.DAT, "%d/%d/%d", &collection_date.day, &collection_date.month, &collection_date.year );
      sscanf( PHPB.TIM, "%d:%d:%lf", &collection_time.hour, &collection_time.minute, &collection_time.second );

   } else {
      fprintf( stderr,"  PHAS Param Block Read NOT OK!!\n" );
      fprintf( stderr,"  Using DATE & TIME from Param Block : %s\n", specblock );
      fprintf( stderr,"     Date and Time String   :  %16s %16s\n", SPPB.DAT, SPPB.TIM );
      // get date and time into structures, from strings
      sscanf( SPPB.DAT, "%d/%d/%d", &collection_date.day, &collection_date.month, &collection_date.year );
      sscanf( SPPB.TIM, "%d:%d:%lf", &collection_time.hour, &collection_time.minute, &collection_time.second );
   } //return(11);

   adjust_date_time( &collection_date, &collection_time, site.sec_offs );

   utc_coll_date = collection_date;
   utc_coll_time = collection_time;

   adjust_date_time( &utc_coll_date, &utc_coll_time, site.utc_offs*3600 );

   /* calculate zenith angle */
   az_and_zen( &utc_coll_date, &utc_coll_time, site.n_lat, site.w_lon, &res );

	// Calculate the radius of the earth ellipsoid at observation lat & solar azimuth
	rad_of_earth( site.n_lat, site.w_lon, &res );

   fprintf( stderr,"  Corrected UTC Date Time from Phase block: %02d/%02d/%02d\t%02d:%02d:%02.0lf\n",
      utc_coll_date.month, utc_coll_date.day,    utc_coll_date.year,
      utc_coll_time.hour,  utc_coll_time.minute, utc_coll_time.second );

   fprintf( stderr, "     N Lat, W Lon, Alt      :  %16.2lf %16.2lf %16.1f\n", site.n_lat, site.w_lon, site.altud );
   fprintf( stderr, "     S Azi, S Zen, RoE      :  %16.4lf %16.4lf %16.4lf\n\n", res.az, res.zen, res.roe );

   return(0);

}

int showdb( char *filename ) {

   char fword[7];
   int offs=0;

   offs = site.utc_offs*3600 + site.sec_offs;

   printf( "%s %6i %4s %02d/%02d/%02d  %02d:%02d:%02.0lf  ", filename, offs, specblock,
      utc_coll_date.month,  utc_coll_date.day,    utc_coll_date.year,
      utc_coll_time.hour,   utc_coll_time.minute, utc_coll_time.second );

   //printf( "%.2e\t%.2lf\t%.2lf\t%.1f\t",
   printf( "%7.1lf %7.2lf  %8.2lf  %6.1f  ", SPPB.SN1, site.n_lat, site.w_lon, site.altud );

   //printf( "\n%s\n", filter );
   // TAB only
   sscanf( OPPB.OPF, "%6s %i", fword, &fltr );

   //printf( "\n%s\n %s\n %i\n", filter, fword, fltr );

   //printf( "%.2lf\t%.2lf\t%.4lf\t%6.2lf\t%.4lf\t%s\t%.4lf\t%.3lf\t%.3lf\t%i\t%.3le\t%.3le\n",
   printf( "%8.2lf  %6.2lf  %9.4lf  %6.2lf  %6.4lf  %2s  %6.4lf  %9.3lf  %9.3lf  %1i  %.3le  %.3le\n",
      res.az, res.zen, res.roe, INPB.DUR, AQPB.RES, FTPB.APF, field_of_view,
      SPPB.FXV, SPPB.LXV, fltr, SPPB.MXY, SPPB.MNY );

   return(0);

}


int opusmws( char *filename, int SWAP ) {

// this filename is not actal opus filename

	FILE   *inp, *out, *spc;

	char   path[80], mwfn[80][40], buf[80];
	char   blocktype[] = "NONE\0";

	int    j, n1, n2, i, length, pointer, fnum_points, ftype, nmws, rc;

	float  *ydata, temp;
	float  aperature;

	double fstartw, fstopw, spacing;
	double w, scan_time, resolution, focal_length;
	double sum, mws[2][40];
	double field_of_view;

	if(( out = fopen( "cutmws.out", "w" )) == (FILE *) NULL){
	   fprintf( stdout, "Cannot open %s output file.\n", "cutmws.out" );
	   exit(1);
	}

	if(( inp = fopen( "cutmws.inp", "r" )) == (FILE *) NULL){
	   fprintf( stdout, "Cannot open %s input file.\n", "cutmws.inp" );
	   exit(1);
	}

	skipComment( inp );

	/*  Read input file name and type */
	fscanf( inp, "%s", filename );
	fscanf( inp, "%d", &ftype );
	fprintf( out,"Opus file and type %s : %d\n", filename, ftype );
	switch( ftype )
	{
	case 0:
		  fprintf( out, "Only support OPUS files exiting...\n" );
		  exit(0);
		  break;
	 case 1:
		  fprintf( out, "Only support OPUS files exiting...\n" );
		  exit(0);
		  break;
	 case 2:
		  fprintf( out, "Only support OPUS files exiting...\n" );
		  exit(0);
		  break;
	 case 3:
		  fprintf( out, "Opening OPUS file...\n" );
		  break;
	 default:
		  fprintf( out, "Only support OPUS files exiting...\n" );
		  exit(0);
		  break;
	}

	skipComment( inp );

	/* Read number of micro windows */
	fscanf( inp, "%d", &nmws );
	fprintf( out,"# mws : %d\n", nmws );

	skipComment( inp );

	/* Read micro windows */
	/*	for( i=0; i<nmws; i++ ) fscanf( input, "%lf, %lf %s", &mws[0][i], &mws[1][i], rem );*/
	for( i=0; i<nmws; i++ ) fscanf( inp, "%lf,%lf", &mws[0][i], &mws[1][i] );
	for( i=0; i<nmws; i++ ) fprintf( out, "%12.6lf  %12.6lf\n", mws[0][i], mws[1][i] );
   fprintf( out,"End of mws\n" );

	skipComment( inp );

	/* Read output path */
	fscanf( inp, "%s", path );
	fprintf( out, "%s\n", path );

	skipComment( inp );

	/* Read output file names */
	for( i=0; i<nmws; i++ ) fscanf( inp, "%s", mwfn[i] );
	for( i=0; i<nmws; i++ ) fprintf( out, "%s\n", mwfn[i] );
	for( i=0; i<nmws; i++ ) {
		strcpy( buf, path );
		strcat( buf, mwfn[i] );
		strcpy( mwfn[i], buf );
	 }
   fprintf( out,"End of mws filenames\n" );

	for( i=0; i<nmws; i++ ) fprintf( out, "%s\n", mwfn[i] );
   fprintf( out,"End of mws path/filenames\n" );

	fclose( inp );

	if(( rc = initialize( filename, blocktype, SWAP )) != 0 ) {
		fclose( opus );
	   fprintf( stderr, " Initialize error rc = %i \n", rc );
	   return(0);
	}

   fstartw     = SPPB.FXV;
   fstopw      = SPPB.LXV;
   fnum_points = SPPB.NPT;
   spacing = (fstopw-fstartw) / (fnum_points-1);
   if( spacing < 0.0 ) spacing = (fstartw-fstopw) / (fnum_points-1);

   fprintf( stderr, "  %12.4lf to %12.4lf %8d points spaced at %12.8lf\n", fstartw, fstopw, fnum_points, spacing );
	fprintf( stderr, "  Reading Spectra data block : %Xx\n", datablocktype );

   if( ! opus_searchdir( opus, datablocktype, &length, &pointer, SWAP )) {
      fprintf( stderr," *Cannot find data block : %s %Xx\n", blocktype, datablocktype );
      fclose( opus );
      return(1);
   } else fprintf( out,"  %s Data block Found.\n", blocktype );

	ydata = (float *) malloc ( length*sizeof(float) );

	if( ydata == (float *) 0 ) {
		fprintf( stderr, " *Could not allocate array for y data.\n" );
		fclose( opus );
		return(2);
	}

	if( opus_read_spectrum( opus, &fstartw, &fstopw, length, pointer, SPPB.CSF, SPPB.NPT, ydata, SWAP )) {
		fprintf( stderr," *Cannot read data.\n" );
		fclose( opus );
		return(3);
	}
	fclose( opus );

	/* linefit microwindow output: */
	for( i=0; i<nmws; i++ ) {
	  n1 = (int) (( mws[0][i] - fstartw ) / spacing );
	  n2 = (int) (( mws[1][i] - fstartw ) / spacing );
	  fprintf( out, "%3i:  %i %i %lf %lf\n", i, n1, n2, fstartw+n1*spacing, fstartw+n2*spacing );
	  if(( spc = fopen( mwfn[i], "w" )) == (FILE *) NULL) {
		  fprintf( stderr, "Cannot open %s output file.\n", mwfn[i] );
		  exit(1);
	  }
	  sum =0.0;
	  for( j=n1; j<=n2; j++ ) {
			sum = sum + ydata[j] ;
	  }
	  sum = sum / (n2-n1+1);
	  for( j=n1; j<=n2; j++ ) {
			fprintf( spc, "%14.6f,%14.6f\n", fstartw + j*spacing, ydata[j]/sum );
	  }
	  close( spc );

	}

	free(ydata);

   sscanf(OPPB.APT," %f ",&aperture); //,units); // in mm
   if( INPB.FOC != 0.0 ) field_of_view = aperture/INPB.FOC * 1000.;

	fprintf( out, "Resolution[cm-1]   : %10.4lf\n", AQPB.RES );
	fprintf( out, "Field of View [mr] : %10.4lf\n", field_of_view );
	fprintf( out, "Low Wavenumber     : %10.4lf\n", fstartw );
	fprintf( out, "High Wavenumber    : %10.4lf\n", fstopw );

   return(0);

}



int main( int argc, char *argv[] ) {

   static char version[] = {"Check OPUS v2.0"};
                 /* Version 1.1-A..A ( 7 Apr 98) by PSM */

   char  *optarg;
   // 4 char string EMIS |SNGC | IFGM | PHAS
   char  blocktype[] = "NONE\0";
   // 3 char string TAB | MLO | SGP | PKF | KPK | FL0
   char  sitetag[] = "UKN\0";
   char  filename[255];

   int   rc, c, type='\0';
   int   optind=0, SWAP=0, utc=0, sec=0;

   site = setsite( DEFx );

   while( --argc > 0 && (*++argv)[0] == '-' ) {
      while( c = *++argv[0] )
         switch( c ) {
            case 'w' :
               if(( site.w_lon = atof( ++argv[0] )) == 0 ) site.w_lon = 0.0;
               break;
            case 'n' :
               if(( site.n_lat = atof( ++argv[0] )) == 0 ) site.n_lat = 0.0;
               break;
            case 'e' :
               if(( site.altud = atof( ++argv[0] )) == 0 ) site.altud = 0.0;
               break;
            case 't' :
               if(( sec = atoi( ++argv[0] )) == 0 ) sec = 0;
               //printf( "len argv %d %s\n", strlen( argv[0] ), argv[0] );
               memset( argv[0], '\0', strlen( argv[0] ) );
               break;
            case 'u' :
               if(( utc = atoi( ++argv[0] )) == 0 ) utc = 0;
               break;
            case 's' :
               SWAP = 1;
               break;
            case 'S' :
               strncpy( sitetag, ++argv[0], 3 );
               memset( argv[0], '\0', 3 );
               //printf(" char site : '%s'\n", csite );
               break;
            case 'C' :
               type = 'C';
               break;
            case 'L' :
               type = 'L';
               break;
            case 'B' :
               type = 'B';
               break;
            case 'P' :
               type = 'P';
               break;
            case 'D' :
               type = 'D';
               break;
            case 'M' :
               type = 'M';
               break;
            case 'R' :
               type = 'R';
               strncpy( blocktype, ++argv[0], 4 );
               memset( argv[0], '\0', 4 );
               //printf(" block : '%s' %c\n", blocktype, type );
               break;
            case 'F' :
               type = 'F';
               strncpy( blocktype, ++argv[0], 4 );
               memset( argv[0], '\0', 4 );
               //printf(" block : '%s' %c\n", blocktype, type );
               break;
            case '?' :
            case 'h' :
               printf( "usage: ckopus -wXXX -nXXX -eXXX -tNNN -s -Ssss -[S|L|B|P|R] -[R|F]ssss file1 file2 ...\n" );
               printf( "where:\n" );
               printf( "   -w is west longitude defaults to %.2lf [decimal degrees]\n",site.w_lon );
               printf( "   -n is north latitude defaults to %.2lf [decimal degrees]\n", site.n_lat );
               printf( "   -e is elevation defaults to %.2lf [meters]\n", site.altud );
               printf( "   -t is time offset defaults to %i [seconds]\n", site.sec_offs );
               printf( "   -u is UTC time offset defaults to %i [hours]\n", site.utc_offs );
               printf( "   -s swap bytes on opus read defaults to No Swap\n" );
               printf( "   -S set lat lon & alt for [TAB | FL0 | MLO | KPK | PKF | MSA | SGP | TMK]\n" );
               printf( "   -C short listing of file contents\n" );
               printf( "   -L long listing of file contents\n" );
               printf( "   -B list file blocks\n" );
               printf( "   -P list time stamps\n" );
               printf( "   -M make Linefit microwindow files\n" );
               printf( "   -R write 'C' bnr for type     [TRAN | SGN2 | IFG2 | EMIS | IFGM | PHAS | SNGC] ex. -RSNGC \n" );
               printf( "   -F write FORTRAN bnr for type [TRAN | SGN2 | IFG2 | EMIS | IFGM | PHAS | SNGC] ex. -FEMIS \n" );
               printf( "   -D one line param list for database\n" );
               return(0);
          }
   }

   site = setmysite( sitetag );

   site.utc_offs = utc;
   site.sec_offs = sec;

   //printf(" block : '%s' %c\n", blocktype, type );

   if( argc <= 0 ) {
      fprintf( stderr, "You must also enter an OPUS data file name(s).\n" );
      fprintf( stderr, "STOP.\n" );
      return( 0 );
   } else {
      while( argc-- > 0 ) {

         strcpy( &filename[0], (argv++)[0] );
         if( VERBOSE ) fprintf( stderr, "main: filename: %s\n", filename );

         switch( type ) {
            case 'C' :
                if(( rc = initialize( filename, blocktype, SWAP )) == 0 ) {
                  fclose( opus );
               } else printf( " Initialize error rc = %i \n", rc );
            break;
            case 'L' :
               rc = opus_listParams( filename, SWAP );
            break;
            case 'B' :
               rc = opus_listBlocks( filename );
            break;
            case 'P' :
               rc = showtime( filename, SWAP );
            break;
            case 'M' :
               rc = opusmws( filename, SWAP );
            break;
            case 'D' :
               if(( rc = initialize( filename, blocktype, SWAP )) == 0 ) {
                  fclose( opus );
                  rc = showdb( filename );
               }  else printf( " Initialize error rc = %i \n", rc );
            break;
            case 'R' :
               if(( rc = initialize( filename, blocktype, SWAP )) == 0 ) {
                  rc = write_bnr( blocktype, SWAP, 'C' );
                  fclose( opus );
               } else printf( " Initialize error rc = %i \n", rc );
            break;
            case 'F' :
               if(( rc = initialize( filename, blocktype, SWAP )) == 0 ) {
                  rc = write_bnr( blocktype, SWAP, 'F' );
                  fclose( opus );
               } else printf( " Initialize error rc = %i \n", rc );
            break;
         }
      }
   }
   //printf("end %d\n", rc);
   return(rc);
}
