"""
convert_examples.py: Convert all examples input files to Linux.
Run this file in the main PROFAST folder.

author: Lena Feld
date: 5.5.2021
"""

import os
from glob import glob


def convert_to_linux(infile, outfile):
    # read line
    with open(infile, "r") as f:
        lines = f.readlines()

    # convert lines
    converted_lines = []
    for i, line in enumerate(lines):
        tmp_line = remove_carriage_return(line)
        tmp_line = replace_paths(tmp_line)
        converted_lines.append(tmp_line)

    with open(outfile, "w") as f:
        f.write("".join(converted_lines))


def remove_carriage_return(line):
    """Return lines with removed carriage return signs."""
    line = line.replace("\r\n", "\n")
    return line


def replace_paths(line):
    """Check all lines and replace pathes.

    Backslashes are replaced, quotes are added and
    capitalisation is checked.
    If a path does not exist, a RuntimeError is raised.
    """
    if "\\" not in line:
        return line
    print(line)
    line = line.replace("\\", "/")
    line = fix_case_sensitivity(line)
    print(line)
    if os.path.exists(line.strip()):
        print("exists ...")
        line = f"\"{line.strip()}\"\n"
        return line
    else:
        raise RuntimeError(f"Encountered unvalid path: {line}")


def fix_case_sensitivity(line):
    """Replace paths with wrong capitalization."""
    corrections = {
        "coccon": "COCCON",
        "dpt": "DPT",
        "hbr-trm40scans": "HBr-trm40scans"
    }

    exceptions = [
        "b-limb"
    ]

    for key, correction in corrections.items():
        # check if there is an exception
        excepiton = any(map(lambda v: v in line, exceptions))
        if not excepiton:
            line = line.replace(key, correction)

    return line


if __name__ == "__main__":
    # glob paths
    windows_inp_files = glob("examples-win/**/lft14.inp", recursive=True)

    linux_inp_files = [
        inp_file.replace("examples-win", "examples-linux")
        for inp_file in windows_inp_files
    ]

    print(windows_inp_files, linux_inp_files)
    # loop over all examples
    for infile, outfile in zip(windows_inp_files, linux_inp_files):
        print(infile)
        convert_to_linux(infile=infile, outfile=outfile)
