"""
compare_ergs.py: Compares results generated with Windows and Linux.
The file `win-linux-comparison.html` is generated
using color coding to highlight the differences.

author: Lena Feld
date: 5.5.2021
"""


import pandas as pd
import glob
import os


def get_comparison_html_from(erg_file, example):
    """Return comparison html sting,
     that compares the Windows and Linux versions of the erg files."""
    html_out = [f"<h3> {erg_file} </h3> <br>"]

    win_path = os.path.join("examples-win", example, "ergs", erg_file)
    linux_path = os.path.join("examples-linux", example, "ergs", erg_file)

    # write html sting if erg.dat file can be read to a dataframe
    if erg_file in ["specre01.dat", "modulat.dat", "ilsparms.dat"]:
        if erg_file in ["specre01.dat", "ilsparms.dat"]:
            df_win = pd.read_csv(
                win_path, delim_whitespace=True, header=None)
            df_lin = pd.read_csv(
                linux_path, delim_whitespace=True, header=None)
        else:
            df_win = pd.read_csv(win_path, delim_whitespace=True)
            df_lin = pd.read_csv(linux_path, delim_whitespace=True)

        ratio = df_win / df_lin
        diff = df_win - df_lin
        mean = (df_win + df_lin) / 2

        # loop over lines of dataframe
        for (i, r), (i, d), (i, m) in zip(
                ratio.iterrows(), diff.iterrows(), mean.iterrows()):
            html_row = ""

            for col in ratio.keys():
                r_a = abs(r[col]-1)
                # print(r_a)
                if r_a > 0.1:
                    html_row += '<font color="red">'
                elif r_a > 0.01:
                    html_row += '<font color="purple">'
                elif r_a > 0.001:
                    html_row += '<font color="blue">'
                elif r_a == 0:
                    html_row += '<font color="green">'
                else:
                    html_row += '<font color="black">'
                html_row += f"{m[col]:.3e};{d[col]:.3e} </font>&emsp;"
            html_row += "<br>"
            html_out.append(html_row)
        return html_out

    # write html string if erg.dat contains single line floats
    with open(win_path) as f:
        win_lines = f.readlines()
    with open(linux_path) as f:
        linux_lines = f.readlines()
    for win_line, linux_line in zip(win_lines, linux_lines):
        html_row = ""
        try:
            x_win = float(win_line)
            x_lin = float(linux_line)
            if x_lin != 0:
                r = x_win / x_lin
            else:
                r = 1
            d = x_win - x_lin
            m = (x_win + x_lin) / 2
            r_a = abs(r-1)
            if r_a > 0.1:
                html_row += '<font color="red">'
            elif r_a > 0.01:
                html_row += '<font color="purple"> '
            elif r_a > 0.001:
                html_row += '<font color="blue"> '
            elif r_a == 0:
                html_row += '<font color="green">'
            else:
                html_row += '<font color="black"> '
            html_row += f"{m:.3e};{d:.3e} </font>&emsp;"
        except ValueError:
            html_row = win_line.strip()

        html_row += "<br>"
        html_out.append(html_row)
    return html_out


html_out = [
    "<h1> Comparison between ergs/ produced with Windows and Linux </h1>",
    "Color coding: r = abs(1-ratio) <br>",
    "<font color='red'> red: r > 0.1</font>,"
    "<font color='purple'> purple: r > 0.01</font>,"
    "<font color='blue'> blue: r > 0.001</font>,"
    "<font color='green'> green r = 0, </font> black: otherwise <br>"
    "Numbers: mean; difference <br>"
]


input_files = glob.glob(r"examples-win/**/lft14.inp", recursive=True)
examples = []
for input_file in input_files:
    # remove lft14.inp ending
    examples.append(input_file[13:-10])

erg_files = [
    # "specre01.dat",
    "modulat.dat",
    "colparms.dat",
    "spsparms.dat",
    "ilsparms.dat",
]

example = examples[0]
for example in examples:
    print(example)
    html_out.append(f"<h2> {example} </h2> <br>")
    for erg_file in erg_files:
        if example.startswith("lineshapes"):
            if erg_file in ["modulat.dat", "ilsparms.dat"]:
                continue
        tmp_out = get_comparison_html_from(erg_file, example)
        html_out += tmp_out

    comparison = "\n".join(html_out)
    with open("win-linux-comparison.html", "w") as f:
        f.write(comparison)
