;+
; NAME:
;   Regrid (Function)
;
; PURPOSE:
;      This Routine is a wrapper for the MGS_RGrid object's regridding
;   capability. You can pass it any 2- or 3-dimensional data set and
;   convert the data from one rectangular geophysical grid to
;   another. For a 3-dimensional field, the dimensions can be arranged
;   as [lon, lat, xyz], [lon, xyz, lat], or [xyz, lon, lat].
;   If you supply the names of the old and new grid (and they are
;   stored in the grid catalogue of MGS_RGrid), the program will run
;   without user interaction (unless an error occurs). If you don't
;   supply grid names, you will be presented a selection list from
;   which you can choose the grids you want.
;      Regrid automatically takes into account all cases where grids
;   start at different longitudes or latitudes, or where latitudes
;   are reversed from one grid to the other; and it might even work
;   for regionally constrained (non-global) fields.
;
; AUTHOR:
;   Dr. Martin Schultz
;   Max-Planck-Institut fuer Meteorologie
;   Bundesstr. 55, D-20146 Hamburg
;   email: martin.schultz@dkrz.de
;
; CATEGORY:
;   CTM tools
;
; CALLING SEQUENCE:
;   result = Regrid(data)
;
; INPUTS:
;   DATA -> The 2- or 3-dimensional data array
;
;   OLDGRIDNAME -> The name of the "source" grid. The user will be
;       prompted if this argument is undefined.
;
;   NEWGRIDNAME -> The name of the "target" grid. The user will be
;       prompted if this argument is undefined.
;   
; KEYWORD ARGUMENTS:
;   AREA_WEIGHTED (input, boolean) -> Set this keyword to perform area
;       weighted averaging instead of fractional weighted averaging
;
;   OLDGRID (input|output, object ref.) -> A named variable will
;       contain the object reference to the oldgrid object. If you use
;       the use_grids keyword, this keyword must contain a valid
;       object reference which will be used as "source" grid.
;
;   NEWGRID (input|output, object ref.) -> A named variable will
;       contain the object reference to the newgrid object. If you use
;       the use_grids keyword, this keyword must contain a valid
;       object reference which will be used as "target" grid.
;
;  USE_GRIDS (input, boolean) -> Set this keyword to use predefined
;       grid objects (oldgrid and newgrid keywords) instead of new
;       grid definitions.
;
;   RESULT (output, string) -> The status of the operation. 'OK' if successful,
;       'FAILED' otherwise
;
; OUTPUTS:
;   A 2- or 3-dimensional data array with dimensions according to the
;        target grid definition.
;
; SUBROUTINES:
;   None.
;
; REQUIREMENTS:
;   The Regrid function relies on several objects from the MGS_ object
;   hierarchy:
;     - MGS_BaseObject
;     - MGS_BaseGUI
;     - MGS_Container
;     - MGS_List
;     - MGS_GridVector
;     - MGS_LonVector
;     - MGS_LatVector
;     - MGS_RGrid
;
; NOTES:
;    (1) The regridding algorithm may not be the fastest, but it is
;    strictly "mass" conserving (if you use the area_weighted
;    keyword). The regrid method of the rgrid object loops through
;    each individual grid box of the target grid and determines the
;    fractions of overlap for the source grid boxes. With the
;    area_weighted keyword, these fractions are multiplied by the
;    relative grid box area, then used as weighting factors for
;    averaging.
;    (2) In contrast to simple bilinear (or cubic) interpolation, the
;    output array always contains information from all input grid
;    boxes. Consider the case of a "smokestack" where you have one
;    extreme value in a number of fine resolution cells, and you
;    aggregate these data to a coarse resolution grid. With
;    interpolation, you may easily miss out the extreme value
;    completely, or you may exaggerate it's importance for the coarse
;    grid box. Using (area_weighted) averaging ensures that it will be
;    given the attention it deserves.
;
; EXAMPLES:
;    t42data = fltarr(128, 3, 64)+1.   ; dummy data T42 as [lon, lev, lat]
;    t21data = Regrid(t42data, 'T42', 'T21')
;
; MODIFICATION HISTORY:
;    mgs, 23 Aug 2001: - VERSION 1.0
;-
;
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  2000 Martin Schultz
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution. 
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################





FUNCTION regrid, data, ogrname, ngrname, $
                 area_weighted=area_weighted, $
                 oldgrid=oldgrid, newgrid=newgrid, use_grids=use_grids, $
                 result=result


   result = 'FAILED'
   retval = -1L

   IF N_Elements(data) EQ 0 THEN BEGIN
      Message, 'Data must be a 2d- or 3d array!', /Continue
      print, 'Usage: result = regrid(data, oldgrid, newgrid)'
      RETURN, retval
   ENDIF 
   ;; Get dimension information of data array
   dims = Size(data, /Dimensions)
   IF N_Elements(dims) LT 2 THEN BEGIN
      Message, 'Data must have at least 2 dimensions!', /Continue
      RETURN, retval
   ENDIF 
   IF N_Elements(dims) GT 3 THEN BEGIN
      Message, 'Data may not have more than 3 dimensions!', /Continue
      RETURN, retval
   ENDIF 

   ;; Create grid objects unless they are passed into the routine
   IF Keyword_Set(use_grids) THEN BEGIN
      IF NOT Obj_Valid(oldgrid) THEN BEGIN
         Message, 'Oldgrid must be a valid object!', /Continue
         RETURN, retval
      ENDIF 
      IF NOT Obj_Valid(newgrid) THEN BEGIN
         Message, 'Newgrid must be a valid object!', /Continue
         RETURN, retval
      ENDIF 
   ENDIF ELSE BEGIN

      oldgrid = Obj_New('MGS_RGrid', gridname=ogrname, $
                        select=(N_Elements(ogrname) EQ 0), $
                        seltitle='Old grid:')

      newgrid = Obj_New('MGS_RGrid', gridname=ngrname, $
                        select=(N_Elements(ngrname) EQ 0), $
                        seltitle='New grid:')

   ENDELSE 

   ;; Try to match data dimensions with grid dimensions
   oldgrid->GetProperty, nlon=nlon, nlat=nlat
   dimlon = -1L
   dimlat = -1L
   loopdim = -1L
   FOR i=0L, N_Elements(dims)-1 DO BEGIN
      IF dims[i] EQ nlon THEN BEGIN
         dimlon = i 
      ENDIF ELSE IF dims[i] EQ nlat THEN BEGIN
         dimlat = i
      ENDIF ELSE BEGIN
         loopdim = [ loopdim, i ]
      ENDELSE 
   ENDFOR 
   IF N_Elements(loopdim) GT 1 THEN loopdim = loopdim[1:*]

   IF dimlon LT 0 OR dimlat LT 0 THEN BEGIN
      Message, 'Data contains no dimension matching with grid longitude or latitude!', $
         /Continue
      print,'Grid dimensions : ',nlon,nlat
      print,'Data dimensions : ',dims
      RETURN, retval
   ENDIF 
   
   IF dimlon GT dimlat THEN BEGIN
      Message, 'Data with transposed dimensions (lat,lon) currently not supported!'
   ENDIF 

   ;; Do the regridding
   ;; If data is 2D, it's straightforward. If data is 3D, loop
   IF N_Elements(dims) EQ 2 THEN BEGIN
      retval = newgrid->Regrid(data, oldgrid, area_weighted=area_weighted)
   ENDIF ELSE IF N_Elements(dims) EQ 3 THEN BEGIN
      ;; Set up 3D result array
      nvals = (dims[loopdim])[0]
      newgrid->GetProperty, nlon=nnlon, nlat=nnlat
      retval = Make_Array(nnlon, nnlat, nvals, type=Size(data, /type))
      FOR i=0L, nvals-1 DO BEGIN
         CASE loopdim[0] OF
            0 : retval[*,*,i] = newgrid->Regrid(Reform(data[i,*,*]), oldgrid, $
                                               area_weighted=area_weighted)
            1 : retval[*,*,i] = newgrid->Regrid(Reform(data[*,i,*]), oldgrid, $
                                               area_weighted=area_weighted)
            2 : retval[*,*,i] = newgrid->Regrid(Reform(data[*,*,i]), oldgrid, $
                                               area_weighted=area_weighted)
         ENDCASE 
      ENDFOR 
      ;; Transpose result array if necessary
      CASE loopdim[0] OF
         0 : retval = Transpose(retval, [2,0,1])
         1 : retval = Transpose(retval, [0,2,1])
         2 : BEGIN 
             END                   ;  no transposition necessary
      ENDCASE 
   ENDIF ELSE BEGIN
      Message, 'Data may not have more than 3 dimensions!', /Continue
      RETURN, retval
   ENDELSE 
   

   ;; Destroy the objects
   IF Arg_Present(oldgrid) EQ 0 THEN Obj_Destroy, oldgrid
   IF Arg_Present(newgrid) EQ 0 THEN Obj_Destroy, newgrid

   
   result = 'OK'
   RETURN, retval

END
