;*********************************************************************
; Class Progress_Window
;
;  This is a widget which cantians a progress bar and a cancel
;  button.  This can be used to show the progress of a long process
;  and can be used to cancel the process.  See the test routines below
;  for an example of how this class may be used.
;
;*********************************************************************


;=====================================================================
; pro Progress_Window::finish
;
;  Invoke this routine when the monitored process is finished.  This
;  will complete the progress bar and close the window.
;
;=====================================================================
Pro Progress_Window::finish

  ; Make the draw widget the current window.

  WSet, self.wid

  ; Calculate the extent of the box to be drawn (full box).

  xbox_coords = [0, 0, Fix(self.drawsize), Fix(self.drawsize), 0]

  ; Draw the box

  Polyfill, xbox_coords, self.ybox_coords, Color=1, /Device

  ; close this window
  Widget_Control, self.top, /Destroy
  ;; Destroy this object
  ;
  ;obj_destroy, self

End ;  Progress_Window::finish

;====================================================================
; pro Progress_Window::update
;
;  Invoke this routine to update the progress bar.
;
;  Inputs:
;
;    percentDone: A floating point value which represents the
;                 percentage of the progress of the process.
;                 Ranges from 0. to 1.
;
;====================================================================
Pro Progress_Window::update, percentDone

  ; Update the progress box in the display

  ; Make the draw widget the current window. and bring forward
  widget_control, self.cancel_wid, /show
  WSet, self.wid

  ; Calculate the extent of the box to be drawn in the draw widget

  xextent = Fix(self.drawsize*percentDone)
  xbox_coords = [0, 0, xextent, xextent, 0]

  ; Draw the box

  Polyfill, xbox_coords, self.ybox_coords, Color=1, /Device

End ; Progress_Window::update

;=====================================================================
; Function Progress_Window::canceled
;
;  Returns 1 (true) if the cancel button has been pressed.  Otherwise,
;  returns 0 (false)
;=====================================================================
Function Progress_Window::canceled

  ; Did the "Cancel Operation" button get clicked?
  ; Look for an event. Don't turn off the hourglass cursor!

  progressEvent = Widget_Event(self.cancel_wid, /NoWait)

  ; Is this a button event?

  eventName = Tag_Names(progressEvent, /Structure_Name)
  IF strcmp( eventName, 'WIDGET_BUTTON', 10, /fold_case ) THEN BEGIN

    ; If it IS a button event, destroy the widget program and
    ; issue an informational message to the user to alert him or her.


    ;result = Message,'Operation Canceled!!', /Information

    ; Escape from the loop. Interrupt code would go here.

    ;obj_destroy, self
    ; close this window
    Widget_Control, self.top, /Destroy
  return, 1 ; true

  ENDIF

  return, 0 ; false

End ;  Progress_Window::canceled

;=====================================================================
; FUNCTION Progress_Window::init
;
; This is invoked when a Progress_Window object is instansiated.
;
; Inputs:
;   group_leader: The widget group leader
;
;   title: String containing the title of the progress window
;
;   message: String containing the message displayed over the progress
;            bar
;
;   drawsize: The x size of the progress bar
;
;   buttonTitle: String containing the text that goes on the 'cancel'
;                button.
;   xoffset: the x offset of the window
;   yoffset: the y offset of the window
;=====================================================================
FUNCTION Progress_Window::init, group_leader=group_leader, title=title, $
                     message=message, drawsize=drawsize, buttonTitle=buttonTitle, $
                     xoffset=xoffset, yoffset=yoffset

  ; Check for keywords

  IF N_Elements(message) EQ 0 THEN BEGIN
    message = ''
    testMessage = 0
  ENDIF ELSE testMessage=1

  IF N_Elements(title) EQ 0 THEN title = ''
  IF N_Elements(drawsize) EQ 0 THEN drawsize=100
  IF N_Elements(buttonTitle) EQ 0 THEN buttonTitle = 'Cancel Operation...'

  self.drawsize = drawsize

  ; Find out how big the display is, so you can position this widget
  ; near the center of the display if offsets are not passed in.

  Device, Get_Screen_Size=sizes

  IF N_Elements(xoffset) EQ 0 THEN $
    xlocation = (sizes(0) / 2.0) - 75 ELSE xlocation = xoffset
  IF N_Elements(yoffset) EQ 0 THEN $
    ylocation = (sizes(1) / 2.0) + 25 ELSE ylocation = yoffset



  ; Define the top-level base. Make it unsizeable.
  ; Give it a title. Make its children centered in it.

  tlb = Widget_Base( group_leader=group_leader, Column=1, Title=title, $
                                TLB_Frame_Attr=1, Base_Align_Center=1, $
                                XOffSet=xlocation, YOffSet=ylocation )

  ; If there is a message, make a label to hold it.

  IF testMessage NE 0 THEN label = Widget_Label(tlb, Value=message)

  ; Make a draw widget to hold the "progress" bar

  draw = Widget_Draw(tlb, XSize=drawsize, YSize=10)

  ; Make a "Cancel Operation" button

  self.cancel_wid = Widget_Button(tlb, Value=buttonTitle)

  ; Realize the widget program.

  Widget_Control, tlb, /Realize

  ; Get the window index number of the draw widget

  Widget_Control, draw, Get_Value=wid
  WSet, wid

  self.top = tlb
  self.wid = wid

  ; Don't call XManager because this program is going to be manaaged
  ; by the calling program using Widget_Event to see if the "Cancel"
  ; button has been clicked.


  ; Take over color index 1 to draw a red box in the draw widget

  TVLct, r, g, b, /Get
  self.oldDrawColor = [r(1), g(1), b(1)]
;  TVLct, 255, 0, 0, 1

  ; Set the y coordinates of the red box

  self.ybox_coords = [0, 10, 10, 0, 0]

  return,1

End ;  Progress_Window::init

;======================================================================
; PRO Progress_Window::cleanup
;   Invoked when the object is destroyed
;======================================================================
PRO Progress_Window::cleanup

  ; close this window
  ;Widget_Control, self.top, /Destroy

  ; restore the old colors in index 1
  TvLct, self.oldDrawColor(0), self.oldDrawColor(1), self.oldDrawColor(2), 1

End ;  Progress_Window::cleanup

PRO Progress_Window__define

  struct={ Progress_Window,               $ ; class name
           cancel_wid:0L,    $
           wid:0L,      $
           top:0L,      $
           drawsize:0l,      $
           oldDrawColor:lonarr(3), $
           ybox_coords:lonarr(5) $


  }

END ;  Progress_Window__define


;***********************************************************************
; Test Routines
;***********************************************************************

PRO CenterTLB, tlb

Device, Get_Screen_Size=screenSize
xCenter = screenSize(0) / 2
yCenter = screenSize(1) / 2

geom = Widget_Info(tlb, /Geometry)
xHalfSize = geom.Scr_XSize / 2
yHalfSize = geom.Scr_YSize / 2

Widget_Control, tlb, XOffset = xCenter-xHalfSize, $
   YOffset = yCenter-yHalfSize

END

Pro Test_Progress_Window_Event, event

Widget_Control, event.id, Get_UValue=buttonUValue

CASE buttonUValue OF

   'QUIT': Widget_control, event.top, /Destroy
   'CALC': BEGIN


      ; Create the ProgressWindow and get it on the screen
      ; with the current offsets.

      res = widget_info( event.top, /geometry)

   ;   Widget_Control, event.top, TLB_Get_Offset=offsets

   ;print, 'offsets = ', offsets

      progress = obj_new ('Progress_Window', group_leader=event.top,  $
                                         Message='Performing Large Calculation...', $
   ;                                      XOffset=offsets(0)+50, $
   ;                                      YOffset=offsets(1)+50, DrawSize=200, $
                                          xoffset=res.xoffset+50, $
                                          yoffset=res.yoffset+50, $
                                          DrawSize=200, $
                                          ButtonTitle='Cancel Local Operation')

         ; Set counters. These will indicate how often you want to update
         ; the "progress" display and check for a "cancel operation"
         ; event.

      loops = 10000000L
      counter = 10000L
      counterIncrement = 10000L
      percentIncrement = float(counterIncrement)/float(loops)
      percentDone = 0.0



         ; Go into loop

      For j=0L, loops DO BEGIN

            ; This is the place where you do your calculation,
            ; perform your operation, or whatever it is you do.

            ; Calculation .......

            ; This is where you update the "progress" display and
            ; check to see if the "Cancel Operation" button was clicked.

         IF j GT counter THEN BEGIN

               ; Reset counter. Can't be bigger than loop index.

            counter = loops < (counter + counterIncrement)
            percentDone = percentDone + percentIncrement


               ; Slow the loop at bit.

             Wait, .02

               ; Update the progress

             progress->update,percentDone


             ; Escape from the loop. Interrupt code would go here.

             if ( progress->canceled() ) then GoTo, outSideLoop

          ENDIF

            ; If the calculation progessed to completion, execute this code.

          IF j EQ loops THEN BEGIN

             progress->finish


          ENDIF

       ENDFOR

       outSideLoop: ; Come here when calculation or operation is canceled.
       if (obj_valid(progress)) then obj_destroy, progress


    END

ENDCASE

END ;*******************************************************************

Pro Test_Progress_Window

  Device, Decomposed=0
  tlb = Widget_Base(Column=1, TLB_Frame_Attr=1)
  button = Widget_Button(tlb, Value='Big Calculation', UValue='CALC')
  quitter = Widget_Button(tlb, Value='Quit', UValue='QUIT')
  CenterTLB, tlb
  Widget_Control, tlb, /Realize

  XManager, 'Test_Progress_Window', tlb

END ;*******************************************************************
