;; ncdf_timelabel
;; This function returns a formatted date string or a string array
;; of time values, where the unit is "days since YYYY-MM-DD
;; [HH:MM:SS]"
;; --
;; mgs, 08 Jul 2002
;; --


PRO tlbl_extract_date, refdate, year, month, day, hour, minute

   ;; expected format : 'YYYY-MM-DD HH:MM:SS'
   ;; everything optional except for year
   ;; set defaults
   year = -1L
   month  = 1L
   day    = 1L
   hour   = 0L
   minute = 0L

   ;; check validity of reference date and time
   p0 = StrPos(refdate,'-')
   IF p0 NE 4 THEN BEGIN
      Message, 'Invalid date format in units!', /CONTINUE
      RETURN
   ENDIF
   p1 = StrPos(refdate,'-',p0+1)
   IF p1 NE 7 THEN BEGIN
      Message, 'Invalid date format in units!', /CONTINUE
      RETURN
   ENDIF

   ;; extract date value
   year   = Long(StrMid(refdate,0,4))
   IF StrLen(refdate) GT  5 THEN month   = Long(StrMid(refdate,5,2))
   IF StrLen(refdate) GT  8 THEN day     = Long(StrMid(refdate,8,2))
   IF StrLen(refdate) GT 11 THEN hour    = Long(StrMid(refdate,11,2))
   IF StrLen(refdate) GT 14 THEN minute  = Long(StrMid(refdate,14,2))

END

;; ------------------------------------------------------------------

FUNCTION tlbl_format_date, julian, format=format, months=months, $
                           days_OF_week=days_OF_week

   ;; initialise label_date format
   IF n_elements(format) EQ 0 THEN format='%D %M %Y  %H:%I'
   if( !version.release ge '5.4' ) then begin
     dummy = label_date(date_format=format, months=months, $
                     days_OF_week=days_OF_week)
   endif else begin
     dummy = label_date(date_format=format, months=months)
   endelse

   ;; format output string
   result = StrArr(n_elements(julian))
   FOR i=0L, n_elements(julian)-1 DO BEGIN
      result[i] = label_date(0,0,julian[i])
   ENDFOR

   RETURN, result

END

;; ------------------------------------------------------------------

PRO tlbl_help, group_leader=group_leader

   text = [ 'Usage: res=ncdf_timelabel(value, units, options)', $
            'Options:', $
            '   format = format string (default: "%D %M %Y  %H:%I")', $
            '            %Y : 4-digit year', $
            '            %Z : 2-digit year', $
            '            %N : 2-digit month', $
            '            %M : month name', $
            '            %D : 2-digit day', $
            '            %W : day of week', $
            '            %H : hour', $
            '            %I : minute', $
            '   months = a string array with the 12 month names', $
            '   days_of_week = a string array with 7 weekdaynames (0=Sunday)', $
            '   /help = display this help text', $
            '   /debug = display debugging info'  ]

   XDisplayfile, '', done_button='Dismiss', group=group_leader, $
                 text=text, title='ncdf__timelabel', /MODAL

END

;; --------------------------------------------------
; computes the julian day for the time(s) -
;    time is an array containing the number of dec days since the ref date/time
;
function julian_365date, time, ref_month, ref_day, ref_year, $
                               ref_hour,  ref_min, ref_sec

  ; handy data
  mon_DOYs = [0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 366 ]

  ; compute the reference day of year
  ref_day_of_yr = mon_DOYs[ref_month-1] + ref_day

  ; compute date data
  year = ref_year + long(time/365)
  rem_time = time - ((year - ref_year)*365.)
  day_of_yr = ref_day_of_yr + long( rem_time )
  rem_time = rem_time - (day_of_yr - ref_day_of_yr)
  hour = ref_hour + long( rem_time*24 )
  rem_time = rem_time - (hour - ref_hour)/24.
  min = ref_min + long( rem_time*24*60 )
  rem_time = rem_time - (min - ref_min)/(24.*60.)
  sec = ref_sec + long( rem_time*24*60*60 )

  ; don't think we need to adjust for overflows - julday seems to work ok with overflows
  ; may need to adjust the day_of_year - so we can pick out the correct month
  for i=0, n_elements(time)-1 do begin
;    while (sec[i] ge 60) do begin
;      sec[i] = sec[i] - 60
;      min[i] = min[i] + 1
;    endwhile
;    while (min[i] ge 60) do begin
;      min[i] = min[i] - 60
;      hour[i] = hour[i] + 1
;    endwhile
;    while (hour[i] ge 24) do begin
;      hour[i] = hour[i] - 24
;      day_of_year[i] = day_of_year[i] + 1
;    endwhile
    while (day_of_yr[i] ge 365) do begin ; caution: only good for non leap year
      day_of_yr[i] = day_of_yr[i] - 365
      year[i] = year[i] + 1
    endwhile
    while (day_of_yr[i] lt 0 ) do begin
      day_of_yr[i] = day_of_yr[i] + 365
      year[i] = year[i] - 1
    endwhile
  endfor

  ; determine month(s) and day(s) of month
  month = lonarr( n_elements(day_of_yr) )
  day = lonarr( n_elements(day_of_yr) )

  for i=0, n_elements(day_of_yr)-1 do begin

    imon = 0
    d = 0

    while ( d le day_of_yr[i] ) do begin
      d = mon_DOYs[imon]
      imon = imon+1
    endwhile

    month[i] = imon-1
    day[i] = day_of_yr[i] - mon_DOYs[imon-2]

  endfor

  ; make sure year is not zero - year zero is not valid, julday will give error
  ix = where( year eq 0 )
  while ( ix[0] ge 0 ) do begin
    message, 'Year is zero -> add one to year',/continue
    year = year+1
    ix = where( year eq 0 )
  endwhile

  ; get the julian days for the times...
  ; here julian is an array since all the date data are arrays
  if( !version.release ge '5.4' ) then begin
    julian = julday(month, day, year, hour, min, sec)
  endif else begin
    julian = dblarr(n_elements(month))
    for i =0, n_elements(julian)-1 do $
      julian[i] = julday(month[i], day[i], year[i], hour[i], min[i], sec[i]) + .5 
  endelse

  return, julian

End

;; ------------------------------------------------------------------

FUNCTION ncdf_timelabel, value, units, format=format, $
                         months=months, days_OF_week=days_OF_week, $
                         refyear=refyear, ignore_gregor=ignore_gregor, $
                         debug=debug, calendar=calendar,$
                         help=help

   ;; display help text if desired
   IF keyword_set(help) OR  n_elements(value) EQ 0 THEN BEGIN
      tlbl_help
      RETURN, ''
   ENDIF

   ;; return values as string(s) if units not present
   IF n_elements(units) EQ 0 THEN RETURN, String(value)


   ;; convert value to double
   theValue = double(value)

   ;; analyse units
   theUnits = StrLowCase(StrCompress(StrTrim(units,2)))
   p0 = StrPos(theUnits, 'days since')
   pstart = 11
   IF p0 ne 0 THEN BEGIN
      p0 = StrPos(theUnits, 'day since')
      pstart = 10
   ENDIF
   IF p0 NE 0 THEN BEGIN
      Message, 'Units must be "days since YYYY-MM-DD [HH:MM:SS]"', $
               /CONTINUE
      RETURN, ''
   ENDIF

   refdate = StrMid(theUnits,pstart,19)
   ;; extract date value from units string
   tlbl_extract_date, refdate, year, month, day, hour, minute
   refyear=year
   IF keyword_set(debug) THEN print,'year=',year,'  month=',month, $
                      '  day=',day,'  hour=',hour,'  minute=',minute

   gregorian = 1 ; default is gregorian calendar

   ; make adjustment if cal is 365_day
   if (n_elements(calendar) gt 0) then begin
     if ( strcmp( calendar, '365_day', 5, /fold_case) or strcmp( calendar, 'noleap', 5, /fold_case)  ) then begin
       julian = julian_365date( value, month, day, year, hour, minute, 0)
       refyear = 1600 ; set so that this routine is not invoked again.
       gregorian = 0 ; this is not gregorian calendar
     endif
   endif 

   if gregorian then begin ; else assume gregorian calendar

     IF (year LT 1583 AND (not keyword_set(ignore_gregor))) THEN BEGIN
       Message, 'Gregorian calendar only valid after 1582', /CONTINUE
       RETURN, ''
     ENDIF

     IF year EQ 0 THEN year = 1  ;; dirty quirk fix

     ;; compute julian date for refdate
     theRefDate = JulDay(month, day, year, hour, minute, 0)

     ;; compute julian date for value(s)
     julian = theRefDate + value

   endif

   ;; format output string
   result = tlbl_format_date(julian, format=format, months=months, $
                            days_OF_week=days_OF_week)

   RETURN, result

END
