;-------------------------------------------------------------
;+
; NAME:
;    mgsWPage
;
; PURPOSE:
;    Object class for pages (i.e. screen windows or paper)
;    Inherited from mgsWBox.
;
; CATEGORY:
;    Graphics, Objects
;
; PROPERTIES:
;    public properties:
;       X0, X1 : lower left corner of box (inherited)
;       XWidth, YWidth : lenght and height of box (inherited)
;       Unit : measuring unit of the above [pixel,cm,in[ch]] (inherited)
;       Title : a title string (inherited)
;       Zoom : an enlargement factor used to determine screen window size
;    new private properties:
;       Window : window number for screen display
;
; METHODS:
;    public methods:
;       GetProperty
;       SetProperty
;       Show
;       Init
;       Cleanup
;    private methods:
;       ComputePrivateCoordinates
;       ChangeUnit
;       CheckUnit
;    
; DESCRIPTION:
;    WPage stores page size information and a page (screen window) title.
;    Page size can be defined in pixel, cm, or in(ch). For simplification,
;    the user can use the A4, LETTER, PORTRAIT, or LANDSCAPE keywords.
;
; INTERFACES:
;
; SIDE EFFECTS:
;
; MODIFICATION HISTORY:
;    12 Nov 1999 :  written by Martin G. Schultz
;-
; Copyright
;-------------------------------------------------------------


; **********************************************************************
; **********************************************************************
; Get Properties of mgsWPage
; Additional keyword is Zoom
;
pro mgsWPage::GetProperty, Zoom=Zoom,            $
                           _ref_extra=extra


    self->mgsWBox::GetProperty,_extra=extra

    if arg_present(Zoom) then Zoom = self.Zoom

end  ;  mgsWPage::GetProperty


; **********************************************************************
; **********************************************************************
; Set Properties of mgsWPage
; Additional keywords from parent object are:
;     XWidth, YWidth, Unit, Title
; Notes:
; LANDSCAPE is only honored together with A4 or LETTER
; PORTRAIT is a dummy keyword merely for code readability
; LETTER overwrites A4 if both are present
; explicit size information overrides A4 and LETTER keywords
;
pro mgsWPage::SetProperty, A4=A4,                $
                           LETTER=Letter,        $
                           LANDSCAPE=Landscape,  $
                           ZOOM=Zoom,            $
                           _ref_extra=extra


print,'## In page::setproperty'
   ; check additional inputs
   ; zoom factor
   if n_elements(Zoom) eq 1 then $
       self.Zoom = ( float(Zoom) > 0.001 ) < 100.

   if keyword_set(A4) then begin
      thisXWidth = 21.0
      thisYWidth = 29.7
      thisUnit = 'cm'
      if keyword_set(Landscape) then begin
         thisXWidth = 29.7
         thisYWidth = 21.0
      endif
      self->mgsWBox::SetProperty,XWidth=thisXWidth,YWidth=thisYWidth,Unit=thisUnit
   endif

   if keyword_set(Letter) then begin
      thisXWidth = 8.5
      thisYWidth = 11.0
      thisUnit = 'in'
      if keyword_set(Landscape) then begin
         thisXWidth = 11.0
         thisYWidth = 8.5
      endif
      self->mgsWBox::SetProperty,XWidth=thisXWidth,YWidth=thisYWidth,Unit=thisUnit
   endif

   ; Call superclass routine
   self->mgsWBox::SetProperty,_extra=extra

   ; Redisplay if window on screen and size has changed
   self->Show,/ForceNew


end  ;  mgsWPage::SetProperty


; **********************************************************************
; **********************************************************************
; Display mgsWPage
;
; Opens a window on a window (screen) device or brings it
; to the front if already created
; NB: This would better be done with widgets so that the program
; can keep track of user actions (closing, resizing, etc.)
; Currently, the windows may get confused if the user closes the object 
; window, opens another window with /FREE, and then forces a redraw
; of the object window.
;

function mgsWPage_GetNextWindow

   ; utility function for guessing of next available window number
   ; in order to construct window title
   ; Somewhat tricky since the number of elements in device,window_state
   ; changes during the IDL session

   nextw = -1
   device,window_state=w
   if n_elements(w) lt 33 then $
      nextw = 32  $
   else begin
      ind = where(w[32:*] eq 0)
      nextw = min(ind)+32 
      if nextw lt 0 then nextw = n_elements(w)  
   endelse

   return,nextw
end

pro mgsWPage::Show,  ForceNew=ForceNew

   ; Does device support windows?
   if (!D.Flags AND 256) gt 0 then begin

      ; test if window already there
      theError = 0
      catch, theError
      if theError ne 0 then begin
         catch,/Cancel
         self.Window = -1
      endif

      ; close old window if it had been resized
      if keyword_set(ForceNew) AND self.Window ge 0 then begin
         WDelete,self.Window
         self.Window = -1
      endif
      
      ; open new window if necessary
      if self.Window ge 0 then begin
         WSet,self.Window
         WShow,self.Window
      endif else begin
         ; construct title if necessary
         ; need to guess number of next window (tricky)
         nextw = mgsWPage_GetNextWindow()
         thisTitle = Replace_Token(self.Title,'WINDOW',nextw)
         Window,/FREE,title=thisTitle,             $
                XSize=self.DXWidth*self.Zoom,      $
                YSize=self.DYWidth*self.Zoom
         self.Window = !D.Window
      endelse

   endif  ; device supports windows

end  ;  mgsWPage::Show


; **********************************************************************
; **********************************************************************
; Cleanup method
;
pro mgsWPage::CleanUp

   print,'mgsWPage::CleanUp'

   ; Close window if on screen
   if (!D.Flags AND 256) gt 0 then begin
       theError = 0
       catch,theError
       if theError ne 0 then begin
           catch,/Cancel
           self.Window = -1
       endif
      
       if self.Window ge 0 then $ 
          WDelete,self.Window
   endif

end  ;  mgsWPage::CleanUp


; **********************************************************************
; **********************************************************************
; Initialization method
; Additional keywords from parent object are:
;     XWidth, YWidth, Unit, Title
; Notes:
; LANDSCAPE is only honored together with A4 or LETTER
; PORTRAIT is a dummy keyword merely for code readability
; LETTER overwrites A4 if both are present
; A4 and LETTER set the default zoom factor to 0.70
; TITLE is set to a default value (class name + window number)
;
function mgsWPage::Init,   $
                    A4=A4,              $
                    LETTER=Letter,      $
                    PORTRAIT=Portrait,  $
                    LANDSCAPE=Landscape,$
                    ZOOM=Zoom,          $
                    TITLE=Title,        $
                    _ref_extra=extra


   self.Window = -1

   ; set default title
   if n_elements(Title) eq 0 then $
       Title = Obj_Class(self) + ' %WINDOW%'

   if not (self->mgsWBox::init(Title=Title,_extra=extra)) then begin
      ok = Error_Message('Failed to initialize mgsPage!')
      return,0
   endif

   ; Make sure LETTER and A4 are used exclusively
   if keyword_set(LETTER) then A4 = 0

   ; Determine appropriate zoom factor
   ; Make it easy: use knowledge of approximate screen resolution
   ; and paper size
   if n_elements(Zoom) eq 0 then begin
       ; test for screen device
       if (!D.Name eq 'X' or !D.Name eq 'WIN' or !D.Name eq 'MAC') then begin
          Device,Get_Screen_Size=ScrSize  
          if keyword_set(LANDSCAPE) then  $
              ScrSize = ScrSize[0]  $
          else  $
              ScrSize = ScrSize[1]
          Zoom = 0.67 * ScrSize/1188.        ; 0.67 as default screen fraction
                                             ; 1188. = 29.7 cm * 40 pixel/cm
       endif else  begin
          Zoom = 1.0
       endelse
   endif else $
       Zoom = float(Zoom)

help,zoom
   ; pass additional settings to SetProperty
   self->SetProperty,A4=A4,LETTER=Letter,LANDSCAPE=Landscape,  $
            Zoom=Zoom,Error=Error
            

   return,(Error eq 0)


end  ;  mgsWPage::Init


; **********************************************************************
; **********************************************************************
; Structure definition
;
pro mgsWPage__Define

   temp = {mgsWPage,             $
                    Zoom:1.0,    $
                    Window:-1,   $
                    inherits mgsWBox }

end  ;  mgsWPage__Define



