;+
; NAME:
;    mgs_varinfoobject
;
; PURPOSE:
;    This object provides a GUI interface for setting variable
;  properties. It is either cross linked with an existing variable
;  object or it can be used to create a (empty) variable object. The
;  user can change the variable name, long name, units, missing value,
;  min_valid and max_valid attributes, and the variable type. The
;  names and sizes of the variable dimensions will be displayed.
;  The object inherits from the generic MGS_GUIObject.
;
; CATEGORY:
;  Widget Objects
;
; CALLING SEQUENCE:
;
; ARGUMENTS:
;
; KEYWORDS:
;
;    inherited keywords:
;    widget_title (string, input) -> The title of the top level
;          base. This string is displayed as window title AND at the
;          top of the dialog window.
;
;    widget_defaultfont (string, input) -> A font name to be used for
;          the text fields in the widget. Note: Font names are
;          platform dependent!
;
;    widget_labelfont (string, input) -> A font name to be used for
;          the title labels of widget sections. Note: Font names are
;          platform dependent!
;
;    no_dialog (boolean, input) -> Display error messages as text
;          rather than in a dialog box. NOTE: This is not fully
;          implemented. This program uses a mixture between MGS and DF
;          error handling code.
;
; REQUIREMENTS:
;    Inherits from mgs_guiobject and mgs_baseobject 
;
; DESCRIPTION:
;
; MODIFICATION HISTORY:
;    mgs, 16 Mar 2001: Version 0.1
;-
;
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  2001 Martin Schultz
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution.
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################

; =============================================================================
; Object methods
; =============================================================================

; --- Set... methods: These methods are called from the generic Event
;     handler. See the BuildGUI method to learn how they are activated
;     by storing them as method tag in the widget's UValue field. Note
;     that a Set... method is not required for object widgets where
;     the value can always be queried from the object directly.
; ---

; -----------------------------------------------------------------------------
; SetType: (private)
;    This method sets the new variable type upon selection

FUNCTION MGS_Varinfoobject::SetType, event

   Catch, theError
   IF theError NE 0 THEN BEGIN
      MGS_GUIObject_Error_Message, /Traceback
      RETURN, 0
   ENDIF

   type = *event.selection
   print,'New type = ',type

   IF Obj_Valid(self.varobject) THEN BEGIN
      IF self.varobject->ContainsData() THEN $
         self.varobject->ConvertType, tname=type
   ENDIF

   RETURN, self->CheckEvent(event)
END 


; -----------------------------------------------------------------------------
; GetState:
;    This method constructs a structure containing fields with values
; set to the current state of the object.
;    This method overwrites the generic GetState method (and does not
; even call it).
;    The ok keyword can be used to return information about the validity
; of the current object state.
;    As background is not an allowed keyword for map_set, it must be
; retrieved independently and used with an Erase command.

FUNCTION MGS_Varinfoobject::GetState, ok=ok, $
                      varname=varname, $
                      long_name=long_name, $
                      units=units,  $
                      missing_value=missing_value,  $
                      min_valid=min_valid,  $
                      max_valid=max_valid,  $
                      _Ref_Extra=e

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      MGS_GUIObject_Error_Message, /Traceback
      RETURN, 0
   ENDIF

   ;; Make sure the structure reflects the current object state
   self->UpdateObject

   varname = StrTrim( self.varnameObj->Get_Value(), 2 )
   long_name = StrTrim( self.long_nameObj->Get_Value(), 2 )
   units = StrTrim( self.unitsObj->Get_Value(), 2 )

   missing_value = self.missvalObj->Get_Value()
;;   IF Finite(missval) EQ 0 THEN missval = 0.

   min_valid = self.minvalObj->Get_Value()
   max_valid = self.maxvalObj->Get_Value()
    
   struct = Create_Struct( $
                           'name', varname,  $
                           'long_name', long_name,  $
                           'missing_value', missing_value,  $
                           'min_valid', min_valid,  $
                           'max_valid', max_valid )

   RETURN, struct
END


; -----------------------------------------------------------------------------
; UpdateObject:  (private)
;    This method simply gathers all the information in the graphical
; user interface (that doesn't update itself) and updates the self
; object. If fields are undefined, they are set to their default values.
;   This method demonstrates how to access information from compound
; object widgets.

PRO MGS_Varinfoobject::UpdateObject

   ;; Call inherited method (does nothing here, but for the sake of
   ;; demonstration)
   self->MGS_GUIObject::UpdateObject

   ;; Make sure that input fields contain valid values, or reset to defaults
   ;; (strings are always valid ???? Maybe test for non-blank?)
;   IF Obj_Valid(self.varnameObj) NE 0 THEN BEGIN
;      IF Finite(self.varnameObj->Get_Value()) EQ 0 THEN BEGIN
;         MGS_GUIObject_Error_Message, 'Invalid entry for variable name!'
;         value = ''
;         self.varnameObj->Set_Value, value
;      ENDIF ELSE value = self.varnameObj->Get_Value()
;   ENDIF

   ;; Missing value (can also be not finite ;-)
;   IF Obj_Valid(self.missvalObj) NE 0 THEN BEGIN
;      IF Finite(self.missvalObj->Get_Value()) EQ 0 THEN BEGIN
;         MGS_GUIObject_Error_Message, 'Invalid entry for missing value!'
;         value = !Values.F_NaN
;         self.missvalObj->Set_Value, value
;      ENDIF ELSE value = self.missvalObj->Get_Value()
;   ENDIF

;; test that min_valid is smaller than max_valid and vice versa
;; (try to find out which was altered lat and modify that one)
;; ....


END


; -----------------------------------------------------------------------------
; AcceptDialog:  (private)
;    This method reacts on the user's click on the Accept button
; (blocking widgets). It collects the current values from the dialog
; and stores them in the associated variable object. Then, the
; inherited AcceptDialog method is called which destroys the widget.

FUNCTION MGS_VarInfoObject::AcceptDialog, event

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      MGS_GUIObject_Error_Message, /Traceback
      RETURN, 0
   ENDIF

   IF Obj_Valid(self.varobject) THEN BEGIN
      name          = self.varnameObj->Get_Value()
      long_name     = self.long_nameObj->Get_Value()
      units         = self.unitsObj->Get_Value()
      missing_value = self.missvalObj->Get_Value()
      min_valid     = self.minvalObj->Get_Value()
      max_valid     = self.maxvalObj->Get_Value()

      self.varobject->SetProperty, name=name, long_name=long_name, $
         units=units, missing_value=missing_value, $
         min_valid=min_valid, max_valid=max_valid
   ENDIF

   RETURN, self->MGS_GUIObject::AcceptDialog(event)

END 


; -----------------------------------------------------------------------------
; ApplyDialog:  (private)
;    This method is called from the event handler if the Apply button
; is pressed (non blocking widgets). It collects all information
; currently stored in the object and "sends an event" ???

FUNCTION MGS_VarInfoObject::ApplyDialog, event

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      MGS_GUIObject_Error_Message, /Traceback
      RETURN, 0
   ENDIF


   IF Obj_Valid(self.varobject) THEN BEGIN
      name          = self.varnameObj->Get_Value()
      long_name     = self.long_nameObj->Get_Value()
      units         = self.unitsObj->Get_Value()
      missing_value = self.missvalObj->Get_Value()
      min_valid     = self.minvalObj->Get_Value()
      max_valid     = self.maxvalObj->Get_Value()

      self.varobject->SetProperty, name=name, long_name=long_name, $
         units=units, missing_value=missing_value, $
         min_valid=min_valid, max_valid=max_valid
   ENDIF

   RETURN, self->MGS_GUIObject::ApplyDialog(event)

END 


; -----------------------------------------------------------------------------
; BuildGUI:  (private)
; This method builds the graphical user interface. Here, just a simple
; compound widget for text input is added to the empty frame provided
; by the generic GUIObject.

PRO MGS_Varinfoobject::BuildGUI

   ;; Arrange the following widgets in one row (not actually needed)
   col1base = Widget_Base(self.layoutID, Col=1)

   ;; top row: variable information
   firstrow = Widget_Base(col1base, col=1, frame=3)
   label1 = Widget_Label(firstrow, Value='Variable properties', $
                             font=self.labelfont, /Align_Left)

   IF Obj_Valid(self.varobject) THEN BEGIN
      self.varobject->GetProperty, name=varname, $
         long_name=long_name, $
         units=units, $
         tname=tname, $
         missing_value=missing_value, $
         min_valid=min_valid, $
         max_valid=max_valid
   ENDIF ELSE BEGIN
      varname = 'var1'
      longname = ''
      units = ''
      tname = 'FLOAT'
      missing_value = !Values.F_NaN
      min_valid = !Values.F_NaN
      max_valid = !Values.F_NaN
   ENDELSE

   varnameID = MGS_Field(firstrow,  $
                      Value=varname, $
                      Title='Name: ', $
                      Label_Left=1, $
                      Object=varnameObj, $
                      XSize=20, $
                      UValue={method:'EVENT_HANDLER', object:self}, $
                      LabelSize=96, $
                      /Label_Right, $
                      /Cr_Only, $
                      Event_Pro="MGS_GUIObject_Widget_Events")

   long_nameID = MGS_Field(firstrow, $
                      Value=long_name, $
                      Title='Long Name: ', $
                      Label_Left=1, $
                      Object=long_nameObj, $
                      XSize=40, $
                      UValue={method:'EVENT_HANDLER', object:self}, $
                      LabelSize=96, $
                      /Label_Right, $
                      /Cr_Only, $
                      Event_Pro="MGS_GUIObject_Widget_Events")

   unitsID = MGS_Field(firstrow, $
                      Value=units, $
                      Title='Units: ', $
                      Label_Left=1, $
                      Object=unitsObj, $
                      XSize=40, $
                      UValue={method:'EVENT_HANDLER', object:self}, $
                      LabelSize=96, $
                      /Label_Right, $
                      /Cr_Only, $
                      Event_Pro="MGS_GUIObject_Widget_Events")

   missvID = MGS_Field(firstrow, $
                      Value=missing_value, $      ; normally type float or double
                      Title='Missing value: ', $
                      Label_Left=1, $
                      Object=missvalObj, $
                      XSize=12, $
                      UValue={method:'EVENT_HANDLER', object:self}, $
                      LabelSize=96, $
                      /Label_Right, $
                      /Cr_Only, $
                      Event_Pro="MGS_GUIObject_Widget_Events")

   minvID = MGS_Field(firstrow, $
                      Value=min_valid, $      ; normally type float or double
                      Title='Min. valid value: ', $
                      Label_Left=1, $
                      Object=minvalObj, $
                      XSize=12, $
                      UValue={method:'EVENT_HANDLER', object:self}, $
                      LabelSize=96, $
                      /Label_Right, $
                      /Cr_Only, $
                      Event_Pro="MGS_GUIObject_Widget_Events")

   maxvID = MGS_Field(firstrow, $
                      Value=max_valid, $      ; normally type float or double
                      Title='Max. valid value: ', $
                      Label_Left=1, $
                      Object=maxvalObj, $
                      XSize=12, $
                      UValue={method:'EVENT_HANDLER', object:self}, $
                      LabelSize=96, $
                      /Label_Right, $
                      /Cr_Only, $
                      Event_Pro="MGS_GUIObject_Widget_Events")


   alltypes = [ 'DOUBLE', 'FLOAT', 'LONG64', 'LONG', 'INT', 'ULONG64', $
                'ULONG', 'UINT', 'BYTE' ]
   typeindex = (Where(alltypes EQ tname))[0] > 0
   typeselObj = FSC_Droplist(firstrow, Value=alltypes, Index=typeIndex, $
      UValue={method:'SetType', object:self}, Title='Data Type:', Space=[1,1])
   dummylabel = Widget_Label(firstrow, $
                             Value='(type change not fully implemented yet!)', $
                             font=self.defaultfont, /Align_Left)

   ;; Define order of tab's
   IF Obj_Valid(varnameObj) THEN varnameObj->SetTabNext, long_nameID
   IF Obj_Valid(long_nameObj) THEN long_nameObj->SetTabNext, unitsID
   IF Obj_Valid(unitsObj) THEN unitsObj->SetTabNext, missvID
   IF Obj_Valid(missvalObj) THEN missvalObj->SetTabNext, minvID
   IF Obj_Valid(minvalObj) THEN minvalObj->SetTabNext, maxvID
   IF Obj_Valid(maxvalObj) THEN maxvalObj->SetTabNext, typeselObj->GetID()

   ;; Second row: Dimension information
   secondrow = Widget_Base(col1base, col=1, frame=3)
   label2 = Widget_Label(secondrow, Value='Dimensions', $
                             font=self.labelfont, /Align_Left)


   ;; Get dimension names and sizes from variable object
   IF Obj_Valid(self.varobject) THEN BEGIN
      self.varobject->GetProperty, ndims=ndims, dims=dims, dimnames=dimnames
      IF ndims GT 0 THEN BEGIN
         FOR i = 0, ndims-1 DO BEGIN
            IF dimnames[i] NE '' THEN BEGIN
               thelabel = '   '+dimnames[i]+' : '+StrTrim(dims[i],2)
            ENDIF ELSE BEGIN
               thelabel = '   Internal size : '+StrTrim(dims[i],2)
            ENDELSE
            labeld1 = Widget_Label(secondrow, $
                                   Value=thelabel, $
                                   font=self.defaultfont, /Align_Left)
         ENDFOR
      ENDIF ELSE BEGIN
         labeld1 = Widget_Label(secondrow, $
                                Value='This is a scalar variable (??)', $
                                font=self.defaultfont, /Align_Left)
      ENDELSE
   ENDIF ELSE BEGIN
      labeld1 = Widget_Label(secondrow, $
                             Value='Variable is undefined', $
                             font=self.defaultfont, /Align_Left)
   ENDELSE


   IF Obj_Valid(varnameObj) THEN self.varnameObj = varnameObj
   IF Obj_Valid(long_nameObj) THEN self.long_nameObj = long_nameObj
   IF Obj_Valid(unitsObj) THEN self.unitsObj = unitsObj
   IF Obj_Valid(missvalObj) THEN self.missvalObj = missvalObj
   IF Obj_Valid(minvalObj) THEN self.minvalObj = minvalObj
   IF Obj_Valid(maxvalObj) THEN self.maxvalObj = maxvalObj
   IF Obj_Valid(typeselObj) THEN self.typeselObj = typeselObj

   RETURN

END


; -----------------------------------------------------------------------------
; GetProperty:
; This method extracts specific object values and returns them to the
; user. Normally, the user should use the GetState() method to
; retrieve the object state in a usable form.

PRO MGS_Varinfoobject::GetProperty, $
     varname=varname,        $ ; The variable name
     long_name=long_name,    $ ; The variable long name
     units=units,            $ ; The physical unit of the variable
     varobject=varobject,    $ ; A replacement object
     _Ref_Extra=extra         ; Inherited and future keywords
                              ;
                              ; Inherited keywords:
                              ; name      : The variable name
                              ; uvalue    : a user-defined value
                              ; window_title
                              ; widget_title
                              ; widget_defaultfont
                              ; widget_labelfont
                              ; set_value_pro
                              ; get_value_func


   ;; Get properties from base object
   self->MGS_GUIObject::GetProperty, _Extra=extra

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error retrieving object properties!'
      RETURN
   ENDIF

   ;; Return object properties
   IF Obj_Valid(self.varnameObj) THEN varname = self.varnameObj->GetValue()


   varobject = self.varobject


END


; -----------------------------------------------------------------------------
; SetProperty:
; This method sets specific object values.

PRO MGS_Varinfoobject::SetProperty, $
     varname=varname,        $ ; The variable name
     long_name=long_name,    $ ; The variable long name
     units=units,            $ ; The physical unit of the variable
     varobject=varobject,    $ ; A replacement object
     _Extra=extra                  ;
                                   ; Inherited keywords:
                                   ; name      : The variable name
                                   ; no_copy   : Don't keep local copy
                                   ;             of uvalue
                                   ; no_dialog : Don't display
                                   ;             interactive dialogs
                                   ; uvalue    : a user-defined value
                                   ; window_title
                                   ; widget_title
                                   ; row_layout
                                   ; no_frame
                                   ; widget_defaultfont
                                   ; widget_labelfont
                                   ; set_value_pro
                                   ; get_value_func


   ;; Set Properties of base object
   self->MGS_GUIObject::SetProperty, _Extra=extra

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      MGS_GUIObject_Error_Message, /Traceback
      RETURN
   ENDIF

   ;; Replace value of center coordinates
   IF N_Elements(varname) GT 0 AND Obj_Valid(self.varnameObj) THEN BEGIN
      self.varnameObj->Set_Value, StrTrim(varname,2)
      IF Obj_Valid(self.varobject) THEN $
         self.varobject->SetProperty, name=StrTrim(varname,2)
      update_title = 1
   ENDIF
   IF N_Elements(long_name) GT 0 AND Obj_Valid(self.long_nameObj) THEN BEGIN
      self.long_nameObj->Set_Value, StrTrim(long_name,2)
   ENDIF
   IF N_Elements(units) GT 0 AND Obj_Valid(self.unitsObj) THEN BEGIN
      self.unitsObj->Set_Value, StrTrim(units,2)
   ENDIF

   ;; Replace current variable Object
   IF N_Elements(varobject) GT 0 THEN BEGIN
      IF Obj_Valid(varobject) THEN BEGIN
         ;; destroy current variable ?????
         ;; Obj_Destroy, self.varobject
         self.varobject = varobject
         update_title = 1
      ENDIF
   ENDIF

   ;; Update the title of the widget if necessary
   IF Keyword_Set(update_title) THEN BEGIN
      ;; Check if current title starts with 'Variable properties'
      IF StrMid(self.window_title, 0, 19) EQ 'Variable properties' THEN BEGIN
         IF Obj_Valid(self.varobject) THEN BEGIN
            self.varobject->GetProperty, name=thevarname
            self->SetProperty, window_title='Variable properties: '+thevarname
         ENDIF
      ENDIF
   ENDIF

   ;; Make sure object is up-to-date and redisplay
   self->UpdateObject
   self->Show

END


; -----------------------------------------------------------------------------
; Cleanup:
;   This method destroys the GUI widget should it still exist, then
; calls the inherited cleanup method.

PRO MGS_Varinfoobject::Cleanup, KillVariable=killvariable

   ;; Clean up child objects 
   Obj_Destroy, self.varnameObj
   Obj_Destroy, self.long_nameObj
   Obj_Destroy, self.unitsObj
   Obj_Destroy, self.missvalObj
   Obj_Destroy, self.minvalObj
   Obj_Destroy, self.maxvalObj
   Obj_Destroy, self.typeselObj  ;; might not be necessary as this object
   ;; should automatically be destroyed together with the widget

   ;; Free pointers
;   Ptr_Free, self.value

   ;; Call parent's cleanup method
   self->MGS_GUIObject::Cleanup

   ;; Kill variable object only if requested
   IF Keyword_Set(Killvariable) THEN Obj_Destroy, self.varobject

END


; -----------------------------------------------------------------------------
; Init:
;   This method initializes the map object. It does not build a GUI,
; but only sets the object property varioables.

FUNCTION MGS_Varinfoobject::Init, $
     varobject,         $
     _Extra=extra  ; Extra keywords from inherited objects
                                ;
                                ; Inherited keywords (from
                                ; MGS_GUIObject and MGS_BaseObject):
                                ; name      : The object name
                                ; no_copy   : Don't retain a copy
                                ;             of uvalue
                                ; no_dialog : Don't display
                                ;             interactive dialogs
                                ; uvalue    : a user-defined value
                                ; window_title
                                ; widget_title
                                ; row_layout
                                ; no_frame
                                ; widget_defaultfont
                                ; widget_labelfont
                                ; set_value_pro
                                ; get_value_func


   ;; Initialize parent object first
   IF not self->MGS_GUIObject::Init(_Extra=extra) THEN RETURN, 0

   ;; Error Handler 
   Catch, theError
   IF theError NE 0 THEN BEGIN
      MGS_GUIObject_Error_Message, /Traceback
      RETURN, 0
   ENDIF

   ;; Check for keywords and parameters.
   IF Obj_Valid(varobject) THEN BEGIN
      self.varobject = varobject
      varobject->GetProperty, name=varname 
   ENDIF ELSE BEGIN
      self.varobject = Obj_New('MGS_Variable')
      varname = ''
   ENDELSE

   IF self.window_title EQ 'Generic widget object' THEN $
      self.window_title = 'Variable properties: '+varname

   RETURN, 1
END


; -----------------------------------------------------------------------------
; MGS_Varinfoobject__Define:
; This is the object definition for the variable info tool.

PRO MGS_Varinfoobject__Define

   struct = { MGS_VARINFOOBJECT, $  ; The MGS_VARINFO object class.
            varobject: Obj_New(),      $    ; The parent object 

            varnameObj: Obj_New(),        $    ; The variable name
            long_nameObj: Obj_New(),   $    ; The variable long_name
            unitsObj: Obj_New(),       $    ; The units object
            missvalObj: Obj_New(),     $    ; The missing value
            minvalObj: Obj_New(),      $    ; The min_valid value
            maxvalObj: Obj_New(),      $    ; The max_valid value
            typeselObj: Obj_New(),     $    ; The droplist object for type selection

           ;;; Object reference to and values of FSC_Inputfield
;           theField: Obj_New(), $
;           fieldname: '',       $    ; The label of the input field
;           value: Ptr_New(),    $    ; Th evalue of the input field

           inherits MGS_GUIObject  $ ; provides basic general properties
         }
END




PRO Example

   ;; Create a variable
   lon = Obj_New('MGS_Variable', findgen(36)*10., name='lon', $
                 long_name='Longitude', units='degrees_east')
   lat = Obj_New('MGS_Variable', findgen(19)*10.-90., name='lat', $
                 long_name='Latitude', units='degrees_north')

   var = Obj_New('MGS_Variable', dist(36,19), lon, lat, $
                 name='Testvariable', long_name='Test for varinfoobject', $
                 units='kg*m^2/s',missing_value=-999.99, $
                 min_valid=0.,max_valid=1000.)

   ;; Now create the variableinfo object
   ;; (Alternatively you can call the ShowInfo method of the variable
   ;; object once it is written ;-)
   info = Obj_New('MGS_VarInfoObject',var)

   ;; Display variable information
   ;; Block the widget, because otherwise we mess things up here
   info->GUI, /block

   ;; Get the variable properties that were displayed in the dialog
   ;; from the variable. They should be changed if you changed them
   ;; and pressed Accept
   lon->GetProperty, name=name, long_name=long_name, units=units, $
      missing_value=missing_value, min_valid=min_valid, max_valid=max_valid
   print, 'Name, long_name, units, missing_value, min_valid, max_valid = ', $
      name, ' ',long_name, ' ', units, missing_value, min_valid, max_valid
      
   ;; Change the variable associated with the info object to display
   ;; information on the longitude variable
   info->SetProperty, varobject=lon

   ;; ... and display info
   info->GUI, /block

   ;; Get the variable properties that were displayed in the dialog
   ;; from the variable. They should be changed if you changed them
   ;; and pressed Accept
   lon->GetProperty, name=name, long_name=long_name, units=units, $
      missing_value=missing_value, min_valid=min_valid, max_valid=max_valid
   print, 'Name, long_name, units, missing_value, min_valid, max_valid = ', $
      name, ' ',long_name, ' ', units, missing_value, min_valid, max_valid
      
    ;; Now delete the info object
   Obj_Destroy, info
   
   ;; and delete the variables
   Obj_Destroy, var
   Obj_Destroy, lon
   Obj_Destroy, lat

END
