;+
; NAME:
;    MGS_Page
;
; PURPOSE:
;    Define a (printable) page and manage the positions of plot panels
;    on this page. All coordinates can be given in cm. The object also
;    supports screen windows, which will be dimensioned according to
;    the page dimensions and sized automatically to fit on the screen.
;
; CATEGORY:
;  Generic Objects
;
; CALLING SEQUENCE:
;    thePage = Obj_New('mgs_page', pageid='AUTO')
;    ; open a screen window
;    dummy = thePage->GetWindowSize(/Open_Window)
;    ; prepare set-up for 3 panels stacked vertically
;    thePage->SetProperty, nrows=3, ncols=1
;    ; get position vector for first plot and plot
;    position = thePage->GetPosition(1,1)
;    plot, findgen(100), position=position
;    ; now add 2nd and third plot
;    position = thePage->GetPosition(1,2)
;    plot, findgen(100), position=position, /noerase
;    position = thePage->GetPosition(1,3)
;    plot, findgen(100), position=position, /noerase
;    ; add the page label and destroy the object
;    thePage->PutLabel
;    Obj_Destroy, theWidget
;
; REQUIREMENTS:
;    None
;
; INHERITING:
;    None
;
; MODIFICATION HISTORY:
;    mgs, 27 Mar 2002: Version 1.0
;
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  2002 Martin Schultz
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution.
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################



; open the postscript device and set margins etc. according to the
; page settings. Default settings are made for color, 8 bits per pixel
; and font=helvetica, font_size=12. The _extra keyword accepts all
; possible device settings.
;    Furthermore, !p.thick is set to 3-times the current value

PRO MGS_Page::OpenPS, _Extra=extra

   ;; store current device for later restoration
   self.olddev = !D.Name
   
   ;; open postscript device and set device options
   set_plot, 'PS'
   
   landscape = (self.pagex GT self.pagey)
   margin = self->GetMargins()

   xsize=self.pagex-margin[0]-margin[2]
   ysize=self.pagey-margin[1]-margin[3]
   IF landscape THEN BEGIN
      xoffset=margin[1]
      yoffset=self.pagex-margin[0]
   ENDIF ELSE BEGIN
      xoffset=margin[0]
      yoffset=margin[1]
   ENDELSE 

   device, /color, bits_per_pixel=8, xsize=xsize, ysize=ysize, $
           xoffset=xoffset, yoffset=yoffset, landscape=landscape, $
           inches=0, /helvetica, font_size=12, _Extra=extra

   !P.thick = !P.thick*3

END 

; close the postscript file and reset to old (screen) device
; if the label keyword is set, the pageid will be printed at the 
; bottom of the page

PRO MGS_Page::ClosePS, label=label

   IF !D.Name NE 'PS' THEN return

   IF keyword_set(label) THEN self->PutLabel, /pageid
   device, /close

   !P.thick = !P.thick/3.
   set_plot, self.olddev

END 

; Print a text at the specified position in absolute coordinates (cm)
; All options of xyouts are available
; If called with no arguments and keywords, PutLabel will print the
; pageid in the bottom right corner of the page.
; Text alignment defaults to "left" unless the pageid is printed
; Default position is 1 cm from right side and 4 mm from page bottom
; pageid is (almost) a dummy keyword to enhance readibility.

PRO MGS_Page::PutLabel, x, y, text, charsize=charsize, align=align, $
                        _Extra=extra, pageid=pageid

   IF n_elements(x) EQ 0 THEN x = self.pagex-1.
   IF n_elements(y) EQ 0 THEN y = 0.4
   IF n_elements(text) EQ 0 OR keyword_set(pageid) THEN text = self.pageid
   IF n_elements(charsize) EQ 0 THEN charsize = 0.5
   IF n_elements(align) EQ 0 THEN BEGIN
      IF text EQ self.pageid THEN align = 1. ELSE align = 0.
   ENDIF 

   xyouts, x/self.pagex, y/self.pagey, text, /norm, $
           charsize=charsize, align=align, _Extra=extra

END 


; compute "optimal" window size for current page setup
; only work sif screen device is active
FUNCTION MGS_Page::GetWindowSize, scale_factor, open_window=open_window

   IF NOT ((!D.Name EQ 'X') OR (!D.Name EQ 'WIN') OR (!D.Name EQ 'MAC')) THEN BEGIN
      message, 'Device must be X, Win, or Mac!', /cont
      return, [0., 0. ]
   ENDIF 

   IF n_elements(scale_factor) EQ 0 THEN scale_factor = self.scale_factor

   ;; get screen size
   device, get_screen_size = scrsize

   scraspect = scrsize[1]/scrsize[0]
   pageaspect = self.pagey/self.pagex

   ytest = scrsize[0]*pageaspect
   IF ytest GT scrsize[1] THEN BEGIN
      reduce = scrsize[1]/ytest
      xsize = scrsize[0]*reduce
      ysize = scrsize[1]
   ENDIF ELSE BEGIN
      xsize = scrsize[0]
      ysize = ytest
   ENDELSE 
   
   IF keyword_set(open_window) THEN window, $
      xsize=xsize*scale_factor, $
      ysize=ysize*scale_factor

   return, [ xsize, ysize ] * scale_factor
END 

; compute plot position for panel in column ic and row ir
; if ic or ir is omitted, a warning is printed, and a full page panel assumed
FUNCTION MGS_Page::GetPosition, ic, ir

   IF n_elements(ic) EQ 0 THEN ic = 0
   IF n_elements(ir) EQ 0 THEN ir = 0
   IF ic EQ 0 OR ir EQ 0 THEN BEGIN
      print,'%MGS_Page::GetPosition: column=0 or row=0; return full page coordinates'
      ncols = 1
      nrows = 1
      panelsize = self->GetPanelSize(1,1)
   ENDIF ELSE BEGIN
      ncols = self.ncols
      nrows = self.nrows
      panelsize = self->GetPanelSize()
   ENDELSE 

   myic = (ic > 1) < ncols
   myir = (ir > 1) < nrows
 
   ;; compute lower left corner and plot width (panel -safety)
   x0 = self.lmar+(myic-1)*panelsize[0]
   y0 = self.bmar+(nrows-myir)*panelsize[1]
   width = panelsize[0]*(1.-self.sx[0]-self.sx[1])
   height = panelsize[1]*(1.-self.sy[0]-self.sy[1])

   ;; adjust lower left corner for "safety margin" and compute upper right
   x0 = x0 + self.sx[0]*panelsize[0]
   y0 = y0 + self.sy[0]*panelsize[1]
   x1 = x0+width
   y1 = y0+height
   position = [ x0, y0, x1, y1 ]

   return, position
END 

; store "safety margins"
; panelmargins = [ left, top, right, bottom ]
; NOTE: if argument is scalar, all panelmargin values assumed identical!!
PRO MGS_Page::SetPanelMargins, margin

   current = [ self.sx[0], self.sy[0], self.sx[1], self.sy[1] ]

   IF n_elements(margin) EQ 0 THEN return

   IF n_elements(margin) EQ 1 THEN BEGIN
      current[*] = margin
   ENDIF ELSE BEGIN
      current[0] = margin[0] 
      current[1] = margin[1]
   ENDELSE 
   IF n_elements(margin) GT 2 THEN current[2] = margin[2]
   IF n_elements(margin) GT 3 THEN current[3] = margin[3]

   self.sx[0] = current[0]
   self.sy[0] = current[1]
   self.sx[1] = current[2]
   self.sy[1] = current[3]

END

; convert margins from cm to page fraction and store in object fields
; margins = [ left, top, right, bottom ]
; NOTE: if argument is scalar, all margin values assumed identical!!
PRO MGS_Page::SetMargins, margin

   current = self->GetMargins()

   IF n_elements(margin) EQ 0 THEN return

   IF n_elements(margin) EQ 1 THEN BEGIN
      current[*] = margin
   ENDIF ELSE BEGIN
      current[0] = margin[0] 
      current[1] = margin[1]
   ENDELSE  
   IF n_elements(margin) GT 2 THEN current[2] = margin[2]
   IF n_elements(margin) GT 3 THEN current[3] = margin[3]

   self.lmar = current[0] / self.pagex
   self.rmar = current[2] / self.pagex
   self.tmar = current[1] / self.pagey
   self.bmar = current[3] / self.pagey

END

; return page margins (and size) in cm
FUNCTION MGS_Page::GetMargins, pagesize=pagesize
   
   pagesize = [ self.pagex, self.pagey ]
   result = [ self.lmar*self.pagex, self.tmar*self.pagey, $
              self.rmar*self.pagex, self.bmar*self.pagey ]

   return, result
END 

; compute panel size as page fraction and store
FUNCTION MGS_Page::GetPanelSize, ncols, nrows

   IF n_elements(ncols) EQ 0 THEN ncols = self.ncols
   IF n_elements(nrows) EQ 0 THEN nrows = self.nrows
   self.panelx = (1.-(self.lmar+self.rmar))/float(ncols) ; > 0.05
   self.panely = (1.-(self.tmar+self.bmar))/float(nrows) ; > 0.05

   RETURN, [ self.panelx, self.panely ]
END


; revert to standard settings 
PRO MGS_Page::Reset

   ;; set default values
   self.pagex = 21.
   self.pagey = 30.
   self.lmar = 3./self.pagex
   self.rmar = 2.5/self.pagex
   self.tmar = 2.5/self.pagey
   self.bmar = 3./self.pagey
   self.nrows = 1L
   self.ncols = 1L
   self.sx = [0.15, 0.05]
   self.sy = [0.1, 0.1]
   self.pageid = ''
   self.scale_factor = 0.95

END 


; overwrite current settings where specified
; NOTE: if pagesize is changed, margins scale accordingly. You must
;     set margins explicitely if you want to preserve the absolute
;     value

PRO MGS_Page::SetProperty, $
                         nrows=nrows, $
                         ncols=ncols, $
                         letter=letter, $
                         a4=a4, $
                         pagesize=pagesize, $   ; width, height [cm]
                         landscape=landscape, $
                         margin=margin, $       ; left, bottom, right, top [cm]
                         panelmargin=panelmargin, $  ; dto. [% of panel]
                         pageid=pageid, $
                         scale_factor=scale_factor   ; for screen windows  

                         
   IF n_elements(nrows) GT 0 THEN self.nrows = long(nrows)
   IF n_elements(ncols) GT 0 THEN self.ncols = long(ncols)
   IF keyword_set(letter) THEN BEGIN
      self.pagex = 8.5*2.54
      self.pagey = 11.*2.54
   ENDIF 
   IF keyword_set(a4) THEN BEGIN
      self.pagex = 21.
      self.pagey = 30.
   ENDIF 
   IF n_elements(pagesize) GT 0 THEN BEGIN
      self.pagex = pagesize[0] 
      IF n_elements(pagesize) GT 1 THEN self.pagey = pagesize[1]
   ENDIF 
   IF keyword_set(landscape) THEN BEGIN
      swap = self.pagex
      self.pagex = self.pagey
      self.pagey = swap
   ENDIF 
   self->SetMargins, margin
   self->SetPanelMargins, panelmargin
   IF n_elements(pageid) GT 0 THEN BEGIN
      IF StrUpCase(pageid) EQ 'AUTO' THEN BEGIN
         german = (keyword_set(letter) EQ 0)
         self.pageid = getenv('USER')+', '+StrDate(GERMAN=german)
      ENDIF ELSE BEGIN
         self.pageid = pageid
      ENDELSE 
   ENDIF 
   IF n_elements(scale_factor) GT 0 THEN BEGIN
      self.scale_factor = scale_factor
   ENDIF 
   ;; Compute default panel size and store in object
   dummy = self->GetPanelSize()

END 


PRO MGS_Page::Cleanup

END 


FUNCTION MGS_Page::Init, $
                         _Extra=extra   ;; same keywords as SetProperty
                         
   ;; set default values
   self->Reset

   ;; overwrite defaults where specified
   self->SetProperty, _extra=extra

   ;; store current device for safety
   self.olddev = !D.Name

   RETURN, 1

END 



PRO MGS_Page__Define

   struct = { MGS_PAGE,    $      ; The object class.
              pagex: 0.,   $      ; page size horizontal in cm
              pagey: 0.,   $      ; page size vertical in cm
              lmar:  0.,   $      ; left margin as fraction of pagex
              rmar:  0.,   $      ; right margin
              tmar:  0.,   $      ; top margin
              bmar:  0.,   $      ; bottom margin
              nrows: 0L,   $      ; number of rows
              ncols: 0L,   $      ; number of columns
              panelx:0.,   $      ; standard panel width
              panely:0.,   $      ; standard panel height
              sx: [0., 0.], $     ; "safety margin" in panel
              sy: [0., 0.], $     ; dto.
              pageid: '',  $      ; page identifier - try 'AUTO'
              scale_factor:0., $  ; scale_factor for screen window size
              olddev:''    $      ; save device name while activating ps
          }

END 
