;+
; NAME:
;    MGS_DrawColors
;
; PURPOSE:
;    This object widget demonstrates the process of creating a widget
;  application with the MGS_BaseGUI class hierarchy. It builds a
;  widget containing a number of color buttons (default is three). 
;  All of these are realized as compound widgets and managed
;  completely via the generic BaseGUI's compound container.
;
; CATEGORY:
;    Widget Objects
;
; CALLING SEQUENCE:
;    thecolors = Obj_New('MGS_DrawColors' [,keywords])
;    thecolors->GUI, /block
;    state = thecolors->GetValue()
;    print,'Color values : ',state
;    Obj_Destroy, thecolors
;
;    You can also use the example procedure at the end of this program
;    file: type ".run mgs_drawcolors__define" then "example, /block"
;
; ARGUMENTS:
;    colors (string vector, input) -> The names of the drawing colors
;
; KEYWORDS:
;    startindex (byte) -> first index in colortable where colors shall
;       be stored (only valid for 8-bit devices)
;    
;    Several other keywords from MGS_ColorButton and MGS_BaseGUI can be
;    passed as well.
;
;    inherited keywords:
;    widget_title (string, input) -> The title of the top level
;          base. This string is displayed as window title AND at the
;          top of the dialog window.
;
;    widget_defaultfont (string, input) -> A font name to be used for
;          the text fields in the widget. Note: Font names are
;          platform dependent!
;
;    widget_labelfont (string, input) -> A font name to be used for
;          the title labels of widget sections. Note: Font names are
;          platform dependent!
;
;    ....
;
; MODIFICATION HISTORY:
;    mgs, 04 Apr 2001: Version 1.0 released
;    mgs, 27 Apr 2001: - changed name (formerly MGS_GUIDemoObject)
;            - now accepts arbitrary number of colors
;-
;
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  2001 Martin Schultz
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution.
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################



; -----------------------------------------------------------------------------
; GetState:
;   This method returns the currently selected color names in a
; structure.

;FUNCTION MGS_DrawColors::GetState

;   self->GetProperty, color1=color1, color2=color2, color3=color3, value=value

;   IF N_Elements(color1) EQ 0 THEN print, 'Color 1 undefined'
;   IF N_Elements(color2) EQ 0 THEN print, 'Color 2 undefined'
;   IF N_Elements(color3) EQ 0 THEN print, 'Color 3 undefined'
;   IF N_Elements(value) EQ 0 THEN print, 'Input field value undefined'

;   RETURN, { colors: [color1, color2, color3], value:value }

;END


; -----------------------------------------------------------------------------
; GetValue:
;    This method returns the latest valid "value" of the widget as a
; vector of colornames.

FUNCTION MGS_Drawcolors::GetValue, ok=ok, _Extra=e

   ;; Error handler
   ;; ...

   ok = 0

   ;; Get the object identifiers for the field widgets
   fobs = self.compound->Get(/All)

   FOR i=0L, N_Elements(fobs)-1 DO BEGIN
      IF Obj_Valid(fobs[i]) THEN BEGIN
         ;; get value for this field
         thevalue = fobs[i]->GetValue()
         IF N_Elements(retval) EQ 0 THEN retval = thevalue $
         ELSE retval = [ retval, thevalue ]
      ENDIF
   ENDFOR

   RETURN, retval
END


; -----------------------------------------------------------------------------
; SetFieldValue:
;    This method changes the color of all colorbuttons whose name
; matches the fieldname expression. Fieldname may contain wildcards
; ('*' or '?').

PRO MGS_DrawColors::SetFieldValue, fieldname, color

   IF N_Elements(fieldname) EQ 0 THEN BEGIN
      self->ErrorMessage, 'Must supply a fieldname'
      RETURN
   ENDIF

   IF N_Elements(color) EQ 0 THEN BEGIN
      self->ErrorMessage, 'Must supply a color name'
      RETURN
   ENDIF

   ;; Retrieve all compound widgets that match the fieldname
   ;; expression
   IF Obj_Valid(self.compound) THEN BEGIN
      cwobjects = self.compound->Get(name=fieldname[0])
      FOR i = 0L, N_Elements(cwobjects)-1 DO BEGIN
         IF Obj_Valid(cwobjects[i]) THEN BEGIN
            cwobjects[i]->GetProperty, name=theName
            cwobjects[i]->SetValue, color[i < (N_Elements(color)-1)]
         ENDIF 
      ENDFOR
   ENDIF
 
END


; -----------------------------------------------------------------------------
; BuildGUI:  (private)
;   This method only adds a category label to the layout widget. All
; the compound widgets will be added automatically.

PRO MGS_DrawColors::BuildGUI

   ;; All we need to do here is supply a title label
   labelID = Widget_Label(self.layoutID, Value=self.window_title, $
                         font=self.labelfont)

END


; -----------------------------------------------------------------------------
; GetProperty:
;    Not overwritten.

; PRO MGS_DrawColors::GetProperty, $
;   _Ref_Extra=extra         ; Inherited and future keywords

   ;; Get properties from base object
;   self->MGS_BaseGUI::GetProperty, _Extra=extra

; END


; -----------------------------------------------------------------------------
; SetProperty:
;    Currently, we allow changing of colors only via the SetFieldValue
; method. There are no additional keywords to the BaseGUI SetProperty
; method, so we won't overwrite it.

; PRO MGS_DrawColors::SetProperty, $
;   _Extra=extra  ; Extra keywords from inherited objects

   ;; Set Properties of base object
;   self->MGS_BaseGUI::SetProperty, _Extra=extra

   ;; Error Handler
; END

; -----------------------------------------------------------------------------
; Cleanup:
;   It is not necessary to overwrite the genric GUI object cleanup method!

; -----------------------------------------------------------------------------
; Init:
;   This method initializes the colorpicker object.

FUNCTION MGS_DrawColors::Init, $
   Colors=colors, $
   Labeltext=labeltext, $
   Startindex=startindex, $
   Title=title,  $
   _Extra=extra  ; Extra keywords from inherited objects
                                ;
                                ; Inherited keywords (from
                                ; MGS_BaseGUI and MGS_BaseObject):
                                ; name      : The object name
                                ; no_copy   : Don't retain a copy
                                ;             of uvalue
                                ; no_dialog : Don't display
                                ;             interactive dialogs
                                ; uvalue    : a user-defined value
                                ; window_title
                                ; widget_title
                                ; row_layout (ignored)
                                ; no_frame (ignored)
                                ; widget_defaultfont
                                ; widget_labelfont


   ;; Initialize parent object
   IF not self->MGS_BaseGUI::Init(_Extra=extra) THEN RETURN, 0

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error initialising object'
      RETURN, 0
   ENDIF

   ;; Check keywords and parameters.
   IF N_Elements(colors) EQ 0 THEN BEGIN
      colors = [ 'NAVY', 'YELLOW', 'RED' ]
   ENDIF 
   IF N_Elements(startindex) EQ 0 THEN $
      startindex = ( !D.Table_Size - N_Elements(colors) -1 ) > 0

   IF N_Elements(labeltext) EQ 0 THEN BEGIN
      labeltext = 'Color '+String(lindgen(N_Elements(colors))+1,format='(i3)')
   ENDIF ELSE BEGIN
      IF N_Elements(labeltext) NE N_Elements(colors) THEN BEGIN
         self->ErrorMessage, 'Number of labels must equal number of colors'
         RETURN, 0
      ENDIF 
   ENDELSE 

   IF N_Elements(title) EQ 0 THEN title='Colors'

   ;; Create compound widget objects (color pickers)
   FOR i=0L,N_Elements(colors)-1 DO BEGIN
      colorObj = Obj_New('MGS_ColorButton', $
                         color=colors[i], $
                         index=startindex+i, $
                         LabelText=labeltext[i], $
                         LabelSize=76, $
                         /Align_Left, $
                         Name=labeltext[i], $
                         _Extra=extra)
      IF Obj_Valid(colorObj) THEN self.compound->Add, colorObj
   ENDFOR

   ;; Reset some of the generic default properties
   IF self.window_title EQ 'Generic widget object' THEN $
      self.window_title = 'Drawcolors'
   self.layout_frame = 0    ;; Don't put a frame around this widget

   self.window_title=title

   RETURN, 1
END


; -----------------------------------------------------------------------------
; MGS_DrawColors__Define:
; This is the object definition for the color picker object.
; It inherits from MGS_BaseGUI which provides the generic widget
; functionality, and from MGS_BaseObject the abilities to set and
; query an object name and a uvalue. The base object also provides a
; general method for display of error messages which can be directed
; to a message dialog or to the log screen via the no_dialog flag.

PRO MGS_DrawColors__Define

   struct = { MGS_DrawColors, $

              inherits MGS_BaseGUI  }

END


; -----------------------------------------------------------------------------
; Example:
;    Demonstrate the functionality of this object
; NOTE:
; (1) To test the SetFieldValue method, call example as "example,
; object=o", then e.g. "o->SetFieldValue,'curve*','yellow'" or
; "o->SetFieldValue, 'curve*', ['yellow','red']".

PRO Example, labels, colors, block=block, object=object, _EXTRA=extra

   IF N_Elements(labels) EQ 0 THEN $
      labels = [ 'foreground', 'background', 'axes', 'curve 1', 'curve 2']

   IF N_Elements(colors) EQ 0 THEN $
      colors = [ 'black', 'coral', 'black', 'red', 'violet red']

   thegui = Obj_New('MGS_DrawColors', $
                    colors=colors, $
                    labeltext=labels, $
                    _Extra=extra)

   thegui->GUI, block=keyword_Set(block)
   state = thegui->GetValue()
   print, 'Selected colors : ',state

   IF Arg_Present(object) EQ 0 THEN Obj_Destroy, thegui ELSE object=thegui

END
