;+
; NAME:
;    MGS_ColorButton
;
; PURPOSE:
;    This is a widget object derived from the MGS_BaseGUI Object. It contains a
;  label and a color button that can be pressed to display a color
;  picking tool. Much of this has been written by David Fanning who
;  has a similar program named CW_ColorButton. However, this program
;  carries object orientation one step further and demonstrates how to
;  build a compound widget object using the functionality provided by
;  the generic MGS_BaseGUI Object. 
;    One of the advantages of this approach is that this object can be
;  used as a blocking or non-blocking standalone widget with no extra
;  code. The difference is only in the call to the GUI method.
;    See MGS_GUIDemoObject for an example ho weasy it is to use the
;  drawcolor object as a compound object in your applications.
;
; AUTHOR:
;
;   Dr. Martin Schultz
;   Max-Planck-Institut fuer Meteorologie
;   Bundesstr. 55, D-20146 Hamburg
;   email: martin.schultz@dkrz.de
;
; CATEGORY:
;    Widget Objects
;
; CALLING SEQUENCE:
;    ;; As a standalone widget:
;    thecolor = Obj_New('MGS_ColorButton' [,keywords])
;    thecolor->GUI
;    color = FSC_Color(thecolor->GetValue(), !D.Table_Size-2)
;    Obj_Destroy, thecolor
;    ;; As compound widget:
;    ;; (1) In the master widget Init method
;    thecolor = Obj_New('MGS_ColorButton' [,keywords])
;    IF Obj_Valid(thecolor) THEN self.compound->Add, thecolor
;    ;; (2) In the master widget GetState or GetValue method
;    color = FSC_Color(self.thecolor->GetValue(), !D.Table_Size-2)
;
; ARGUMENTS:
;
; KEYWORDS:
;    color (string, input) -> The name of the default color. Use
;       FSC_Color(/Names) to retrieve the possible names of the 88
;       standard colors.
;
;    index (integer, input) -> The color table index of the
;          color. Only relevant for 8 bit displays. Default is
;          !D.Table_Size-2. 
;
;    xsize (integer, input) -> Width of the color button
;
;    ysize (integer, input) -> Height of the color button
;
;    LabelText (string, input) -> Text value of the label
;
;    LabelSize (integer, input) -> Width of the label
;
;    Align_Left_Label (boolean, input) -> Align the label text to the
;          left
;
;    Align_Right_Label (boolean, input) -> Align the label text to the
;          right
;
;    inherited keywords:
;    widget_title (string, input) -> The title of the top level
;          base. This string is displayed as window title AND at the
;          top of the dialog window.
;
;    widget_defaultfont (string, input) -> A font name to be used for
;          the text fields in the widget. Note: Font names are
;          platform dependent!
;
;    widget_labelfont (string, input) -> A font name to be used for
;          the title labels of widget sections. Note: Font names are
;          platform dependent!
;
; REQUIREMENTS:
;    Inherits from mgs_guiobject and mgs_baseobject 
;    Uses FSC_Color, PickColorName by David Fanning
;
; DESCRIPTION:
;
; MODIFICATION HISTORY:
;    mgs, 18 Feb 2001: Version 0.1
;    mgs, 04 Apr 2001: Version 1.0 released
;-
;
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  2001 Martin Schultz
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution.
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################



; -----------------------------------------------------------------------------
; GetState:
;   This method returns the currently selected color name and the
; color table index in a structure.

;FUNCTION MGS_ColorButton::GetState

;   RETURN, { color:self.color, index:self.index }

;END


; -----------------------------------------------------------------------------
; GetValue:
;   This method returns the currently selected color name.

;FUNCTION MGS_ColorButton::GetValue

;   RETURN, self.color

;END


; -----------------------------------------------------------------------------
; Pickcolor: (private)
;   This method is called if the color button is pressed. It calls the
; pickcolorname tool and lets the user select a new
; color. Afterwards, the draw widget is refreshed.

FUNCTION MGS_ColorButton::Pickcolor, event

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error selecting color by name'
      RETURN, 0
   ENDIF

   ;; React only on button down events
   IF event.press NE 1 THEN RETURN, 0

   ;; Ignore double clicks
   IF event.clicks GT 1 THEN RETURN, 0

   ;; Get current color value 
   currentColor = *self.value

   ;; Display color picking tool
   newcolor = PickColorName(currentColor, $
                            Group_Leader=self.tlb, $
                            Title=self.widget_title, $
                            Cancel=cancelled)
   IF NOT cancelled THEN *self.value = newcolor

   ;; Swallow event if no color change occured
   IF currentColor EQ newcolor THEN RETURN, 0

   ;; Create an event message for notification of other objects
   retval = { object: self, $
             eventtype: 'COLORCHANGE' }

   ;; Update graphical display
   self->Show

   RETURN, retval
END


; -----------------------------------------------------------------------------
; Show:
;   This method draws the content of the draw widget. Here, it is
; simply a filled rectangle.

PRO MGS_ColorButton::Show

   IF NOT Widget_Info(self.tlb, /Valid_ID) THEN RETURN

   Widget_Control, self.drawID, Get_Value=wid
   currentwindow = !D.Window
   WSet, wid
   PolyFill, [0, 0, 1, 1, 0], [0, 1, 1, 0, 0], /Normal, $
      Color=FSC_Color(*self.value, self.index)
   IF currentwindow NE -1 THEN WSet, currentwindow

   ;; Call inherited Show method
   self->MGS_BaseGUI::Show

END


; -----------------------------------------------------------------------------
; BuildGUI:  (private)
; This method builds the graphical user interface. The drawcolor
; widget simply consists of a label and a draw widget.

PRO MGS_ColorButton::BuildGUI

   ;; Arrange the following widgets in one row (not actually needed)
   row1base = Widget_Base(self.layoutID, Row=1)

   ;; Get label properties
   IF NOT Ptr_Valid(self.thelabel) THEN BEGIN
      self->ErrorMessage, 'Invalid pointer for label properties!'
      RETURN
   ENDIF
   thelabel = *self.thelabel

   labelID = Widget_Label(self.layoutID, Value=thelabel.value, $
                          Scr_XSize=thelabel.size, $
                          Align_Left=thelabel.align_left, $
                          Align_Right=thelabel.align_right)

   drawID = Widget_Draw(self.layoutID, XSize=self.drawxsize, $
                        YSize=self.drawysize, $
                        UValue={method:'PICKCOLOR', object:self}, $
                        Button_Events=1)

   self.drawID = drawID

END

; -----------------------------------------------------------------------------
; GetProperty:
; This method extracts specific object values and returns them to the
; user. Normally, the user should use the GetState() method, or more
; specific, GetValue(), to retrieve the object state in a usable form.

PRO MGS_ColorButton::GetProperty, $
     XSize=xsize,   $              ; The width of the color drawing widget
     YSize=ysize,   $              ; The height of the color drawing widget
     color=color, $                ; A new color name
     index=index, $                ; A new color index
     labelproperties=label,  $     ; The properties of the label
     _Ref_Extra=extra         ; Inherited and future keywords
                              ;
                              ; Inherited keywords:
                              ; name      : The variable name
                              ; uvalue    : a user-defined value
                              ; window_title
                              ; widget_title
                              ; widget_defaultfont
                              ; widget_labelfont
                              ; set_value_pro
                              ; get_value_func


   ;; Get properties from base object
   self->MGS_BaseGUI::GetProperty, _Extra=extra

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error retrieving object properties!'
      RETURN
   ENDIF

   ;; Return object properties
   xsize = self.DrawXSize
   ysize = self.DrawYSize
   color = *self.value
   index = self.index
   IF Ptr_Valid(self.thelabel) THEN $
      label = *self.thelabel  $
   ELSE  $
      self->Undefine, label


END


; -----------------------------------------------------------------------------
; SetProperty:
; This method sets specific object values.
; NOTE: XSize and YSize only take effect when the widget is rebuilt.

PRO MGS_ColorButton::SetProperty, $
     XSize=xsize,   $              ; The width of the color drawing widget
     YSize=ysize,   $              ; The height of the color drawing widget
     color=color, $                ; A new color name
     index=index, $                ; A new color index
     _Extra=extra                  ;
                                   ; Inherited keywords:
                                   ; name      : The variable name
                                   ; no_copy   : Don't keep local copy
                                   ;             of uvalue
                                   ; no_dialog : Don't display
                                   ;             interactive dialogs
                                   ; uvalue    : a user-defined value
                                   ; window_title
                                   ; widget_title
                                   ; row_layout
                                   ; no_frame
                                   ; widget_defaultfont
                                   ; widget_labelfont
                                   ; set_value_pro
                                   ; get_value_func


   ;; Set Properties of base object
   self->MGS_BaseGUI::SetProperty, _Extra=extra

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error setting object properties'
      RETURN
   ENDIF

   ;; Replace value of draw widget size
   IF N_Elements(xsize) GT 0 THEN $
      self.DrawXSize = xsize
   IF N_Elements(ysize) GT 0 THEN $
      self.DrawYSize = ysize

   ;; Replace color name or index value
   IF N_Elements(color) GT 0 THEN BEGIN 
      Ptr_Free, self.value
      self.value = Ptr_New(String(color))
   ENDIF 
   IF N_Elements(index) GT 0 THEN $
      self.index = (index > 0) < (!D.Table_Size-1)

   ;; Make sure object is up-to-date and redisplay
   self->UpdateObject
   self->Show

END


; -----------------------------------------------------------------------------
; Cleanup:
;   This method cleans up the memory, then calls the inherited cleanup
; method.


PRO MGS_ColorButton::Cleanup

   ;; Free pointers
   Ptr_Free, self.thelabel

   ;; Call parent's cleanup method
   self->MGS_BaseGUI::Cleanup

END


; -----------------------------------------------------------------------------
; Init:
;   This method initializes the file colorpicker object.

FUNCTION MGS_ColorButton::Init, $
   Color=color, $
   Index=index, $
   Align_Left_Label=align_left, $
   Align_Right_Label=align_right, $
   LabelSize=labelsize, $
   LabelText=label, $
   XSize=xsize,  $   ; Size of draw widget
   YSize=ysize,  $
   _Extra=extra  ; Extra keywords from inherited objects
                                ;
                                ; Inherited keywords (from
                                ; MGS_BaseGUI and MGS_BaseObject):
                                ; name      : The object name
                                ; no_copy   : Don't retain a copy
                                ;             of uvalue
                                ; no_dialog : Don't display
                                ;             interactive dialogs
                                ; uvalue    : a user-defined value
                                ; window_title
                                ; widget_title
                                ; row_layout (ignored)
                                ; no_frame (ignored)
                                ; widget_defaultfont
                                ; widget_labelfont
                                ; set_value_pro
                                ; get_value_func


   ;; Initialize parent object
   IF not self->MGS_BaseGUI::Init(_Extra=extra) THEN RETURN, 0

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error initialising object'
      RETURN, 0
   ENDIF

   ;; Check keywords and parameters.
   IF N_Elements(color) EQ 0 THEN color = 'WHITE'
   IF N_Elements(index) EQ 0 THEN index = !D.Table_Size-2
   IF N_Elements(label) EQ 0 THEN label = "Color"
   IF N_Elements(labelsize) EQ 0 THEN labelsize = 0
   IF N_Elements(xsize) EQ 0 THEN xsize = 20
   IF N_Elements(ysize) EQ 0 THEN ysize = 20
   label_left = Keyword_Set(align_left)
   label_right = Keyword_Set(align_right)

   ;; Store label properties in a structure
   ;; (even better would be to create a label object!)
   thelabel = { value:label, size:labelsize, $
                align_left:label_left, align_right:label_right }
   self.thelabel = Ptr_New(thelabel)

   self.drawXSize = xsize
   self.drawYSize = ysize
   self.value = Ptr_New(String(color))
   self.index = (index > 0) < (!D.Table_Size-1)

   ;; Reset some of the generic default properties
   IF self.window_title EQ 'Generic widget object' THEN $
      self.window_title = 'Color'
   self.layout_frame = 0    ;; Don't put a frame around this widget
   self.column_layout = 0   ;; Show label to the left instead of above draw widget
   self.updatemode = 2      ;; Always update object value immediately

   RETURN, 1
END


; -----------------------------------------------------------------------------
; MGS_ColorButton__Define:
; This is the object definition for the color picker object.
; It inherits from MGS_BaseGUI which provides the generic widget
; functionality, and from MGS_BaseObject the abilities to set and
; query an object name and a uvalue. The base object also provides a
; general method for display of error messages which can be directed
; to a message dialog or to the log screen via the no_dialog flag.

PRO MGS_ColorButton__Define

   struct = { MGS_ColorButton, $
              thelabel: Ptr_New(),  $
              drawID: 0L,   $
              drawXSize: 0L, $
              drawYSize: 0L, $
              index: 0L,    $

              inherits MGS_BaseGUI  }

END

