;+
; NAME:
;   mgs_basevariable (object)
;
; PURPOSE:
;     This object stores arbitrary data and provides utitilies to
;   access data, extract data slices, convert data to a different
;   type, filter numerical or string data, and query the data type
;   (numeric, string, or other). A rudimentary functionality to store
;   and retrieve attribute values is also implemented.
;
;     The basevariable object inherits from MGS_BaseObject which
;   provides basic functionality for the handling of errors. While the
;   functionality of the basevariable object may be sufficient for
;   some applications, you will typically want to use a derived class
;   for actual use. E.g., the MGS_Variable object provides much greater
;   functionality for numerical data.
;
; AUTHOR:
;
;   Dr. Martin Schultz
;   Max-Planck-Institut fuer Meteorologie
;   Bundesstr. 55, D-20146 Hamburg
;   email: martin.schultz@dkrz.de
;
; CATEGORY:
;   General object programming
;
; CALLING SEQUENCE:
;
; NOTES:
;    *** THIS OBJECT STILL NEEDS TO UNDERGO RIGOROUS TESTING !! ***
;
;   (1) In order to facilitate access and to allow for scanning file
;   headers before loading the actual data, the object contains
;   seperate values for the data type and dimensions. As long as no
;   data are stored in the object, these values can be set
;   independently via the SetProperty method. If you store data or
;   change the data type, however, these values will automatically be
;   adjusted to reflect the current state. If you want to preserve the
;   information from the file header, you should store it elsewhere
;   (e.g. as a structure in the object's UValue).'DATA_TYPE_CHANGED'
;
;   (2) Some methods accept a data argument. This will cause them to
;   operate on "external" data rather than the data that are stored in
;   the object. A data argument should be recognized even if it is
;   undefined (in this case, an error should occur), however, this has
;   not been implemented consiostently, yet.
;
;   (3) Data attributes will usually be structures (possibly
;   nested). You can use the IsStructure method with the tags keyword
;   to determine if a specific attribute value is present.
;
; EXAMPLE:
;
;
;
; MODIFICATION HISTORY:
;   Martin G. Schultz, 30 May 2000: VERSION 1.00
;   mgs, 05 Mar 2001: VERSION 1.10
;       - changed several methods to provide easier and more
;         consistent user interface
;       - added 64 bit and unsigned integer types
;       - clearer distinction between basevariable and (scientific)
;         variable objects (see mgs_variable__define.pro)
;   mgs, 10 Mar 2001: VERSION 1.20
;       - filter now returns index array instead of data (this makes
;         it much more object like)
;       - added several functions to improve filter functionality
;   mgs, 29 May 2001: 
;       - bug fix in validate
;
;-
;
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  2000-2001 Martin Schultz
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution.
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################


; =====================================================================================
; Utility procedures:
;     LinkDimensions  : Given an array of variable objects,
;                       LinkDimensions replaces the names of
;                       dimensions with object references to the
;                       respective variables within the array

; ------------------------------------------------------------------------------------
; LinkDimensions:
;   This procedure goes through a list of variable objects and sets an
; object reference for each dimension pointing to the appropriate
; dimension variable (if defined).
;   This works only if the name of the dimension is the same as the
; name of the variable holding the values for that dimension (this is
; a pretty common netcdf convention).
; NOTE: ****  THIS ROUTINE WILL AT SOME POINT BE REMOVED AND REPLACED
;             BY A METHOD IN A VARIABLE CONTAINER OBJECT ******

PRO LinkDimensions, varobjects

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error linking dimension variables in '+self.name
      RETURN
   ENDIF

   ;; No variables stored or no names available, return
   NVars = N_Elements(varobjects)
   IF NVars EQ 0 THEN RETURN   

   ;; Extract the names of all variables
   varnames = StrArr(NVars)
   FOR i=0L,NVars-1 DO BEGIN
      varobject[i]->GetProperty, name=name
      varnames[i] = name
   ENDFOR
   varnames = StrUpCase(varnames)

   ;; Loop through all variables and look for the dimension names. Try
   ;; to find them in the Varnames array and replace the values by a
   ;; link. 
   FOR i=0L,NVars-1 DO BEGIN
      FOR j=1,8 DO BEGIN
         varobject[i]->GetDimVar, j, name
         ;; Check if string type
         IF Size(name, /TName) EQ 'STRING' THEN BEGIN
            w = Where(varnames EQ StrUpCase(name), wcnt)
            IF wcnt GT 0 THEN BEGIN
               varobject[i]->SetDimVar, j, varobject[w[0]]
            ENDIF
         ENDIF
      ENDFOR
   ENDFOR
   
END

; -------------------------------------------------------------------------------------

; =====================================================================================
; Utility methods:
;     ConvertType : Convert a data argument to a specific type
;     ExtractHyperslice  : Return data array with one dimension fixed

; -------------------------------------------------------------------------------------
; ConvertType:
;   This method accepts a data argument and converts it to a specified
; data type. If no data argument is supplied, the data stored in the
; object will be used. The target data type can be given in four
; ways:
; - set the tname keyword to the name of the data type (string)
; - set one of the logical flags (double, float, ...)
; - set the integer type vale (see IDL online help on data type
;   values)
; - specify the numerical rank (numerical data only; see IsNumeric
;   method)
;
;   The truncate, round, and clip keywords determine, how data are
; converted from a "higher" to a "lower" numerical type rank. The
; truncate keyword is purely for readability of code; truncation is
; IDL's default for type conversion. Alternatively, you can round
; float data to the nearest (long64, long, or short) integer. Be aware
; that you can end up with unreasonable values in both cases, unless
; the clip keyword is set. Example: "print, fix(32768L)" yields
; -32768). Setting the clip keyword will ensure that the data does not
; extend beyond the valid data range given by the new data type. 
;
;   The minimum and maximum possible (valid) data values for the new
; type can be returned via the min_valid and max_valid keywords. In
; order to avoid machine specific differences, the limit for double
; precision values is set to 1.D300, and the limit for float is set to
; 1.E31. If you supply a named variable for the EPS keyword, the
; minimum positive value that is discernible from zero for the target
; data type will be returned. This value is machine dependent (for
; integer types, it will always be 1).
;
;   You can also use this method to convert numerical data to
; formatted strings. Use the format keyword to specify the output format.

pro MGS_BaseVariable::ConvertType,  $
                data,                 $ ; The data to be converted
                tname=tname,          $ ; A string containing the new data type name
                type=type,            $ ; Convert to type number (see online help)
                rank=rank,            $ ; Convert to numerical rank (see IsNumeric method)
                double=double,        $ ; Convert data to double
                float=float,          $ ; Convert data to float
                l64=l64,              $ ; Convert to 64 bit long
                long=long,            $ ; Convert data to long
                integer=integer,      $ ; Convert data to integer
                ul64=ul64,            $ ; Convert to unsigned 64 bit integer
                ulong=ulong,          $ ; Convert to unsigned long
                uint=uint,            $ ; Convert to unsigned int
                byte=byte,            $ ; Convert data to byte
                string=string,        $ ; Convert data to (formatted) strings
                format=format,        $ ; String output format
                truncate=truncate,    $ ; Truncate the data if necessary (default)
                round=round,          $ ; Round the data if converting to integer type
                clip=clip,            $ ; Clip data to valid range before converting
                min_valid=min_valid,  $ ; Return the minimum possible value for the new type
                max_valid=max_valid,  $ ; Return the maximum possible value for the new type
                eps=eps         ; Return machine specific minimum number that is discernible from zero 


   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   newtype = 'UNKNOWN'
   min_valid = 0L
   max_valid = 0L
   eps = 0.

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error converting data type to '+newtype+'!'
      RETURN
   ENDIF

   ;; Determine new data type
   ;; Note: newtype remains 'UNKNOWN' 
   typedef = [ 'BYTE', 'INT', 'LONG', 'FLOAT', 'DOUBLE', $
               '*COMPLEX', 'STRING', '*STRUCTURE', '*DCOMPLEX', $
               '*POINTER', '*OBJREF', 'UINT', 'ULONG', 'LONG64', $
               'ULONG64' ]

   rankdef = [ 'BYTE', 'UINT', 'ULONG', 'ULONG64', $
               'INT', 'LONG', 'LONG64', 'FLOAT', $
               'DOUBLE' ]

   IF N_Elements(tname) GT 0 THEN BEGIN
       newtype = StrUpCase(tname)
       ok = ( Where(typedef EQ newtype) )[0] GE 0
       IF NOT ok THEN BEGIN
          self->ErrorMessage, 'Invalid target type '+newtype+'!'
          RETURN
       ENDIF
   ENDIF ELSE BEGIN
       IF Keyword_Set(double)   THEN newtype = 'DOUBLE'
       IF Keyword_Set(float)    THEN newtype = 'FLOAT'
       IF Keyword_Set(l64)      THEN newtype = 'LONG64'
       IF Keyword_Set(long)     THEN newtype = 'LONG'
       IF Keyword_Set(integer)  THEN newtype = 'INT'
       IF Keyword_Set(ul64)     THEN newtype = 'ULONG64'
       IF Keyword_Set(ulong)    THEN newtype = 'ULONG'
       IF Keyword_Set(uint)     THEN newtype = 'UINT'
       IF Keyword_Set(byte)     THEN newtype = 'BYTE'
       IF Keyword_Set(string)   THEN newtype = 'STRING'

       IF N_Elements(type) GT 0 THEN BEGIN
           newtype = typedef[ ( (type[0]-1) > 0) < 14 ]
           IF StrMid(newtype,0,1) EQ '*' THEN BEGIN
              self->ErrorMessage, 'Invalid target type '+StrMid(newtype,1,999)+'!'
              RETURN
           ENDIF
        ENDIF

        IF N_Elements(rank) GT 0 THEN BEGIN
           newtype = rankdef[ ( rank[0] > 0 ) < 8 ]
        ENDIF
   ENDELSE

   ;; Determine whether to use data argument or object's data:
   argok = self->UseDataArgument(data, arg_present(data))
   IF argok EQ 0 THEN BEGIN
      data = self->GetData(result=result)
   ENDIF ELSE IF argok EQ 1 THEN BEGIN
      result = 'OK'
   ENDIF 
   
   ;; Return if no data passed or no data stored in object
   IF result NE 'OK' THEN BEGIN
      self->ErrorMessage, 'No data to convert!', /Warning
      RETURN
   ENDIF

   ;; Determine current data type and return if type is not
   ;; convertible
   ;; Note: give strings a chance, they may contain valid numbers
   datatype = Size(data, /TYPE)
   IF StrMid(typedef[datatype-1],0,1) EQ '*' THEN BEGIN
      self->ErrorMessage, 'Data type '+StrMid(typedef[datatype-1],1,999)+ $
        ' cannot be converted!'
      RETURN
   ENDIF

   ;; Set minimum, maximum valid values and eps values
   ;; Need to apply trick for most negative integers (???? not for linux!)
   CASE newtype OF 
       'DOUBLE' : BEGIN
                     min_valid = -1.0D300
                     max_valid = +1.0D300
                     eps = (MaChar(/Double)).eps
                  END
       'FLOAT'  : BEGIN
                     min_valid = -1.0E31
                     max_valid = +1.0E31
                     eps = (MaChar()).eps
                  END
       'LONG64' : BEGIN
                     min_valid = -2LL^63   ; -9223372036854775808LL
                     max_valid =  2LL^63-1 ; +9223372036854775807LL
                     eps = 1LL
                  END
       'LONG'   : BEGIN
                     min_valid = -2L^31    ; -2147483648L
                     max_valid =  2L^31-1  ; +2147483647L
                     eps = 1L
                  END
       'INT'    : BEGIN
                     min_valid = -2^15     ; -32768
                     max_valid =  2^15-1   ; +32767
                     eps = 1
                  END
       'ULONG64' : BEGIN
                     min_valid =  0ULL 
                     max_valid =  2ULL^64-1 ; +18446744073709551615
                     eps = 1ULL
                  END
       'ULONG'  : BEGIN
                     min_valid =  0UL
                     max_valid =  2UL^32-1  ; +4294967295
                     eps = 1UL
                  END
       'UINT'   : BEGIN
                     min_valid =  0U
                     max_valid =  2U^16-1   ; +65535
                     eps = 1U
                  END
       'BYTE'   : BEGIN
                     min_valid = 0B
                     max_valid = 255B
                     eps = 1B
                  END
       'STRING' : BEGIN
                  END
   ENDCASE


   ;; Determine the rank of the data and target type
   ;; will be -1 if type is string
   datarank = ( Where(rankdef EQ typedef[datatype-1]) )[0]
   targetrank = ( Where(rankdef EQ newtype) )[0]

   IF datarank EQ targetrank THEN RETURN   ;; no conversion necessary

   ;; Handle string conversion first
   IF targetrank LT 0 THEN BEGIN
      data = Reform( String(data, format=format), Size(data, /Dimensions) )
      IF argok EQ 0 THEN GOTO, Store_Back
      RETURN
   ENDIF

   ;; If clipping requested, get out-of-bounds indices before
   ;; converting. Attention: IDL Quirk! (-2L^31 LT 0) = 1 but (-2L^31
   ;; LT 0ULL) = 0.
   IF Keyword_Set(clip) THEN BEGIN
       low = Where( data LT min_valid, lcount )
       high = Where( data GT max_valid, hcount )
       IF targetrank LT 4 THEN low = Where(long64(data) LT min_valid, lcount)  ; ?????
   ENDIF ELSE BEGIN
       lcount = 0L
       hcount = 0L
   ENDELSE

   ;; Round or truncate data (or convert double to float or vice versa)
   IF Keyword_Set(round) THEN BEGIN
       CASE newtype OF 
           'DOUBLE' : data = Double(data)
           'FLOAT'  : data = Float(data)
           'LONG64' : data = Long64(Round(data))
;;           'LONG64' : data = Round(data, /L64)     ;; IDL 5.4 only !!
           'LONG'   : data = Round(data)
           'INT'    : data = Fix(Round(data))
           'ULONG64': data = ULong64(Round(data))
;;           'ULONG64': data = ULong64(Round(data, /L64))   ;; IDL 5.4 only !! 
           'ULONG'  : data = ULong(Round(data))
           'UINT'   : data = UInt(Round(data))
           'BYTE'   : data = Byte(Round(data))
       ENDCASE
   ENDIF ELSE BEGIN
       CASE newtype OF 
           'DOUBLE' : data = Double(data)
           'FLOAT'  : data = Float(data)
           'LONG64' : data = Long64(data)
           'LONG'   : data = Long(data)
           'INT'    : data = Fix(data)
           'ULONG64': data = ULong64(data)
           'ULONG'  : data = ULong(data)
           'UINT'   : data = UInt(Data) 
           'BYTE'   : data = Byte(data)
       ENDCASE
   ENDELSE

   ;; Apply clipping where necessary
   IF lcount GT 0L THEN data[low] = min_valid
   IF hcount GT 0L THEN data[high] = max_valid

   ;; If self keyword is set, store data back in object
Store_Back:
   IF argok EQ 0 THEN BEGIN
       Ptr_Free, self.data
       self.tname = newtype
       self.data = Ptr_New(data, /No_Copy)
   ENDIF

END


; -------------------------------------------------------------------------------------
; ExtractHyperslice:
;   This method returns an N-1 dimensional subarray of the data
; argument which is extracted by fixing the given dimension to the
; given index value. If no data argument is provided, the hyperslice is
; taken from the object's data.
;
;   You can use the Extra keywords to specify filter criteria if you
; are extracting the slice from the object's data (see the Filter
; method). Be careful to use the substitute keyword in this case in
; order to preserve the shape of your data.
;
; Example:
;    slice = object->ExtractHyperslice(dim=3,index=2,result=result)
;    IF result NE 'OK' THEN print,'No valid data in slice.'

function MGS_BaseVariable::ExtractHyperslice,  $
                      data,         $ ; The data array from which to extract a slice
                      dimension=dimension,    $ ; The dimension to keep fixed
                      index=index,            $ ; The index value for the fixed dimension
                      result=result,          $ ; The result status of the operation
                      _Ref_Extra=extra


   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   retval = self->GetDefaultValue(result=result)
   result = 'INVALID_SELECTION'

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error extracting hyperslice from variable '+self.name+'!'
      RETURN, retval
   ENDIF

   ;; Check if dimension keywords are present and valid 
   IF N_Elements(dimension) NE 1 THEN BEGIN
       self->ErrorMessage, 'Dimension keyword must be set to a scalar integer!'
       RETURN, retval
   ENDIF
   IF N_Elements(index) NE 1 THEN BEGIN
       self->ErrorMessage, 'Index keyword must be set to a scalar!'
       RETURN, retval
   ENDIF

   ;; Determine whether to use data argument or object's data:
   result = 'NO_VALID_DATA'
   argok = self->UseDataArgument(data, arg_present(data))
   IF argok EQ 0 THEN BEGIN
      data = self->GetData(result=result)
   ENDIF ELSE IF argok EQ 1 THEN BEGIN
      result = 'OK'
   ENDIF 
   
   ;; Return if no data passed or no data stored in object
   IF result NE 'OK' THEN BEGIN
      self->ErrorMessage, 'No data to convert!', /Warning
      RETURN, retval
   ENDIF

   ;; Get dimensions of data array 
   ;; (Programmer's note: should actually be able to use GetProperty here)
   ndims = Size(data, /N_Dimensions)
   dims  = Size(data, /Dimensions)

   IF ndims EQ 0 THEN BEGIN     ;; no valid data
      self->ErrorMessage, 'No valid data!'
      RETURN, retval
   ENDIF

   ;; Make sure dimension and index have correct type
   ;; Warning: dimension number is indexed beginning with 1 !!
   ;; The index for this dimension follows the IDL convention (start
   ;; with 0)
   thisdim = Fix(dimension)
   thisindex = Long(index)

   ;; Make sure, dimension and index are valid
   IF thisdim LT 1 OR thisdim GT ndims THEN BEGIN
       self->ErrorMessage, 'Invalid dimension ('+StrTrim(thisdim,2)+') !'
       result = 'INVALID_SELECTION'
       RETURN, retval
   ENDIF
   IF thisindex LT 0L OR thisindex GE dims[thisdim-1] THEN BEGIN
       self->ErrorMessage, 'Invalid index ('+StrTrim(thisindex,2)+') !'
       result = 'INVALID_SELECTION'
       RETURN, retval
   ENDIF   

   ;; OK, go ahead with the extraction
   CASE thisdim OF
       1 : retval = data[thisindex,*,*,*,*,*,*,*]
       2 : retval = data[*,thisindex,*,*,*,*,*,*]
       3 : retval = data[*,*,thisindex,*,*,*,*,*]
       4 : retval = data[*,*,*,thisindex,*,*,*,*]
       5 : retval = data[*,*,*,*,thisindex,*,*,*]
       6 : retval = data[*,*,*,*,*,thisindex,*,*]
       7 : retval = data[*,*,*,*,*,*,thisindex,*]
       8 : retval = data[*,*,*,*,*,*,*,thisindex]
   ENDCASE

   result = 'OK'

   RETURN, retval
end


; =====================================================================================
; Methods for data manipulation:
;     DeleteData   : free data pointer and reset dimension values

; -------------------------------------------------------------------------------------
; DeleteData:
;   This method frees the data pointer and optionally resets the
;   dimension values

pro MGS_BaseVariable::DeleteData,  $
                    reset_dimensions=reset_dimensions

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error while freeing basevariable data!'
      RETURN
   ENDIF

   IF Ptr_Valid(self.data) THEN Ptr_Free, self.data

   IF Keyword_Set(reset_dimensions) THEN BEGIN
      self.ndims = 0
      self.dims = 0L
   ENDIF

   ;; Don't touch dimvars for they may be hard to regenerate. Dimvars
   ;; can be deleted via the SetDimVar method.

end


; =====================================================================================
; Methods for data retrieval and dimension handling:
;     ContainsData  : return boolean flag if data pointer is valid
;     UseDataArgument (private) : checks validity of optional data argument
;     GetDefaultValue (private) : return dummy value for invalid data
;     IsNumeric     : returns 1 for numeric data types (also numeric
;                     rank)
;     IsFloat       : returns 1 for floating point data
;     IsString      : returns 1 for string data
;     IsStructure   : returns 1 for structure data
;     IsPointer     : returns 1 for pointer data (a bit unusual, but ...)
;     IsObject      : returns 1 for object data (a bit unusual, but ...)
;     GetFilterStringMask (private) : returns boolean array for filter
;     GetFilterNumMask (private) : dto.
;     Filter        : Apply filter criteria to data
;     GetData       : return data from the object
;     GetAttributes : return data attributes
;     GetDimVar     : return the values for a specific dimension
;     SetDimVar     : set the values for a specific dimension

; -----------------------------------------------------------------------------
; ContainsData:
;   This method returns 1 if the variable object's data pointer is
; valid and 0 otherwise.

FUNCTION MGS_BaseVariable::ContainsData

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   RETURN, Ptr_Valid(self.data)

END


; -----------------------------------------------------------------------------
; UseDataArgument:
;   This method returns 1 if the data argument is present and contains
; information indicating that this argument should be used in the
; calling method. A zero is returned if no data argument is present
; (i.e. the object's internal data should be used), and -1 is returned
; to signal an error if a data argument is present but undefined.

FUNCTION MGS_BaseVariable::UseDataArgument, data, argpresent

   IF N_Elements(data) EQ 0 AND NOT argpresent THEN BEGIN
      retval = 0
   ENDIF ELSE BEGIN
      retval = 1
      IF N_Elements(data) EQ 0 THEN retval = -1
   ENDELSE

   RETURN, retval

END


; -----------------------------------------------------------------------------
; GetDefaultValue:  (private)
;    This function returns a standard value for invalid data. This
; value is 0 for all numerical types and '' for strings. The result
; keyword will be set to 'NO_VALID_DATA'. 

FUNCTION MGS_BaseVariable::GetDefaultValue,  $
                    Result=result

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   ;; Set default for data as a fail-save return value
   IF self->IsString(result=result) THEN BEGIN
      retval = ''
   ENDIF ELSE IF self->IsNumeric(rank=rank) THEN BEGIN
      retval = 0
      self->ConvertType, data, rank=rank
   ENDIF ELSE BEGIN
      retval = Ptr_New()
      result = 'NO_VALID_DATA'
   ENDELSE

   RETURN, retval

END


; -----------------------------------------------------------------------------
; GetDataType:
;    This function returns the type name of the data argument, or, if
; no argument is provided, the type of the object data. The result
; keyword can be either 'OK' or 'NO_VALID_DATA'. 

FUNCTION MGS_BaseVariable::GetDataType, data,  $
                    Result=result


   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   thetype = 'UNDEFINED'
   result = 'NO_VALID_DATA'

   ;; Determine whether to use data argument or object's data:
   argok = self->UseDataArgument(data, arg_present(data))

   ;; Determine type
   IF argok EQ 0 THEN BEGIN
      data = self->GetData(result=result, /AsPointer)
      IF result EQ 'OK' THEN thetype = size(*data, /TName)
   ENDIF ELSE IF argok EQ 1 THEN BEGIN
      thetype = size(data, /TName)
      result = 'OK'
   ENDIF 
   
   RETURN, thetype

END


; -----------------------------------------------------------------------------
; IsNumeric:
;    This function returns 1 if its argument is of numeric type and 0
; otherwise. If no data argument is supplied, the test will be
; performed on the object's data. Furthermore, a rank value can be
; returned which is used e.g. to decide if the data needs to be
; converted to a different type for mathematical operations. The value
; of result will be 'OK' if the data type could be successfully
; determined and 'NO_VALID_DATA' otherwise.
; Notes: 
; (1) The isfloat flag will also be set for double precision data 
; (2) Complex data currently not recognized as numeric.

FUNCTION MGS_BaseVariable::IsNumeric, $
                         data, $
                         rank=rank, $
                         isfloatingpoint=isfloat, $
                         isdoubleprecision=isdouble, $
                         result=result

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   ;; defaults 
   retval = 0   
   result = 'NO_VALID_DATA'
   rank = -1
   
   ;; Get type of data argument or object data
   thetype = self->GetDataType(data, result=result)
   
   ;; Return if no data passed or no data stored in object
   IF result NE 'OK' THEN BEGIN
      RETURN, retval
   ENDIF

   ;; Determine rank of data type
   MaxRank = 9
   isfloat = 0
   isdouble = 0
   CASE thetype OF
       'DOUBLE' : BEGIN
                    retval = 1
                    rank = MaxRank
                    isfloat = 1
                    isdouble = 1
                 END
       'FLOAT'  : BEGIN
                    retval = 1
                    rank = MaxRank-1
                    isfloat = 1
                 END
       'LONG64' : BEGIN
                    retval = 1
                    rank = MaxRank-2
                 END
       'LONG'   : BEGIN
                    retval = 1
                    rank = MaxRank-3
                 END
       'INT'    : BEGIN
                    retval = 1
                    rank = MaxRank-4
                 END
       'ULONG64' : BEGIN
                    retval = 1
                    rank = MaxRank-5
                 END
       'ULONG'  : BEGIN
                    retval = 1
                    rank = MaxRank-6
                 END
       'UINT'   : BEGIN
                    retval = 1
                    rank = MaxRank-7
                 END
       'BYTE'   : BEGIN
                    retval = 1
                    rank = MaxRank-8
                 END
       
       ELSE :     BEGIN
                 END

   ENDCASE

   result = 'OK'

   RETURN, retval 
END


; -----------------------------------------------------------------------------
; IsFloat:
;    This function returns 1 if its argument is a floating point
; variable. If no data argument is supplied, the test will be
; performed on the object's data. Use the doubleprecision keyword to
; determine if the data are double or single precision.
;
; Programmer's note: The same functionality is also implemented in the
; IsNumeric method. However, an independent IsFLoat test makes code
; more readable.

FUNCTION MGS_BaseVariable::IsFloat, $
                         data, $
                         doubleprecision=isdouble, $
                         result=result

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   ;; defaults 
   retval = 0   
   isdouble = 0
   result = 'NO_VALID_DATA'
   
   ;; Get type of data argument or object data
   thetype = self->GetDataType(data, result=result)
   
   ;; Return if no data passed or no data stored in object
   IF result NE 'OK' THEN BEGIN
      RETURN, retval
   ENDIF

   CASE thetype OF
       'DOUBLE' : BEGIN
                    retval = 1
                    isdouble = 1
                 END
       'FLOAT'  : BEGIN
                    retval = 1
                    isdouble = 0
                 END
       ELSE :    BEGIN
                 END

   ENDCASE

   result = 'OK'

   RETURN, retval 
END


; -----------------------------------------------------------------------------
; IsString:
;    This function returns 1 if its argument is of type string and 0
; otherwise. If no data argument is supplied, the test will be
; performed on the object's data. The value of result will be 'OK' if
; the data type could be successfully determined and 'NO_VALID_DATA'
; otherwise. 

FUNCTION MGS_BaseVariable::IsString, data, result=result

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   ;; defaults 
   retval = 0   
   result = 'NO_VALID_DATA'
   
   ;; Get type of data argument or object data
   thetype = self->GetDataType(data, result=result)
   
   ;; Return if no data passed or no data stored in object
   IF result NE 'OK' THEN BEGIN
      RETURN, retval
   ENDIF

   IF thetype EQ 'STRING' THEN retval = 1

   result = 'OK'

   RETURN, retval
END


; -----------------------------------------------------------------------------
; IsStructure:
;    This function returns 1 if its argument is of type structure and 0
; otherwise. If no data argument is supplied, the test will be
; performed on the object's data. The value of result will be 'OK' if
; the data type could be successfully determined and 'NO_VALID_DATA'
; otherwise. 
;    If the Tags keyword is used (containing a string or string array
; with tag names to be tested for), the result will be a boolean array
; with 1 for each tag that is present and 0 for each tag that is
; missing. If data is not of type structure and tags is used, the
; value of result will be 'NO_STRUCTURE'.

FUNCTION MGS_BaseVariable::IsStructure, data, tags=tags, result=result

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   ;; defaults 
   retval = 0   
   result = 'NO_VALID_DATA'
   
   ;; Get type of data argument or object data
   thetype = self->GetDataType(data, result=result)
   
   ;; Return if no data passed or no data stored in object
   IF result NE 'OK' THEN BEGIN
      RETURN, retval
   ENDIF

   IF thetype EQ 'STRUCTURE' THEN retval = 1

   ;; Check for tag names
   IF N_Elements(tags) GT 0 THEN BEGIN
      ;; Return if data was not a structure
      IF retval EQ 0 THEN BEGIN
         result = 'NO_STRUCTURE'
         RETURN, retval
      ENDIF
      ;; Get tag names of structure and reformat test tags to
      ;; uppercase with no blanks
      datatags = Tag_Names(data[0])
      thetags = StrUpCase(StrCompress(tags, /Remove_All))

      retval = intarr(N_Elements(thetags))
      FOR i=0L, N_Elements(thetags)-1 DO $
        retval[i] = ( Where(datatags EQ thetags[i]) )[0] GE 0

   ENDIF

   result = 'OK'

   RETURN, retval
END


; -----------------------------------------------------------------------------
; IsPointer:
;    This function returns 1 if its argument is of type pointer and 0
; otherwise. If no data argument is supplied, the test will be
; performed on the object's data. The value of result will be 'OK' if
; the data type could be successfully determined and 'NO_VALID_DATA'
; otherwise. 

FUNCTION MGS_BaseVariable::IsPointer, data, result=result

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   ;; defaults 
   retval = 0   
   result = 'NO_VALID_DATA'
   
   ;; Get type of data argument or object data
   thetype = self->GetDataType(data, result=result)
   
   ;; Return if no data passed or no data stored in object
   IF result NE 'OK' THEN BEGIN
      RETURN, retval
   ENDIF

   IF thetype EQ 'POINTER' THEN retval = 1

   result = 'OK'

   RETURN, retval
END


; -----------------------------------------------------------------------------
; IsObject:
;    This function returns 1 if its argument is of type object and 0
; otherwise. If no data argument is supplied, the test will be
; performed on the object's data. The value of result will be 'OK' if
; the data type could be successfully determined and 'NO_VALID_DATA'
; otherwise. 

FUNCTION MGS_BaseVariable::IsObject, data, result=result

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   ;; defaults 
   retval = 0   
   result = 'NO_VALID_DATA'
   
   ;; Get type of data argument or object data
   thetype = self->GetDataType(data, result=result)
   
   ;; Return if no data passed or no data stored in object
   IF result NE 'OK' THEN BEGIN
      RETURN, retval
   ENDIF

   IF thetype EQ 'OBJECT' THEN retval = 1

   result = 'OK'

   RETURN, retval
END


; -----------------------------------------------------------------------------
; GetFilterStringMask: (private)
;   This method constructs a boolean array containing 1 for each data
; element that fulfills the combined filter criteria non_empty or
; non_blank. This method is called from Filter, and it is here where
; you should make changes to override the default filter behaviour or
; add new filters.  
;
;    No test is made if data is a valid string type argument. This
; must be done in the calling routine. The Trim keyword is passed for
; convenience as GetFilterStringMask needs to perform a StrTrim
; operation anyway when looking for non_blank strings. With the trim
; keyword set, the results for /non_empty and /non_blank are
; identical.
;
;   Side effects: If the Trim keyword is set, the data array will be
; changed. 

FUNCTION MGS_BaseVariable::GetFilterStringMask,  $
                   data=data,               $ ; The data to be analyzed
                   non_empty=non_empty,     $ ; Return only non empty strings
                   non_blank=non_blank,     $ ; Return only strings with at least
                   trim=trim,               $ ; Return data in trimmed form
                   _Extra=extra


   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   retval = bytarr(N_Elements(data)) + 1B

   IF Keyword_Set(non_blank) OR Keyword_Set(trim) THEN BEGIN
      tmpdata = StrTrim(data, 2)
      wblank = Where(tmpdata EQ '', cbl)
      IF cbl GT 0 THEN retval[wblank] = 0B
      IF Keyword_Set(trim) THEN data = tmpdata
   ENDIF ELSE IF Keyword_Set(non_empty) THEN BEGIN
      wempty = Where(data EQ '', cem)
      IF cem GT 0 THEN retval[wempty] = 0B
   ENDIF

   RETURN, retval

END


; -----------------------------------------------------------------------------
; GetFilterNumMask: (private)
;   This method constructs a boolean array containing 1 for each data
; element that fulfills the combined filter criteria for numeric
; data. These are finite_only, positive, negative, or
; non_zero. If data is a floating point variable, the test for
; non_zero will be done using the machine specific eps value (see
; IDL's MaChar function). 
;
;    No test is made if data is a valid numeric type argument. This
; must be done in the calling routine. 
;
;   Objects derived from MGS_Basevariable can overwrite this method to
; enhance filtering capabilities. 
;
; Note: It is the IDL definition that infinite values will return false
; for both positive and negative comparisons. This leads to undesired
; effects in this routine since the filters in this object are using
; the inverse filter operation to allow efficient filter
; combination. You must explicitely specify /finite_only if you use
; one of the other numerical filters and you may have infinite data
; values. If you use the /Invert keyword with the Filter method, you
; must call Filter a second time with /finite_only to ensure that the
; result is correct.  

FUNCTION MGS_BaseVariable::GetFilterNumMask,  $
                   data,                    $ ; The data to be analyzed
                   finite_only=finite_only, $ ; Return only valid numerical data
                   positive=positive,       $ ; Return only positive numerical data
                   negative=negative,       $ ; Return only negative numerical data
                   non_zero=non_zero,       $ ; Return only non-zero values
                   _Extra=extra


   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   retval = bytarr(N_Elements(data)) + 1B

   IF Keyword_Set(finite_only) THEN BEGIN
      winf = Where(Finite(data) EQ 0, cinf)
      IF cinf GT 0 THEN retval[winf] = 0B
   ENDIF

   IF Keyword_Set(positive) THEN BEGIN
      wneg = Where(data LE 0, cneg)
      IF cneg GT 0 THEN retval[wneg] = 0B
   ENDIF

   IF Keyword_Set(negative) THEN BEGIN
      wpos = Where(data GE 0, cpos)
      IF cpos GT 0 THEN retval[wpos] = 0B
   ENDIF

   IF Keyword_Set(non_zero) THEN BEGIN
      IF self->IsFloat(data, double=isdouble) THEN BEGIN
         eps = (MaChar(double=isdouble)).eps 
      ENDIF ELSE BEGIN
         eps = 0
      ENDELSE
      wzero = Where(ABS(data) LT eps, czero)
      IF czero GT 0 THEN retval[wzero] = 0B
   ENDIF

   RETURN, retval

END

   
; -----------------------------------------------------------------------------
; Filter:
;   This method returns an index vector to all data that satisfies
; object specific selection criteria. Without any filter options set,
; a vector from 0 to N_Elements(data) is returned. If no data are
; valid, the returned value will be -1L, and result will have the
; value 'NO_VALID_DATA'.  
;
;   Filter criteria are set via keywords and differ for string
; and numerical data. Other data types (e.g. structures) cannot be
; filtered. All filtering options are internally combined with the
; logical AND operator. If you set the mask keyword you can return the
; boolean filter mask (which contains 1 or 0 for each data element)
; instead of the array indices. This can be useful for further
; processing (e.g. in derived objects). With the Invert_Selection
; keyword, you can achieve an OR type filter.
;
; The valid filter options for string data are:
; - non_empty (boolean): returns an index to all strings that are not
;          empty (data NE '') 
; - non_blank (boolean): returns an index to all strings with at least
;          one non-blank character left after a StrTrim(...,2)
;          operation
; Optionally, you can return the filtered strings in a trimmed form by
; setting the trim keyword. Note that non_blank always includes
; non_empty by definition.
;
; The valid numerical filter options are:
; - finite_only (boolean): returns only valid numerical floating point
;          data. For integer type data, the complete index vector is
;          returned 
; - positive (boolean): returns only numbers that are greater zero
; - negative (boolean): returns only numbers less than zero
; - non_zero (boolean): returns only numbers that are discernible from
;          zero. The EPS value from the MACHAR function is used for
;          floating point data
;
; The result keyword may contain 'OK', or 'NO_VALID_DATA'.
;
; The data argument is optional. Per default, the filter method
; operates on the object's internal data array. You may find the data
; argument useful if, for example, you extracted a hyperslice (this
; must be done before filtering!) and you want to limit the result,
; say, to positive values.
;
; A named variable supplied with the bad_index keyword will return the
; complement index. (This uses the complement keyword for IDL versions
; >= 5.4).

FUNCTION MGS_BaseVariable::Filter,  $
                   data,                    $ ; The optional argument containing external 
                                              ; data to be filtered
                   result=result,           $ ; The result status of this operation
                   finite_only=finite_only, $ ; Return only valid numerical data
                   positive=positive,       $ ; Return only positive numerical data
                   negative=negative,       $ ; Return only negative numerical data
                   non_zero=non_zero,       $ ; Return only non-zero values
                   non_empty=non_empty,     $ ; Return only non empty strings
                   non_blank=non_blank,     $ ; Return only strings with at least
                                              ; one non-blank character 
                   trim=trim,               $ ; remove leading and trailing blanks
                   bad_index=bad,           $ ; returns the indices of bad data
                   invert_selection=invert_selection, $ ; applies the inverse filter
                   mask=return_mask,        $ ; Set this keyword to return the boolean 
                                ; filter mask instead of the filtered
                                ; data. This is useful if you want to
                                ; apply consecutive filters afterwards.   
                   _Ref_Extra=extra


   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   ;; Define default return values
   retval = -1L      ;; the "ok" index
   result = 'NO_VALID_DATA'
   bad = -1L
   mask = 0B
   IF Keyword_Set(return_mask) THEN retval = mask

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error filtering data!'
      RETURN, retval
   ENDIF

   ;; Determine whether to use data argument or object's data:
   argok = self->UseDataArgument(data, arg_present(data)) 
   IF argok EQ 0 THEN BEGIN
      data = self->GetData(result=result)
   ENDIF ELSE IF argok EQ 1 THEN BEGIN
      result = 'OK'
   ENDIF 

   ;; Return if no data passed or no data stored in object
   IF result NE 'OK' THEN BEGIN
      self->ErrorMessage, 'No data to convert!', /Warning
      RETURN, retval
   ENDIF

   ;; If data is string type or numeric type, determine boolean filter
   ;; mask, otherwise exit since no filtering is possible
   err = 0
   IF self->IsString(data) THEN BEGIN
      mask = self->GetFilterStringMask(data, $
                                       non_empty=non_empty, $
                                       non_blank=non_blank, $
                                       trim=trim, $
                                       _Extra=extra)
   ENDIF ELSE IF self->IsNumeric(data) THEN BEGIN
      ;; Check for previous 'floating illegal operand' errors
      err = Check_Math(mask=128)

      mask = self->GetFilterNumMask(data, $
                                    finite_only=finite_only, $
                                    positive=positive, $
                                    negative=negative, $
                                    non_zero=non_zero, $
                                    _Extra=extra)
      ;; Test math errors again
      err = Check_Math(mask=128)

   ENDIF ELSE BEGIN
      result = 'INVALID_TYPE'
      self->ErrorMessage, 'Filtering not possible for data type '+ $
         Size(data,/TName)
      RETURN, retval
   ENDELSE

   ;; Invert mask if desired
   IF Keyword_Set(invert_selection) THEN mask = 1B-mask

   ;; If a numeric filter was applied and a 'floating illegal operand'
   ;; error was detected, compute the /finite_only mask and AND
   ;; combine it with the result
   IF self->IsNumeric(data) AND err NE 0 THEN BEGIN
      mask2 = self->GetFilterNumMask(data, /finite_only)
      mask = mask * mask2
   ENDIF

   ;; Compose return values: If the boolean mask is requested, the
   ;; Where function needs only be called if the bad_index keyword is
   ;; a named variable.
   IF Keyword_Set(return_mask) AND NOT Arg_Present(bad) THEN BEGIN
      RETURN, mask
   ENDIF ELSE BEGIN
      ;; Version dependency: complement keyword still rather new
      IF !Version.Release GE 5.4 THEN BEGIN
;;         retval = Where( mask GT 0, Complement=bad )
      ENDIF ELSE BEGIN
         retval = Where( mask GT 0 )
         IF Arg_Present(bad) THEN bad = Where( mask EQ 0 )
      ENDELSE
   ENDELSE

   RETURN, retval

END


; -----------------------------------------------------------------------------
; GetData:
;   This function method returns the data that are stored in the
; object. The result of the operation can be tested with the result
; keyword which will return 'OK' or 'NO_VALID_DATA'. Extra keywords
; can be used to apply filter criteria to the data (the filters
; available in the basevariable object are /finite_only, /positive,
; and /negative for numeric data, and /non_empty for string data (see
; the Filter method above). These criteria can be extended in derived
; objects without changes to the GetData method.
;
;   If you apply a filter without supplying a substitute value, the
; shape of the data will not be preserved, and the GetData method will
; return a 1D vector (as always after a Where operation). If a
; substitute value is given, the "bad" data will be replaced with that
; value (this works for both strings and numbers). The indices for
; "good" and "bad" data are then returned in the ok_index and
; bad_index keywords. Without a substitute value, ok_index and
; bad_index will always be -1L for reasons of efficiency.
;
;   Programmer's Note: No test is done if substitute is a scalar value
; as expected by this program a priori. This fact can be exploited to
; replace portions of data with new values. In this case, the number
; of "bad" data elements must be known beforehand, and substitute must
; contain the same number of elements; otherwise, IDL reports an
; error.
;
;   Experienced programmers may use the AsPointer keyword to retrieve
; a pointer reference to the (unfiltered!) data rather than a physical
; copy. This is a clear violation of the OOP principle "encapsulation"
; and should therefore only be used with great caution and if the
; desired effect cannot be achieved otherwise. It will normally be
; better to improve this object's functionality by adding or modifying
; its methods or by creating a new derived object with the desired
; functionality. The pointer should always be regarded as 'read only'
; and it should not be freed by the external routine.

FUNCTION MGS_BaseVariable::GetData,  $
                   Result=result,           $ ; The result status of the operation
                   substitute=substitute,   $ ; A replacement value for invalid data
                   ok_index=ok,             $ ; Index vector for valid data
                   bad_index=bad,           $ ; Index vector for invalid data
                   AsPointer=aspointer,     $ ; Return data pointer instead of values
                   _Ref_Extra=extra

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   ;; Shall we return pointer to data?
   Returnpointer = Keyword_Set(aspointer)

   ;; Set default for data as a fail-save return value
   IF Returnpointer THEN BEGIN
      retval = Ptr_New() 
      result = 'NO_VALID_DATA'
   ENDIF ELSE BEGIN
      retval = self->GetDefaultValue(result=result)
   ENDELSE

   ok = -1L
   bad = -1L

   ;; Error handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error retrieving data!'
      RETURN, retval
   ENDIF

   ;; If no data are stored, that's all
   IF NOT self->ContainsData() THEN RETURN, retval

   ;; Return a pointer to the object's data if requested, otherwise
   ;; return copy of actual data and pass it through filter if any
   ;; extra keywords are set.
   IF Returnpointer THEN BEGIN
      retval = self.data
      IF Ptr_Valid(retval) THEN result = 'OK'
   ENDIF ELSE IF N_Elements(extra) GT 0 THEN BEGIN
      IF N_Elements(substitute) GT 0 THEN BEGIN
         retval = *self.data
         ok = self->Filter(_Extra=extra, bad_index=bad, result=result)
         IF bad[0] GE 0 THEN retval[bad] = substitute
      ENDIF ELSE BEGIN
         ok = self->Filter(_Extra=extra, bad_index=bad, result=result)
         IF result EQ 'OK' THEN retval = (*self.data)[ok]
         ok = -1L
      ENDELSE
   ENDIF ELSE BEGIN
      retval = *self.data
      result = 'OK'
   ENDELSE

   RETURN, retval

END


; -----------------------------------------------------------------------------
; GetAttributes:
;   This method returns the data attributes. The result will wither be
; 'OK' or 'NO_ATTRIBUTES'. In the latter case, a dummy structure is
; returned.

FUNCTION MGS_BaseVariable::GetAttributes, $
                         result=result

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   IF Ptr_Valid(self.attributes) THEN BEGIN
      retval = *self.attributes
      result = 'OK'
   ENDIF ELSE BEGIN
      retval = { dummy : 0L }
      result = 'NO_ATTRIBUTES'
   ENDELSE

   RETURN, retval

END


; -------------------------------------------------------------------------------------
; GetDimNames:
;   This method attempts to extract the names of the dimensions for
; this variable. If you provide the optional index argument, only the
; name of that dimension is returned (note that dimension indices
; start with 1 instead of zero!), otherwise a vector with up to 8
; string elements is returned depending on the number of dimensions
; defined for this variable.
;
;   GetDimNames first looks for the type of each dimvar; if it is an
; (variable type) object, it will query the object's name property. If
; dimvar[i] is a string, that string will be returned. All other data
; types will cause an error message.

FUNCTION MGS_BaseVariable::GetDimNames, index, result=result

   retval = ''
   result = 'NO_DIMENSION_INFORMATION'

   IF N_Elements(index) EQ 0 THEN BEGIN
      theindex = indgen(8)
   ENDIF ELSE BEGIN
      IF index[0] LT 1 OR index[0] GT 8 THEN BEGIN
         self->ErrorMessage, 'Index must be in range 1..8'
         RETURN, retval
      ENDIF
      theindex = index[0]-1
   ENDELSE

   ;; Loop through indices and determine if dimvars[i] is (1) valid,
   ;; (2) an object reference, or (3) a string
   retval = StrArr(N_Elements(theindex))
   FOR i=0,N_Elements(theindex)-1 DO BEGIN
      IF Ptr_Valid(self.dimvars[theindex[i]]) THEN BEGIN
         IF Obj_Valid(*self.dimvars[theindex[i]]) THEN BEGIN
            *self.dimvars[theindex[i]]->GetProperty, name=thisname
            retval[i] = thisname
         ENDIF ELSE IF Size(*self.dimvars[theindex[i]],/TName) EQ 'STRING' THEN BEGIN
            retval[i] = *self.dimvars[theindex[i]]
         ENDIF ELSE BEGIN
            self->ErrorMessage, 'Cannot determine name of dimension '+ $
               StrTrim(theindex[i],2)
            RETURN, ''
         ENDELSE
      ENDIF
   ENDFOR

   result = 'OK'

   RETURN, retval
END


; -------------------------------------------------------------------------------------
; GetDimVar:
;   This method retrieves the values of a specific dimension. A more
; detailed explanation is given for the corresponding SetDimVar method
; below.
 

PRO MGS_BaseVariable::GetDimVar,  $
                   dimid,                $ ; The dimension number
                   dimvals                 ; The dimension values

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error retrieving dimension information!'
      RETURN
   ENDIF

   thisDim = ( dimid > 1 ) < 8     ; limit range
   
   ;; If dimension values are stored get them, otherwise make sure to
   ;; return an undefined variable
   IF Ptr_Valid( self.dimvars[thisDim-1] ) THEN BEGIN
       dimvals = *(self.dimvars[thisDim-1])
   ENDIF ELSE BEGIN
       self->Undefine, dimvals
   ENDELSE

END

; -------------------------------------------------------------------------------------
; SetDimVar:
;   This method stores a vector (or an array) with dimension values into
; the object. If the dimvals parameter is undefined, the values of the
; respective dimension are deleted from the object.
;   This method allows access to individual dimension variables and is
; called from the Init and SetProperty methods. Note that SetDimVar
; accepts a numerical argument to indicate the dimension number, whereas
; the SetProperty method uses named keywords. 
;   Note: Dimension numbers < 1 or > 8 are silently clipped.
; Example:
;   To store longitude values into the first dimension and latitude
;   values into the second dimension, use:
; theVariable->SetDimVar, 1, lon
; theVariable->SetDimVar, 2, lat
;   The same effect could be achieved via:
; theVariable->SetProperty, dim1=lon, dim2=lat


PRO MGS_BaseVariable::SetDimVar,  $
                   dimid,                $ ; The dimension number
                   dimvals                 ; The dimension values

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error storing dimension information!'
      RETURN
   ENDIF

   thisDim = ( dimid > 1 ) < 8  ; limit range

   IF Ptr_Valid( (self.dimvars)[thisDim-1] ) THEN Ptr_Free, (self.dimvars)[thisDim-1]

   IF N_Elements(dimvals) GT 0 THEN BEGIN
       self.dimvars[thisDim-1] = Ptr_New(dimvals)
print,'### Stored new values in dim ',dimid,' type=',size(dimvals,/Tname)
   ENDIF ELSE BEGIN
       self.dimvars[thisDim-1] = Ptr_New()
   ENDELSE

END

; =====================================================================================
; Standard object methods:
;     Validate    : ensure consistency between data and type and
;                   dimension values
;     GetProperty : retrieve object values
;     SetProperty : set object values
;     Cleanup     : free object pointers and destroy
;     Init        : initialize object

; -------------------------------------------------------------------------------------
; Validate:
;    This method ensures consistency between the data description
; (type name, number of dimensions and dimension sizes) and the actual
; data. If no data are loaded, the result of Validate will always be
; 'NO_VALID_DATA'. Otherwise, the result may indicate that dimensions
; or data type were changed.

FUNCTION MGS_BaseVariable::Validate, _Extra=e

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   status = 'UNKNOWN_STATUS'

   IF self->ContainsData() THEN BEGIN
      ;; Check data type
      thetype = Size(*self.data, /TName)
      statusword = 0
      IF thetype NE self.tname THEN BEGIN
         self.tname = thetype
         statusword = 1
      ENDIF
      ;; Check data dimensions
      ;; Note: IDL returns 0 for N_Dimensions of a scalar. To
      ;; distinguish between a scalar and an undefined value, I prefer
      ;; to set N_Dimensions to 1 and dim[0] to 0 in this case. 
      thendims = Size(*self.data, /N_Dimensions) > 1
      IF thendims NE self.ndims THEN BEGIN
         self.ndims = thendims
         statusword = statusword OR 2
      ENDIF
      IF self.ndims GT 0 THEN BEGIN
         thedims = Size(*self.data, /Dimensions)
         FOR i = 0, self.ndims-1 DO BEGIN
            IF thedims[i] NE self.dims[i] THEN BEGIN
               self.dims[i] = thedims[i]
               statusword = statusword OR 2
            ENDIF
         ENDFOR
      ENDIF ELSE BEGIN
         FOR i = 0, 7 DO self.dims = 0
         statusword = statusword OR 2
      ENDELSE
      
      CASE statusword OF
         0 : status = 'OK'
         1 : status = 'DATA_TYPE_CHANGED'
         2 : status = 'DIMENSIONS_CHANGED'
         3 : status = 'TYPE_AND_DIMENSIONS_CHANGED'
         ELSE : status = 'UNKNOWN_STATUS'
      ENDCASE

   ENDIF ELSE BEGIN
      status = 'NO_VALID_DATA'
   ENDELSE

   RETURN, status

END


; -------------------------------------------------------------------------------------
; GetProperty:
; This method extracts specific object values and returns them to the
; user. Derived objects should overwrite and extend this method to
; return the extra information stored in them.

PRO MGS_BaseVariable::GetProperty,  $
                   tname=tname,          $ ; Data type name
                   data=data,            $ ; The data values (use the GetData function!)
                   ndims=ndims,          $ ; The number of variable dimensions
                   dims=dims,            $ ; The size of each dimension
                   dimvars=dimvars,      $ ; Links to the dimension variables
                   dimnames=dimnames,    $ ; Names of the dimensions
                   dim1=dim1,            $ ; The values of the first dimension
                   dim2=dim2,            $ ; The values of the second dimension
                   dim3=dim3,            $ ; The values of the third dimension
                   dim4=dim4,            $ ; ...
                   dim5=dim5,            $ 
                   dim6=dim6,            $ 
                   dim7=dim7,            $ 
                   dim8=dim8,            $ 
                   attributes=attributes,$ ; The data attributes
                   _Ref_Extra=extra
                                ; Inherited keywords:
                                ; name      : The variable name
                                ; uvalue    : a user-defined value
                                ; no_copy   : Copy pointer values?
                                ; no_dialog : Object uses dialogs for messages?
                                ; debug     : Object is in debug mode?
                                ; status    : Object valid? (string)


   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   ;; Call GetProperty method from parent object
   ;; Note that execution continues even if retrieval of a baseobject
   ;; property fails.
   self->MGS_BaseObject::GetProperty, _Extra=extra

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error retrieving basevariable properties!'
      RETURN
   ENDIF

   tname = self.tname
   ndims = self.ndims
   dims = self.dims[0:ndims-1]
   dimvars = self.dimvars
   
   ;; Handle pointer values
   IF Arg_Present(data) THEN BEGIN
       IF Ptr_Valid(self.data) THEN $
          data = *self.data  $
       ELSE  $
          self->undefine, data
   ENDIF

   ;; Handle requests for dimension values
   IF Arg_Present(dim1) THEN self->GetDimVar, 1, dim1
   IF Arg_Present(dim2) THEN self->GetDimVar, 2, dim2
   IF Arg_Present(dim3) THEN self->GetDimVar, 3, dim3
   IF Arg_Present(dim4) THEN self->GetDimVar, 4, dim4
   IF Arg_Present(dim5) THEN self->GetDimVar, 5, dim5
   IF Arg_Present(dim6) THEN self->GetDimVar, 6, dim6
   IF Arg_Present(dim7) THEN self->GetDimVar, 7, dim7
   IF Arg_Present(dim8) THEN self->GetDimVar, 8, dim8

   IF Arg_Present(dimnames) THEN dimnames = self->GetDimnames()

   ;; Return data attributes
   IF Arg_Present(attributes) THEN BEGIN
      IF Ptr_Valid(self.attributes) THEN $
        attributes = *self.attributes  $
      ELSE  $
        self->Undefine, atributes
   ENDIF

END

; -------------------------------------------------------------------------------------
; SetProperty:
;    This method sets specific object values. Derived objects should
; overwrite and extend this method to allow storing additional
; information.
;
;    You may use the result keyword to determine the status of the
; object. Possible values are:
; - OK
; - CANCELLED  : this signals an error condiditon and is usually
;                accompanied by a warning message
; - NO_VALID_DATA : the object does not contain any data
; - DATA_TYPE_CHANGED : the tname attribute changed as a result of
;                storing new data
; - DIMENSIONS_CHANGED : the ndims or at least one dim attribute
;                changed as a result of storing new data
; - TYPE_AND_DIMENSIONS_CHANGED : both changed
; - UNKNOWN : This shouldn't occur, please contact the author if you
;                encounter this status value.

PRO MGS_BaseVariable::SetProperty,  $
                   data=data,            $ ; The data values
                   dims=dims,            $ ; The size of each dimension
                   dim1=dim1,            $ ; The values of the first dimension
                   dim2=dim2,            $ ; The values of the second dimension
                   dim3=dim3,            $ ; The values of the third dimension
                   dim4=dim4,            $ ; ...
                   dim5=dim5,            $ 
                   dim6=dim6,            $ 
                   dim7=dim7,            $ 
                   dim8=dim8,            $ 
                   attributes=attributes,$  ; data attributes
                   result=result,        $  ; result of Validate
                   _Extra=extra
                                ; Inherited keywords:
                                ; name      : The variable name
                                ; uvalue    : a user-defined value
                                ; no_copy   : don't copy data when
                                ;             creating pointers
                                ; no_dialog : Don't display
                                ;             interactive dialogs
                                ; debug     : put object in debug mode

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   ;; Call SetProperty method of BaseObject
   self->MGS_BaseObject::SetProperty, _Extra=extra

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error setting basevariable properties!'
      result = 'CANCELLED'
      RETURN
   ENDIF

   ;; If data given, determine data type and dimensions
   ;; Otherwise set default dimension values but only if no data is
   ;; loaded 
   IF N_Elements(data) GT 0 THEN BEGIN
      IF Ptr_Valid(self.data) THEN Ptr_Free, self.data
      self.data = Ptr_New( data, no_copy=self.no_copy )
   ENDIF ELSE BEGIN
      IF Ptr_Valid(self.data) THEN BEGIN
         theMessage = ['Cannot change type or dimensions of data!', $
                       'You must call the DeleteData method before doing this.']
         IF N_Elements(tname) GT 0 THEN BEGIN
            result = 'CANCELLED'
            self->ErrorMessage, theMessage
            RETURN
         ENDIF
         IF N_Elements(dims) GT 0 THEN BEGIN
            result = 'CANCELLED'
            self->ErrorMessage, theMessage
            RETURN
         ENDIF
      ENDIF ELSE BEGIN
         IF N_Elements(tname) GT 0 THEN self.tname = StrUpCase(tname)
         IF N_Elements(dims) GT 0 THEN BEGIN
            self.ndims = N_Elements(dims) < 8
            self.dims[0:self.ndims-1] = dims[0:self.ndims-1]
         ENDIF
      ENDELSE
   ENDELSE

   ;; analyze dimension variables
   ;; no consistency check is made whether dimensions of dimvars agree
   ;; with data dimensions. This is done to allow storage of
   ;; irregularily gridded data where X and Y may be 2D arrays.
   IF N_Elements(dim1) GT 0 THEN self->SetDimVar, 1, dim1
   IF N_Elements(dim2) GT 0 THEN self->SetDimVar, 2, dim2
   IF N_Elements(dim3) GT 0 THEN self->SetDimVar, 3, dim3
   IF N_Elements(dim4) GT 0 THEN self->SetDimVar, 4, dim4
   IF N_Elements(dim5) GT 0 THEN self->SetDimVar, 5, dim5
   IF N_Elements(dim6) GT 0 THEN self->SetDimVar, 6, dim6
   IF N_Elements(dim7) GT 0 THEN self->SetDimVar, 7, dim7
   IF N_Elements(dim8) GT 0 THEN self->SetDimVar, 8, dim8

   IF N_Elements(attributes) GT 0 THEN BEGIN
      IF Ptr_Valid(self.attributes) THEN Ptr_Free, self.attributes
      self.attributes = Ptr_New( atributes, no_copy=self.no_copy )
   ENDIF

   ;; make sure everything is ok
   result = self->Validate()

END

; -------------------------------------------------------------------------------------
; Cleanup:
; This method frees all data stored in the object. If dimvars are
; objects pointing to other variables, these will be left intact.

PRO MGS_BaseVariable::Cleanup

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   IF Ptr_Valid(self.data) THEN Ptr_Free, self.data
   FOR i=0,7 DO BEGIN
      IF NOT Obj_Valid(self.dimvars[i]) THEN BEGIN
         IF Ptr_Valid(self.dimvars[i]) THEN Ptr_Free, self.dimvars[i] 
      ENDIF
   ENDFOR

   IF Ptr_Valid(self.attributes) THEN Ptr_Free, self.attributes

   ;; Call parent's cleanup method
   self->MGS_BaseObject::Cleanup

END

; -------------------------------------------------------------------------------------
; Init:
; This method initializes the object values.
; Note that dim1-dim8 can hold any data type. Thus you can either
; specify values directly, or you can pass object references to other
; variable objects which contain the dimensional information. 
; Example:
;   lon=obj_new('mgs_basevariable', findgen(360), name='LON')
;   lat=obj_new('mgs_basevariable', findgen(180)-90., name='LAT')
;   var=obj_new('mgs_basevariable', data, name='Data', dim1=lon, dim2=lat)

FUNCTION MGS_BaseVariable::Init,  $
                   data,                 $ ; The data values (type will be checked)
                   dim1,                 $ ; Values for first dimension (X)
                   dim2,                 $ ; Values for second dimension (Y)
                   dim3,                 $ ; Values for third dimension (Z or T)
                   dim4,                 $ ; ...
                   dim5,                 $ 
                   dim6,                 $ 
                   dim7,                 $ 
                   dim8,                 $ 
                   dims=dims,            $ ; The size of each dimension
                   attributes=attributes,$ ; Data attributes
                   _Extra=extra        ; For future additions
                                ; Inherited keywords:
                                ; name      : The variable name
                                ; uvalue    : a user-defined value
                                ; no_dialog : Don't display message dialogs
                                ; debug     : Put object in debugging state


   ;; Initialize parent object first
   IF not self->MGS_BaseObject::Init(_Extra=extra) THEN RETURN, 0

   IF self.debug GT 1 THEN print,'## '+Routine_Name()+' in '+self.name

   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      self->ErrorMessage, 'Error initializing basevariable object!'
      RETURN, 0
   ENDIF

   ;; If data given, determine data type and dimensions
;;; *** NOTE: It may be useful to handle data pointers differently by
;;; dereferencing them here. This could save some memory
;;; transfers. Must make sure that data is a scalar pointer! 
;;; Need further thought/discussion about this. *** 
   IF N_Elements(data) GT 0 THEN BEGIN
       tname = Size(data, /TName)
       ndims = Size(data, /N_Dimensions) > 1
       dims  = Size(data, /Dimensions)
       self.data = Ptr_New( data, no_copy=self.no_copy )
   ENDIF ELSE BEGIN
   ;; otherwise set default dimension values
       IF N_Elements(tname) EQ 0 THEN tname = 'NOTHING'
       IF N_Elements(dims) EQ 0 THEN dims = 0L
       ndims = N_Elements(dims) < 8
       self.data = Ptr_New()
   ENDELSE

   ;; populate object (data was done above)
   self.tname = StrUpCase(tname)
   self.ndims = ndims
   self.dims = lonarr(8)
   self.dims[0:ndims-1] = dims

   ;; analyze dimension variables
   ;; no consistency check is made whether dimensions of dimvars agree
   ;; with data dimensions. This is done to allow storage of
   ;; irregularily gridded data where X and Y may be 2D arrays.
   self.dimvars = PtrArr(8)
; WHY did I do the following?????
;   IF N_Elements(dim1) GT 0 THEN self.dimvars[0] = Ptr_New(dim1)
;   IF N_Elements(dim2) GT 0 THEN self.dimvars[1] = Ptr_New(dim2)
   IF N_Elements(dim1) GT 0 THEN self->SetDimVar, 1, dim1
   IF N_Elements(dim2) GT 0 THEN self->SetDimVar, 2, dim2
   IF N_Elements(dim3) GT 0 THEN self->SetDimVar, 3, dim3
   IF N_Elements(dim4) GT 0 THEN self->SetDimVar, 4, dim4
   IF N_Elements(dim5) GT 0 THEN self->SetDimVar, 5, dim5
   IF N_Elements(dim6) GT 0 THEN self->SetDimVar, 6, dim6
   IF N_Elements(dim7) GT 0 THEN self->SetDimVar, 7, dim7
   IF N_Elements(dim8) GT 0 THEN self->SetDimVar, 8, dim8

   ;; Store data attributes
   IF N_Elements(attributes) GT 0 THEN BEGIN
      self.attributes = Ptr_New(attributes, no_copy=self.no_copy)
   ENDIF ELSE BEGIN
      self.attributes = Ptr_New()
   ENDELSE

   RETURN, 1
END


; -------------------------------------------------------------------------------------
; This is the object definition. Derived objects should create a new
; structure and append these fields via the INHERITS syntax.

PRO MGS_BaseVariable__Define

   objectClass = { MGS_BaseVariable,  $    ; The object class
                   data          : ptr_new(),  $ ; The data values
                   tname         : '',         $ ; The data type
                   ndims         : 0,          $ ; The number of data dimensions
                   dims          : lonarr(8),  $ ; The size of each dimension
                   dimvars       : ptrarr(8),  $ ; The dimension values
                   attributes    : Ptr_New(),  $ ; The data attributes
                   INHERITS MGS_BaseObject  $
                 }
                   
END


; -------------------------------------------------------------------------------------
; Example: 
;    This example demonstrates a few features of the BaseVariable
; object, notably how to make a copy and how to retrieve data from it.

PRO Example

   test = obj_new('MGS_BaseVariable', indgen(4,4)-5000, name='Test')
   test->ConvertType, /float
   print,test->GetData()
;   test->ConvertType, /byte
;   print,test->GetData()
;   test->ConvertType, /ul64
;   print,test->GetData()
   test->ConvertType, /l64
   print,test->GetData()
   test->ConvertType, /double
   print,test->GetData()
   thedata = test->GetData()
   thedata[12] = !Values.f_NaN
   test->SetProperty, Data=thedata
   print,'Double data with missing value'
   print,test->GetData()
   print,'Valid double data'
   print,test->GetData(/finite)
   print,'Missing value replaced with 9999'
   print,test->GetData(/finite,subs=9999)

   test->ConvertType, /uint
   print,test->GetData()
   test->ConvertType, /int
   print,test->GetData()
   print,'Converting to string ...'
   test->ConvertType, /string,format='(f9.3)'
   print,test->GetData()
   help,test->GetData()
   test->ConvertType, /float
   print,test->GetData()
   test->ConvertType, /int
   print,test->GetData()

   print
   print,test->GetData(/negative,result=result),' ',result
   print,test->GetData(/positive,result=result),' ',result
   print,test->GetData(/finite,result=result),' ',result
   print,test->GetData(/non_empty,result=result),' ',result

   test->ConvertType, /string,format='(f9.3)'
   print,test->GetData(/finite,result=result),' ',result
   print,test->GetData(/non_empty,result=result),' ',result
   print,test->GetData(/non_empty,result=result,/mask),' ',result

   Obj_Destroy, test


   lon  = obj_new('MGS_BaseVariable', findgen(128), name='LON' )
   lat  = obj_new('MGS_BaseVariable', findgen(64), name='LAT'  )

   var1 = obj_new('MGS_BaseVariable', dist(128,64), lon, lat,  $
                  name='Ozone')

   var2 = var1->Copy()
   var2->SetProperty, name='CO'

   ;; retrieve the name property from both variables to check they are different:
   var1->getproperty,name=name1
   var2->getproperty,name=name2
   print,'Name 1 = ',name1,'  Name2 = ',name2

   ;; invert the dimensions of the first variable
   lon->setproperty,data=reverse(findgen(128))
   lat->setproperty,data=reverse(findgen(64))

   ;; print the first element of lon and lat for both variables:
   var1->getproperty,dim1=d1,dim2=d2
   lon1 = d1->getData()
   lat1 = d2->getData()
   var2->getproperty,dim1=d1,dim2=d2
   lon2 = d1->getData()
   lat2 = d2->getData()
   print,'First elements of :'
   print,'lon1 = ',lon1[0],'  lon2 = ',lon2[0]
   print,'lat1 = ',lat1[0],'  lat2 = ',lat2[0]

   ;; clean up
   Obj_Destroy, d1    ; copy of lon in var2
   Obj_Destroy, d2    ; copy of lat in var2
   Obj_Destroy, lon
   Obj_Destroy, lat

   Obj_Destroy, var1
   Obj_Destroy, var2

END
