;+
; NAME:
;    MGS_BaseGUI
;
; PURPOSE:
;    This is a generic object to handle widget applications from
;  simple dialogs to complex menu-driven applications. This base
;  widget object provides a lot of functionality (building the GUI,
;  management of events, notification of other objects, automatic
;  management of compound widgets, etc.), but its GUI is empty except
;  for a menu bar (empty) or a row of dialog buttons if these are
;  defined. Any real dialog or application should inherit from this
;  BaseGUI object and overwrite a couple of methods in order to
;  provide the intended functionality of that dialog or application.
;    This object can handle blocking and non-blocking widgets, and it
;  makes no difference between a compound widget, a widget dialog, or
;  a (menu-driven) widget application. Every widget you build can be
;  used in any of these ways. The object keeps track of its
;  widget realisation and does not allow more than one widget copy of
;  one object instance. If you set the singleton property, only one
;  widget for this object type (class name) will be allowed (already
;  realized widgets will not be deleted, however).
;    Normally, a widget objects lifecycle begins with Obj_New() and
;  ends with Obj_Destroy. Inbetween, you can have as many calls to the
;  object's GUI method as you want, and you will always pick up where
;  you left since the object remembers the widget state. However, as
;  this method requires a little bookkeeping of the object reference
;  (in order to prevent memory leaking), and there are people who are
;  scared of objects in general, and memory leaks in particular, you
;  can also destroy a widget object when you kill the widget, provided
;  you have set the Destroy_Upon_Cleanup keyword when you initialised
;  the object.  
;    You can manage any number of compound widgets (MGS_BaseGUI
;  children) within one widget application by storing the widget
;  objects in the built-in compound container. This allows for
;  automatic update of several compound widget properties and ensures
;  that everything is properly cleaned when the object is
;  destroyed. The compound container makes life REALLY easy as you can
;  see for example in the MGS_InputMask object which provides a great
;  deal of functionality with very few program statements. 
;    
;  Please note that not all possible object and widget combinations
;  have been rigorously tested. This is quite a daunting and complex
;  task, and I need your input to improve this program.
;  At this stage, MGS_BaseGUI should still be considered experimental,
;  and you may face several changes before the first beta version will
;  become available. I will try not to modify the (public) interface
;  routines, but I cannot guarantee for their stability at this point.
;
; CATEGORY:
;  Generic Objects, Widget programming
;
; CALLING SEQUENCE:
;    theWidget = Obj_New('mgs_guiobject', widget_title='Test widget')
;    theWidget->GUI, /block
;    result = theWidget->GetState()
;    help,result, /structure
;    Obj_Destroy, theWidget
;
; REQUIREMENTS:
;    Inherits from mgs_baseobject
;
; INHERITING:
;    Objects that want to use the BaseGUI properties (i.e. 'real
;    world' objects) will normally overwrite several of the following
;    methods (* indicates that inherited method should also be called):
;    * Init:  add keywords providing default values for the widget
;             fields
;    * Cleanup: remove objects and pointers added in your widget
;    * SetProperty: add keywords for setting specific widget properties
;    * GetProperty: Add keywords to retrieve individual object
;             properties
;      BuildGUI: add statements to build the widget 
;    * UpdateObject: add statements to collect information from
;             widget components and store the validated result in
;             object.value. Need to call inherited method to do
;             recursion through compound widgets.
;    (*)Validate: rewrite to include error checking on the argument
;             and type conversion if necessary. 
;      SetValue: needs to be rewritten if you need to store values in
;             compound widgets
;      GetValue: needs to be rewritten if the object value consists of
;             values in compound widgets
;    * Show: needs to be rewritten only for draw widgets where
;             XManager cannot update the contents automatically
;
;
; ACKNOWLEDGEMENTS:
;    I owe a whole lot of this program to David Fanning who pioneered
;    event handling and widget programming in IDL. And thanks to Ben
;    Tupper who tried it out before anyone else.
;
; MODIFICATION HISTORY:
;    mgs, 15 Feb 2001: Version 0.1
;            derived from the MPI_PlotConfig application written by
;            David Fanning.
;    mgs, 04 Apr 2001: Version 1.0 released
;    mgs, 23 Apr 2001: - added ...Widget_Cleanup method for use in
;                        wrapper programs
;                      - added NotifyObject mechanism
;    mgs, 24 Apr 2001: - added menu option 
;                      - made buttons more flexible, including
;                        automatic help and reset.  
;                      - rearranged code and made it even more generic
;                      - improved documentation
;-
;
;###########################################################################
;
; LICENSE
;
; This software is OSI Certified Open Source Software.
; OSI Certified is a certification mark of the Open Source Initiative.
;
; Copyright  2001 Martin Schultz
;
; This software is provided "as-is", without any express or
; implied warranty. In no event will the authors be held liable
; for any damages arising from the use of this software.
;
; Permission is granted to anyone to use this software for any
; purpose, including commercial applications, and to alter it and
; redistribute it freely, subject to the following restrictions:
;
; 1. The origin of this software must not be misrepresented; you must
;    not claim you wrote the original software. If you use this software
;    in a product, an acknowledgment in the product documentation
;    would be appreciated, but is not required.
;
; 2. Altered source versions must be plainly marked as such, and must
;    not be misrepresented as being the original software.
;
; 3. This notice may not be removed or altered from any source distribution.
;
; For more information on Open Source Software, visit the Open Source
; web site: http://www.opensource.org.
;
;###########################################################################

; =============================================================================
; Non object functions and procedures
; =============================================================================

; -----------------------------------------------------------------------------
; MGS_BaseGUI_Widget_Events:
;    The main event handler for the object's widget. It reacts to
; "messages" in the UValue of the widget causing the event. The
; message indicates which object method to call. A message consists of
; an object method and the object reference.
;    The event handler method must be a function with a single event
; argument. If the function returns 0, the event is swallowed,
; otherwise, the returnvalue of the event handler method is passed as
; argument to all objects that are stored in the notifyobject list of
; the object widget that generated the widget event. For this purpose,
; the event handler should return a structure that contains at least
; two tags: 'object', and 'eventtype'. 

PRO MGS_BaseGUI_Widget_Events, event

   ;; The default event handling method is Event_Handler
   theMethod = 'EVENT_HANDLER'
   theObject = 0L  ;; invalid dummy

   ;; Event Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      Catch, /Cancel
      IF Obj_Valid(theObject) THEN BEGIN
         theObject->ErrorMessage, $
            'Specified functionality currently NOT IMPLEMENTED: '+ $
            theMethod
      ENDIF ELSE BEGIN
         Message, 'Invalid event/object reference'
      ENDELSE 
      RETURN
   ENDIF

   ;; The widget's UValue must contain at least a valid object
   ;; reference (then that objects 'Event_Handler' method will be
   ;; called), or it must contain a structure with an 'object' and a
   ;; 'method' tag.
   Widget_Control, event.ID, Get_UValue=info

   IF (Obj_Valid(info))[0] THEN BEGIN
      theObject = info
   ENDIF ELSE BEGIN
      theObject = info.object
      theMethod = info.method
   ENDELSE 
   theObject->GetProperty, debug=debug
   IF debug GT 0 THEN print, 'Calling event method ',theMethod,'...'

   ;; Call the object/widget specific event handler method
   event = Call_Method(theMethod, theObject, event)
   
   ;; If the return value is a valid structure, consider this an event
   ;; "summary" that shall be passed on to objects that are registered
   ;; to be notified
   IF ChkStru(event) THEN Call_Method,'NOTIFYOBJECTS', theObject, event

END


; -----------------------------------------------------------------------------
; MGS_BaseGUI_Widget_Cleanup:
;   This is a generic cleanup procedure for widgets of the MGS_BaseGUI
; hierarchy. It attempts to extract the object reference of the widget
; object from the top level base UValue field, then destroys that
; object. This procedure is only needed if the widget objects are used
; in a non-object wrapper routine; otherwise the user should keep
; track of the object status himself. You need to initialize the
; widget object with the Destroy_Upon_Cleanup keyword in order to
; activate this procedure. 

PRO MGS_BaseGUI_Widget_Cleanup, tlb


   ;; Get the UValue of the top level base
   Widget_Control, tlb, Get_UValue=info

   ;; Check if it is a valid object reference
   IF Obj_Valid(info) THEN Obj_Destroy, info

END


; =============================================================================
; Object methods
; =============================================================================


; -----------------------------------------------------------------------------
; ----- Event handling -----
; -----------------------------------------------------------------------------

; -----------------------------------------------------------------------------
; Event_Handler:
;    This method provides a generic reaction to the object's widget
; events. It simply updates any draw widgets (i.e. calls the Show
; method of self) and returns a modified event structure containing
; the self object reference to allow for notification of other
; objects. For many purposes, this event handler will already be
; sufficient, but occasionally you will want to establish a different
; event handler, e.g. to react to text events in a text widget. It is
; recommended that you add a new method for this purpose rather than
; overwriting this method. 
;   You can link an event handler method to a widget element simply by
; specifying it in the widgets UValue together with the self object
; reference (see BuildGUI for an example). If the widget UValue
; contains only an object reference, 'Event_Handler' is assumed as the
; default event handling method (see MGS_BaseGUI_Widget_Events).


FUNCTION MGS_BaseGUI::Event_Handler, event

;   Catch, theError
;   IF theError NE 0 THEN BEGIN
;      self->ErrorMessage, 'Error handling generic event'
;      IF self.debug GT 0 THEN help, event, /stru
;      RETURN, 0
;   ENDIF

   ;; Update drawing windows
   ;; self->Show

   returnEvent = Create_Struct('object',self, $
                               'eventtype', 'GENERIC_EVENT', $
                               event)

   RETURN, returnEvent

END 


; -----------------------------------------------------------------------------
; NotifyObjects:
;   This method loops through all entries in the self.notifyobject
; list and calls the object methods that are specified therein.
; The event argument should be considered a "summary" of the widget
; event that has already been processed by the object's event handler
; method. It should be a structure with, at a minimum, the object
; reference of the widget object that generated or processed the event
; and the eventtype as a string.
;   You can set up the notifyobject list upon object initialisation or
; with the SetProperty method. See MGS_RangeField for an example.
 
PRO MGS_BaseGUI::NotifyObjects, event 

   ;; Error Handler ...


   ;; Call methods of other widgets to be notified
   IF Ptr_Valid(self.notifyobject) THEN BEGIN
      IF self.debug GT 1 THEN BEGIN
         print,'Notifying other objects ...'
      ENDIF 

      FOR i=0L, N_Elements(*self.notifyobject)-1 DO BEGIN
         theobject = (*self.notifyobject)[i].object
         themethod = (*self.notifyobject)[i].method
         IF self.debug GT 1 THEN BEGIN
            help,(*self.notifyobject)[i],/stru
         ENDIF 
         IF Obj_Valid(theobject) THEN $
            event = Call_Method(themethod, theobject, event)
      ENDFOR 
   ENDIF 

END


; -----------------------------------------------------------------------------
; AcceptDialog:  (private)
;    This method is called from the event handler if the Accept button
; is pressed (blocking widgets). It forces all object information to
; be updated to the current widget state, then destroys the widget
; hierarchy. 

FUNCTION MGS_BaseGUI::AcceptDialog, event

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error accepting dialog'
      RETURN, 0
   ENDIF

   self->UpdateObject
   self->DestroyTLB

   theEvent = { object:self, eventtype:'ACCEPTBUTTON' }

   RETURN, theEvent

END 


; -----------------------------------------------------------------------------
; ApplyDialog:  (private)
;    This method is called from the event handler if the Apply button
; is pressed (non blocking widgets). It collects all information
; currently stored in the widget elements, then updates the display by
; calling the show method.

FUNCTION MGS_BaseGUI::ApplyDialog, event

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error applying dialog values'
      RETURN, 0
   ENDIF

   self->UpdateObject
   self->Show

   ;; Save current value as original value for reset operation
   ;; Need not be done for Accept or OK because there the widget is
   ;; destroyed, and the next GUI call will set oldvalue properly.
   ;; Need to loop over compound widgets, though.
   IF Ptr_Valid(self.value) THEN BEGIN
      Ptr_Free, self.oldvalue
      self.oldvalue = Ptr_New(*self.value)
   ENDIF 
   IF Obj_Valid(self.compound) THEN BEGIN
      cwobjects = self.compound->Get(/All)
      FOR i=0L, N_Elements(cwobjects)-1 DO BEGIN
         thiscw = cwobjects[i]
         IF Obj_Valid(thiscw) THEN BEGIN
            IF Ptr_Valid(thiscw.value) THEN BEGIN
               Ptr_Free, thiscw.oldvalue
               thiscw.oldvalue = Ptr_New(*thiscw.value)
            ENDIF 
         ENDIF 
      ENDFOR
   ENDIF


   theEvent = { object:self, eventtype:'APPLYBUTTON' }

   RETURN, theEvent
END 


; -----------------------------------------------------------------------------
; CloseDialog:  (private)
;    This method is called from the event handler if the Close button
; is pressed. It simply destroys the widget without changing the
; object status.  

FUNCTION MGS_BaseGUI::CloseDialog, event

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error closing dialog'
      RETURN, 0
   ENDIF

   self->DestroyTLB

   theEvent = { object:self, eventtype:'CLOSEBUTTON' }

   RETURN, theEvent
END


; -----------------------------------------------------------------------------
; CancelDialog:  (private)
;   A synonym for CloseDialog, but in addition the ResetDialog method
; is called to restore the initial dialog state.

FUNCTION MGS_BaseGUI::CancelDialog, event

   void = self->ResetDialog(event)
   RETURN, self->CloseDialog(event)

END


; -----------------------------------------------------------------------------
; OKDialog:  (private)
;   A synonym for AcceptDialog

FUNCTION MGS_BaseGUI::OKDialog, event

   RETURN, self->AcceptDialog(event)

END


; -----------------------------------------------------------------------------
; HelpDialog:  (private)
;   This method opens a help window if a Help button is defined

FUNCTION MGS_BaseGUI::HelpDialog, event

   theText = [ 'MGS_BaseGUI object oriented widget programming', $
               '', $
               'You activated the help function without overriding the HelpDialog', $
               'method. HelpDialog should be a function method with an event', $
               'argument.' ]

   self->ErrorMessage, theText, /Info

   RETURN, 0

END


; -----------------------------------------------------------------------------
; ResetDialog:  (private)
;   This method restores the initial value of the dialog.
;   The idea is that you store the initial dialog "value" in the
; object's oldvalue field in the Init method, using a format that is
; accepted by the SetValue method. Afterwards it loops over all
; objects stored in the compound container to do the same.
; NOTE:
; (1) Problem with Reset: after text fields are updated, the cursor
; will be placed in the last text field that was added to the widget
; tree. We cannot restore the input focus straightforwardly, because
; it has changed by clicking on the Reset button.

FUNCTION MGS_BaseGUI::ResetDialog, event

   ;; Error handler
   ;; ...

   IF Ptr_Valid(self.oldvalue) THEN self->SetValue, *self.oldvalue

   ;; Loop through compound widgets and reset their values
   ;; Update all compound widgets
   IF Obj_Valid(self.compound) THEN BEGIN
      cwobjects = self.compound->Get(/All)
      FOR i=0L, N_Elements(cwobjects)-1 DO BEGIN
         IF Obj_Valid(cwobjects[i]) THEN void=cwobjects[i]->ResetDialog(event)
      ENDFOR
   ENDIF
   
   ;; Update widget
   ;; self->Show     ;; *** IF THIS IS ACTIVATED, MGS_Field will
                     ;; break!! ****

   RETURN, 0

END


; -----------------------------------------------------------------------------
; UpdateObject:  (private)
;    This method should be used to gather all the information in the
; GUI and put it into the objects value field.
; In the BaseGUI Object, all compound widgets are updated.

PRO MGS_BaseGUI::UpdateObject

   ;; Update all compound widgets
   IF Obj_Valid(self.compound) THEN BEGIN
      cwobjects = self.compound->Get(/All)
      FOR i=0L, N_Elements(cwobjects)-1 DO BEGIN
         IF Obj_Valid(cwobjects[i]) THEN cwobjects[i]->UpdateObject
      ENDFOR
   ENDIF

END


; -----------------------------------------------------------------------------
; ----- Getting and setting widget values -----
; -----------------------------------------------------------------------------

; -----------------------------------------------------------------------------
; GetState:
;    This method constructs a structure containing fields with values
; set to the current state of the object. While GetState should always
; return a structure, the "sibling" method GetValue may be used to
; return information about the object in its most suitable form
; (e.g. a numeric value or a string from an input field). 
;    The ok keyword can be used to return information about the validity
; of the current object state (1=valid, 0=invalid). In the BaseGUI ok
; will always be returned as 0.
; NOTE:
; (1) The GetValue function calls will take care of updating the
; object if necessary and desired. 

FUNCTION MGS_BaseGUI::GetState, ok=ok, _Ref_Extra=e

   ;; Error Handler
;   Catch, theError
;   IF theError NE 0 THEN BEGIN
;      self->ErrorMessage, 'Error retrieving widget status'
;      RETURN, 0
;   ENDIF

   ok = 0

   ;; Build result structure:
   ;; Use object names as tag names, the result of GetValue as tag
   ;; values
   thename = StrTrim(self.name,2)
   IF thename EQ '' THEN thename = Obj_Class(self)
   struct = Create_Struct( thename, self->GetValue() )

   ;; Get value of all compound widgets
   IF Obj_Valid(self.compound) THEN BEGIN
      cwobjects = self.compound->Get(/All)
      FOR i=0L, N_Elements(cwobjects)-1 DO BEGIN
         IF Obj_Valid(cwobjects[i]) THEN BEGIN
            thename = (cwobjects[i]).name
            struct = Create_Struct( thename, $
                                    cwobjects[i]->GetState(), $
                                    struct )
         ENDIF 
      ENDFOR
   ENDIF

   ok = 1

   RETURN, struct
END


; -----------------------------------------------------------------------------
; GetValue:
;    This method returns the latest valid "value" of the widget in a
; useful format. Depending on the value of self.updatemode, the
; UpdateObject method is called beforhand to make sure that the
; objects value field contains the most recent validated data.
;    You should overwrite the UpdateObject method in order to store
; the relevant widget data in a suitable form. You probably don't need
; to overwrite the GetValue method itself.
; NOTE:
; (1) The values of updatemode are:
;      0 = never update automatically
;      1 = update automatically 
;      2 = update immediately
; See MGS_Field for a more detailed description.

FUNCTION MGS_BaseGUI::GetValue, ok=ok, _Extra=e

   ;; Error handler
   ;; ...

   ok = 0

   IF self.updatemode EQ 1 THEN self->UpdateObject

   retval = 0L
   IF Ptr_Valid(self.value) THEN BEGIN
      retval = *self.value
      ok = 1
   ENDIF ELSE BEGIN
      self->ErrorMessage, $
         ['The object contains no valid "value". This is probably a programming ', $
          'error and should be reported to the author of '+Obj_Class(self)+'.', $
          'It is also possible that you overwrote the UpdateObject and Validate', $
          'methods, but they did not get "installed". In this case, try a', $
          '.Reset_Session command and try again.']
   ENDELSE 
          
   RETURN, retval
END


; -----------------------------------------------------------------------------
; SetValue:
;   This method accepts a value argument, validates it, and stores it
; in the object's value field if the validation was successful or
; displays an error message otherwise.
; NOTE:
; (1) Validate must take care of all error checking including a test
; for an undefined argument. The result keyword of validate must
; return 'OK' if validation was successful.

PRO MGS_BaseGUI::SetValue, value, ok=ok

   newvalue = self->Validate(value, result=result)

   ok = ( StrUpCase(result) EQ 'OK' ) 

   IF ok THEN BEGIN
      Ptr_Free, self.value
      self.value = Ptr_New(newvalue)
      ;; Update display
      self->Show
   ENDIF ELSE BEGIN
      self->ErrorMessage, 'Invalid value argument. Result='+result
   ENDELSE 

   RETURN
END


; -------------------------------------------------------------------------------------
; Validate:
;    This method is intended to check the consistency of the argument
; against the object's expectations. The return value should have a
; type suitable for storing in the object's value field. Use the
; result keyword to return a status information.
;    BaseGUI simply returns the value as it is passed in, or a zero
; integer if arg is undefined. This may be suitable for some widget
; objects, but generally you will have to overwrite this method to
; provide some error checking. 

FUNCTION MGS_BaseGUI::Validate, arg, result=result

   IF N_Elements(arg) EQ 0 THEN BEGIN
      result = 'INVALID_DATA'
      retval = 0L
   ENDIF ELSE BEGIN
      result = 'OK'
      retval = arg
   ENDELSE 

   RETURN, retval

END

; -----------------------------------------------------------------------------
; ----- Accessing and manipulating widget properties -----
; -----------------------------------------------------------------------------

; -----------------------------------------------------------------------------
; GetTLB:
;    This method returns the top-level base of the widget (useful for
; compound widgets). It recurses through the widget hierarchy in case
; the current object is a compound widget.
; **** PROBABLY NOT NEEDED ****

;FUNCTION MGS_BaseGUI::GetTLB, startID

;   ;; Error Handler
;   Catch, theError
;   IF theError NE 0 THEN BEGIN
;      self->ErrorMessage, 'Error retrieving top level base'
;      RETURN, 0
;   ENDIF

;   IF N_Elements(startid) EQ 0 THEN startID = self.tlb

;   parent = Widget_Info(startID, /Parent)
;   IF parent EQ 0 THEN BEGIN
;      parent = startID 
;   ENDIF ELSE BEGIN
;      parent = self->GetTLB(parent)
;   ENDELSE

;   RETURN, parent  

;END


; -----------------------------------------------------------------------------
; CenterTLB:  (private)
; This method computes the center of the screen and places the top
; level base so that it is centered. With the x and y arguments, it
; can be placed anywhere on the screen.
;   This method is based on David Fanning's CenterTLB program (version
; of February 2001).

PRO MGS_BaseGUI::CenterTLB, x, y, NoCenter=nocenter

   IF NOT Widget_Info(self.tlb, /Valid_ID) THEN RETURN
   IF self.parent NE 0 THEN RETURN

   IF N_Elements(x) EQ 0 THEN xc = 0.5 ELSE xc = Float(x[0])
   IF N_Elements(y) EQ 0 THEN yc = 0.5 ELSE yc = 1.0 - Float(y[0])
   center = 1 - Keyword_Set(nocenter)

   screenSize = Get_Screen_Size()
   xCenter = screenSize[0] *xc
   yCenter = screenSize[1] *yc
   
   geom = Widget_Info(self.tlb, /Geometry)
   xHalfSize = geom.Scr_XSize / 2    ; integer division on purpose
   yHalfSize = geom.Scr_YSize / 2

   XOffset = 0 > (xCenter - xHalfSize) < (screenSize[0] - geom.Scr_Xsize)
   YOffset = 0 > (yCenter - yHalfSize) < (screenSize[1] - geom.Scr_Ysize)

   Widget_Control, self.tlb, XOffset=XOffset, YOffset=YOffset

END


; -----------------------------------------------------------------------------
; DestroyTLB: 
;    This method destroys the top-level base of the widget but only if
; this object is not a compound widget.

PRO MGS_BaseGUI::DestroyTLB

   ;; Error Handler
;   Catch, theError
;   IF theError NE 0 THEN BEGIN
;      self->ErrorMessage, 'Error while destroying widget'
;      RETURN
;   ENDIF

   IF self.parent EQ 0 AND Widget_Info(self.tlb, /Valid_ID) THEN $
      Widget_Control, self.tlb, /Destroy
END


; -----------------------------------------------------------------------------
; GetRegistername:  (private)
; This method returns a unique registration name for the XManager. 
; It uses the output of help,self to get the ObjHeapVarXX string which
; is guaranteed to have a unique integer number which is reproducible
; for each object instance. The objects class name is prepended.

FUNCTION MGS_BaseGUI::GetRegistername

   basename = Obj_Class(self)

   count = 1L

   help, self, output=dummy
   test = StRegex(dummy[0], '<.*[0-9]+',/Extract)
   thenumber = StrMid(test,11)
   retval = basename + String(thenumber,format='(i3.3)')

   RETURN, StrUpcase(retval)

END


; -----------------------------------------------------------------------------
; GetParent:  (private)
; This method returns the widget_base id which serves as a parent
; widget for a given compound widget stored in the compound
; container. If the compound widget contains no valid hook number or
; if the cwbaseid list is empty, the compound widget's base will be
; the layoutID.

FUNCTION MGS_BaseGUI::GetParent, cwobject

   retval = self.layoutID

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error getting parent widget ID for compound widget'
      RETURN, retval
   ENDIF

   ;; Check if (and how many) cwbaseid's are stored
   nbases = 0L
   IF Ptr_Valid(self.cwbaseid) THEN BEGIN
      nbases = N_Elements(*self.cwbaseid)
   ENDIF

   ;; Check if the compound widget's hook number is valid
   cwobject->GetProperty, hook=hook, name=name
   IF hook GE 0 AND hook LT nbases THEN BEGIN
      ;; Test validity of cwbaseid value
      tmp = (*self.cwbaseid)[hook]
      IF tmp GT 0 THEN retval = tmp
      IF self.debug GT 1 THEN print,'Attaching compound widget '+name+' to widget '+ $
         StrTrim(tmp,2)
   ENDIF

   RETURN, retval

END


; ******* CAN PROBABLY DELETE THE FOLLOWING ********
; -----------------------------------------------------------------------------
; CheckEvent:  (private)
;    This event handler method checks to see if an event should be
; passed on (compound widgets only???).
; NOTE: Do we need to worry about compound widgets in this architecture??

;FUNCTION MGS_BaseGUI::CheckEvent, event

;   Catch, theError
;   IF theError NE 0 THEN BEGIN
;      self->ErrorMessage, 'Error checking event'
;      RETURN, 0
;   ENDIF

;   ;; Make sure all values are current by updating.
;   self->UpdateObject
;   IF self.all_events THEN self->SendEvent

;   RETURN, 0
;END 


; -----------------------------------------------------------------------------
; SendEvent:  (private)
; This method sends an event to the widget to be notified, if one
; exists.
; QUESTION: Is it possible to notify more than one widget???
; NOTE: Do we need to worry about compound widgets??

;PRO MGS_BaseGUI::SendEvent

;   ;; Error Handler
;   Catch, theError
;   IF theError NE 0 THEN BEGIN
;      self->ErrorMessage, 'Error sending event'
;      RETURN
;   ENDIF

;   RETURN

;   ;; DEACTIVATED THE FOLLOWING *******
;   ;; IF Total(self.notifyID) LE 0 THEN RETURN

;   ;; thisEvent = { MGS_MAP_EVENT, $
;   ;;              ID: self.notifyID[0], $
;   ;;              TOP:self.notifyID[1], $
;   ;;              HANDLER: 0L, $
;   ;;              OBJECT: self }

;   ;; Widget_Control, self.notifyID[0], Send_Event=thisEvent

;END

; ******** END OF DELETION  ***************************


; -----------------------------------------------------------------------------
; ----- Building and displaying the GUI -----
; -----------------------------------------------------------------------------

; -----------------------------------------------------------------------------
; Show:
;   This method is a dummy method which only calls the Show method of
;   all compound widgets stored in the object.

PRO MGS_BaseGUI::Show

   ;; Typically inherited routines will need to save the current
   ;; window id and obtain the id of the draw widget before
   ;; issuing any plot commands:
   ;; IF NOT Widget_Info(self.tlb, /Valid_ID) THEN RETURN
   ;; currentwindow = !D.Window
   ;; Widget_Control, self.drawid, Get_Value=wid
   ;; Wset, wid

   ;; PLOT COMMANDS HERE

   ;; Reset window ID to window that was active before
   ;; IF currentwindow NE -1 THEN WSet, currentwindow


   ;; Show all compound widgets
   IF Obj_Valid(self.compound) THEN BEGIN
      cwobjects = self.compound->Get(/All)
      FOR i=0L, N_Elements(cwobjects)-1 DO BEGIN
         IF Obj_Valid(cwobjects[i]) THEN cwobjects[i]->Show
      ENDFOR
   ENDIF

END


; -----------------------------------------------------------------------------
; BuildGUI:  (private)
;    This is the method to actually build your widgets. Inherited GUI
; objects should override this method (not the GUI method!) and add
; all the labels, buttons, etc. here. 
;    The user on the other hand communicates through the GUI method
; and does not have to know how the widgets are built. 
;    The self.layoutID widget identifier is the base widget ID for all
; dialog elements that you add. 
;    Compound widgets are inserted into the dialog in the GUI
; method after the BuildGUI method is called. The default (compound
; widgets without a "hook" number) is to base the compound widgets on
; self.layoutID in the order as they are stored in the compound
; container. If you want to control the layout more tightly, you can
; enter the widgets IDs of any number of widget bases into the 
; object's cwbaseid field (pointer) and assign each compound widget a
; hook number which is the index into this widget_base list. Note that
; widget IDs cannot be stored directly in the compound widgets because
; they may change from realisation to realisation. See MGS_MapSetup
; for an example how to control the widget layout with compound widgets. 

PRO MGS_BaseGUI::BuildGUI

   ;; Example 1: straightforward widget building
   ;;    dummyID = Widget_Label(self.layoutID, value='Hello World', /Align_Center)


   ;; Example 2: with compound widgets attached to different layout bases
   ;;    leftbase = Widget_Base(self.layoutID, col=1, frame=1)
   ;;    rightbase = Widget_Base(self.layoutID, col=1, frame=1)
   ;;    self.cwbaseid = Ptr_New([leftbase, rightbase])
   ;; Each compound widget should then get a hook number of 0 or 1
   ;; depending on whether it shall be added to the left or right
   ;; base. This is probably best done in the widget application's
   ;; Init method. See MGS_MapSetup for an example.


   ;; Example 3: with optional menu bar
   ;;    IF self.menuID GT 0 THEN BEGIN   ; test if menu is 'alive'
   ;;       fileID = Widget_Button(self.menuID, value='File', $
   ;;                              UValue={object:self, $
   ;;                              method:'FileMenu'})
   ;;    ENDIF


   ;; Display an info message to indicate that this method must be
   ;; overwritten (unless we have a widget that consists only of
   ;; compound widgets)
   hascompounds = 0
   IF Obj_Valid(self.compound) THEN BEGIN
      hascompounds = ( self.compound->Count() GT 0 )
   ENDIF 
   IF NOT hascompounds THEN BEGIN
      self->ErrorMessage, $
         ['Your GUI call called the BuildGUI method of the MGS_BaseGUI object.', $
          'This indicates that you did not overwrite the BuildGUI method in your', $
          'object. If you did overwrite it, you may have to try a .Reset_Session', $
          'in order to make the change work.'], /Info
   ENDIF 

END


; -----------------------------------------------------------------------------
; GUI:
; This method provides the main user interface with the widget
; object. It can handle blocking and non-blocking dialog widgets as
; well as compound widgets or widget applications with a menu bar. The
; GUI changes it's functionality according to the given keywords
; ("polymorphism" in object-speak).  
;
; (1) Blocking or non-blocking dialog widgets (default):
; The GUI method creates a top level base widget with a title, a
; 'layout' base, and buttons. A group_leader can be specified which
; will cause this widget to close if the group_leader is
; closed. Optionally you can add a menu or change the number and
; appearance of the buttons. There are predefined buttons and button
; event handlers for 'Accept', 'Apply', 'Close', 'Cancel', 'OK',
; 'Help', and 'Reset'. All other button values can only be used in
; inheriting objects which must then provide a suitable event handler
; method. 
;    If you don't want this widget to be managed by XManager, set the
; No_Manager keyword.
;    In order to create the widget hierarchy, the GUI method calls the
; BuildGUI method which shall be overwritten by child objects to
; create the actual widget layout (Don't overwrite the GUI method
; unless you really know what you are doing!). After realisation, the
; object's Show method is called which should draw the contents of any
; draw widgets. 
;
; (2) Menu driven applications:
; If you had set the menu keyword in the Init or SetProperty method,
; or if you provide the menu switch to the GUI method, the top level
; base will have a menu bar and the widget ID of the menu will be
; stored in the object's menuID field. Unless you explicitely specify
; buttons in the GUI call, the menu keyword overrides any button
; definitions, so you should get an "application" widget without
; buttons. The menu itself must be built in the BuildGUI method.
;
; (3) Compound widgets:
; If you specify a parent widget argument, the GUI method will
; create a widget base for a compound widget. In this case, the widget
; will not contain any buttons, nor a menu, and it will not be
; realized or registered with the XManager. 
;   Normally, all compound widgets that are stored in the compound
; container will be built automatically. Therefore, you will need this
; call only if you want to use the object widget in a classical
; non-object widget application.
;
; Example calls:
;    object->GUI          ; creates widget in non-blocking mode
;    object->GUI, group_leader=gl  ; same with a group leader
;    object->GUI, /block  ; creates widget in blocking mode
;    object->GUI, /menu   ; creates a widget with a menu bar
;    object->GUI, buttons=['OK', 'Cancel', 'Help']  ; creates a dialog
;                         ; with 3 buttons. The button event handlers
;                         ; are OKDialog, CancelDialog, and
;                         ; HelpDialog.
;    object->GUI, parentID ; creates a compound widget
;
; NOTES:
; (1) The button keyword (or the /AcceptButton, /ApplyButton,
; /CloseButton, /CancelButton, /OKButton, /HelpButton, /ResetButton
; keywords) override the settings from Init or SetProperty. Each
; string element of buttons will be used as text value of a button;
; the event handler method of this button will be set to a function
; named buttons[i]+'Dialog'. The user/programmer must make sure that
; this method exists. The basegui object provides default event
; handlers for those buttons that can be set with a boolean keyword.
;   If no buttons are defined, the object's buttons property is empty,
; the menu keyword is not set (or set to 0), and the GUI method is
; called without the parent argument, then the dialog will have two
; default buttons: 'Apply' and 'Close' for non-blocking widgets, and
; 'Accept', 'Cancel' for blocking widgets.

PRO MGS_BaseGUI::GUI, $
   Parent,  $                  ; This argument indicates that the object
                               ; is a compound widget!
   XPosition=xposition,  $     ; The x position of the widget (default is centered)
   YPosition=yposition,  $     ; The y position of the widget (default is centered)
   UpperLeft=upperleft,  $     ; Compute widget position with respect to upper left corner
   Block=block, $              ; Set this keyword if you want to block the command line.
   Group_Leader=group_leader, $ ; The group leader for this GUI.
   No_Manager=no_manager, $    ; Don't pass control to XManager
   menu=menu, $                ; Create a menu bar
   buttons=buttons, $          ; A list of button values
   AcceptButton=acceptbutton, $ ; Add an 'Accept' button
   ApplyButton=applybutton,  $ ; ...
   CloseButton=closebutton,  $
   CancelButton=cancelbutton, $
   OKButton=okbutton, $
   HelpButton=helpbutton, $
   ResetButton=resetbutton

   ;; Error Handler
;   Catch, theError
;   IF theError NE 0 THEN BEGIN
;      self->ErrorMessage, 'Error creating GUI'
;      ;; Reset object's tlb and layoutID values
;      self.tlb = 0L
;      self.layoutID = 0L
;      RETURN
;   ENDIF

   ;; Allow only one instance of this object's GUI
   ;; First check to see if a widget is registered with the object's
   ;; class name. This indicates a singleton object. Once a singleton
   ;; object is displayed, no other object of the same type shall be
   ;; created. 
   IF XRegistered(Obj_Class(self)) GT 0 THEN RETURN

   ;; Get a new name under which this object's GUI can be registered
   register_name = self->GetRegistername()

   ;; Next, check if an object with the same number as this one was
   ;; registered. This indicates that the GUI method of the same
   ;; object has been called twice which is not allowed.
   IF XRegistered(self.registered_name) GT 0 THEN RETURN
   IF self.debug GT 0 THEN help,self.registered_name,register_name

   ;; If we want to create a singleton object, overwrite the
   ;; register_name with the object's class name
   IF self.singleton THEN register_name = Obj_Class(self)

   ;; Set widget default font
   Widget_Control, Default_Font=self.defaultfont

   ;; Determine whether to add a menu 
   IF N_Elements(menu) GT 0 THEN BEGIN
      createmenu = Keyword_Set(menu)
   ENDIF ELSE BEGIN
      createmenu = Keyword_Set(self.createmenu)
   ENDELSE 

   ;; Determine which buttons to add to the dialog
   thebuttons = ''
   IF N_Elements(parent) EQ 0 THEN BEGIN
      ;; automatic buttons cannot be added for compound widgets
      IF N_Elements(buttons) GT 0 THEN BEGIN
         ;; Test for buttons keyword
         thebuttons = StrTrim(buttons,2)
      ENDIF ELSE BEGIN
         ;; Test for boolean keywords
         thebuttons = ''
         IF Keyword_Set(AcceptButton) THEN thebuttons = [ thebuttons, 'Accept' ]
         IF Keyword_Set(ApplyButton) THEN thebuttons = [ thebuttons, 'Apply' ]
         IF Keyword_Set(CloseButton) THEN thebuttons = [ thebuttons, 'Close' ]
         IF Keyword_Set(CancelButton) THEN thebuttons = [ thebuttons, 'Cancel' ]
         IF Keyword_Set(OKButton) THEN thebuttons = [ thebuttons, 'OK' ]
         IF Keyword_Set(HelpButton) THEN thebuttons = [ thebuttons, 'Help' ]
         IF Keyword_Set(ResetButton) THEN thebuttons = [ thebuttons, 'Reset' ]
         
         IF N_Elements(thebuttons) GT 1 THEN BEGIN
            thebuttons = thebuttons[1:*]
         ENDIF ELSE IF NOT createmenu THEN BEGIN
            ;; Set default buttons unless menu is defined
            IF Ptr_Valid(self.buttons) THEN BEGIN
               thebuttons = StrTrim(*self.buttons,2)
            ENDIF ELSE BEGIN
               IF Keyword_Set(block) THEN thebuttons = [ 'Cancel', 'Accept' ] $ 
               ELSE thebuttons = [ 'Close', 'Apply' ]
            ENDELSE 
         ENDIF 
      ENDELSE 
   ENDIF 

   ;; Create top level widget or compound widget base
   self.parent = 0L
   self.menuID = 0L

   IF N_Elements(parent) GT 0 THEN BEGIN   ;; Compound widget
      tlb = Widget_Base(parent, Column=1, Title=self.window_title, $
                        Base_Align_Center=1, UValue=self)
      self.parent = parent
   ENDIF ELSE IF N_Elements(group_leader) EQ 0 THEN BEGIN
      ;; No group leader
      IF createmenu THEN BEGIN
         tlb = Widget_Base(Column=1, Title=self.window_title, $
                           Base_Align_Center=1, UValue=self, $
                           MBar=menuID)
         self.menuID = menuID
      ENDIF ELSE BEGIN
         tlb = Widget_Base(Column=1, Title=self.window_title, $
                           Base_Align_Center=1, UValue=self)
      ENDELSE 
   ENDIF ELSE BEGIN
      ;; With group leader
      IF createmenu THEN BEGIN
         tlb = Widget_Base(Column=1, Title=self.window_title, $
                           Base_Align_Center=1, $
                           Group_Leader=group_leader, UValue=self, $
                           MBar=menuID)
         self.menuID = menuID
      ENDIF ELSE BEGIN
         tlb = Widget_Base(Column=1, Title=self.window_title, $
                           Base_Align_Center=1, $
                           Group_Leader=group_leader, UValue=self)
                           
      ENDELSE 
   ENDELSE
   self.tlb = tlb

   ;; Create layout section of widget
   layoutID = Widget_Base(tlb, Base_Align_Center=1, $
                          Column=self.column_layout, $
                          Row=1-self.column_layout, $
                          Frame=self.layout_frame, $
                          UValue=self)
   self.layoutID = layoutID

   ;; Add the widget title if one is defined
   IF StrTrim(self.widget_title,2) NE '' THEN BEGIN
      self.titleID = Widget_Label(layoutID, $
                                  Value=self.widget_title, $
                                  Font=self.labelfont)
   ENDIF

   ;; Delete old compound widget base IDs
   Ptr_Free, self.cwbaseid

   ;; Call BuildGUI method to build the actual GUI
   self->BuildGUI

   ;; Now build all compound widgets 
   IF Obj_Valid(self.compound) THEN BEGIN
      cwobjects = self.compound->Get(/All)
      FOR i=0L, N_Elements(cwobjects)-1 DO BEGIN
         IF Obj_Valid(cwobjects[i]) THEN BEGIN
            thisparent = self->GetParent(cwobjects[i])
            cwobjects[i]->GUI, thisparent
         ENDIF 
      ENDFOR 
   ENDIF

   ;; Add the buttons to the dialog
   IF thebuttons[0] NE '' THEN BEGIN
      buttonbase = Widget_Base(tlb, Row=1)
      FOR i=0L, N_Elements(thebuttons)-1 DO BEGIN
         eventmethod = StrUpCase(StrCompress(thebuttons[i],/Remove_All)) + $
            'DIALOG'
         button = Widget_Button(buttonbase, Value=thebuttons[i], $
                                UValue={Object:self, Method:eventmethod})
      ENDFOR 
   ENDIF 

   ;; Remember initial state of widget
   Ptr_Free, self.oldvalue
   IF Ptr_Valid(self.value) THEN BEGIN
      self.oldvalue = Ptr_New(*self.value)
   ENDIF ELSE BEGIN
      self.oldvalue = Ptr_New()
   ENDELSE 

   ;; Realize dialog or application and pass control to XManager
   ;; unless this is a compound widget or No_Manager is set
   IF self.parent EQ 0 THEN BEGIN
      ;; Center the top level base
      self->CenterTLB, XPosition, YPosition, NoCenter=Keyword_Set(UpperLeft)

      ;; Show the GUI
      Widget_Control, tlb, /Realize

      ;; Show contents of draw widgets if any
      self->Show

      ;; Register the GUI with XManager and pass control to the IDL event
      ;; manager
      IF Keyword_Set(no_manager) EQ 0 THEN BEGIN
IF NOT Obj_Valid(self) THEN RETURN
         IF self.autodestroy THEN BEGIN
            XManager, register_name, tlb, $
               Event_Handler='MGS_BaseGUI_Widget_Events', $
               Cleanup='MGS_BaseGUI_Widget_Cleanup', $
               No_Block = 1 - Keyword_Set(block)
         ENDIF ELSE BEGIN
            XManager, register_name, tlb, $
               Event_Handler='MGS_BaseGUI_Widget_Events', $
               No_Block = 1 - Keyword_Set(block)
         ENDELSE 
         IF Obj_Valid(self) THEN self.registered_name = register_name
      ENDIF

   ENDIF

END


; -----------------------------------------------------------------------------
; ----- Retrieving and setting of object properties -----
; -----------------------------------------------------------------------------

; -----------------------------------------------------------------------------
; SetWidgetFonts:  (private)
; This method sets the widget font names. It is called by the Init or
; the SetProperty method. If one or both of the keywords are undefined
; or empty, the respective property is set to a operating system
; dependent default value.

PRO MGS_BaseGUI::SetWidgetFonts, $
     reset=reset,            $  ; Set this keyword to use default values if
                                ; the respective ...font keyword is not given
     widget_defaultfont=widget_defaultfont, $  ; A new default font for widgets
     widget_labelfont=widget_labelfont,     $  ; A new font for widget titles
     _Extra=extra                              ; For future additions


   thisOS = StrUpCase(!Version.OS_Family)
   CASE thisOS OF
      'WINDOWS': BEGIN
         os_labelfont = 'Helvetica*Bold'
         ;; if you prefer times ...
         ;; os_labelfont = 'Times*Bold'
         os_defaultfont = 'MS Sans Serif*10'
      END
      'MACOS': BEGIN
         os_labelfont = 'Helvetica*Bold'
         os_defaultfont = 'Helvetica*10'
         ;; if you prefer times ...
         ;; os_labelfont = 'Times*Bold'
         ;; os_defaultfont = 'Times*10'
      END
      ELSE: BEGIN
         os_labelfont = '-*-helvetica-bold-r-*-*-12-*'
         os_defaultfont = '-*-helvetica-medium-r-*-*-12-*'
         ;; if you prefer times ...
         ;; os_labelfont = '-*-times-bold-r-*-*-12-*'
         ;; os_defaultfont = '-*-times-medium-r-*-*-12-*'
      END
   ENDCASE

   IF N_Elements(widget_defaultfont) EQ 0 THEN BEGIN
      IF Keyword_Set(reset) THEN self.defaultfont = os_defaultfont
   ENDIF ELSE IF widget_defaultfont EQ '' THEN BEGIN
      self.defaultfont = os_defaultfont
   ENDIF ELSE BEGIN
      self.defaultfont = widget_defaultfont
   ENDELSE

   IF N_Elements(widget_labelfont) EQ 0 THEN BEGIN
      IF Keyword_Set(reset) THEN self.labelfont = os_labelfont
   ENDIF ELSE IF widget_labelfont EQ '' THEN BEGIN
      self.labelfont = os_labelfont
   ENDIF ELSE BEGIN
      self.labelfont = widget_labelfont
   ENDELSE

   ;; Loop through all compound widgets and change their font as well
   IF Obj_Valid(self.compound) THEN BEGIN
      cwobjects = self.compound->Get(/All)
      FOR i=0L, N_Elements(cwobjects)-1 DO $
         IF Obj_Valid(cwobjects[i]) THEN BEGIN
         cwobjects[i]->SetWidgetFonts, reset=reset, $
            widget_defaultfont=widget_defaultfont, $
            widget_labelfont=widget_labelfont,     $
            _Extra=extra
      ENDIF 
   ENDIF
   
END


; -----------------------------------------------------------------------------
; GetProperty:
;    This method extracts specific object values and returns them to
; the user. Normally, the user should use the GetValue() or GetState()
; method to retrieve the object state in a usable form.

PRO MGS_BaseGUI::GetProperty, $
   Window_title=window_title,  $  ; The title of the widget window
   Widget_title=widget_title, $   ; The title label of the tlb widget
   widget_defaultfont=widget_defaultfont, $  ; The current default font for widgets
   widget_labelfont=widget_labelfont,     $  ; The current font for widget titles
   hook=hook,  $  ; The Widget_Base index number for a compound widget
   notifyobject=notifyobject,  $  ; The object reference of the parent object
   _Ref_Extra=extra           ; Inherited and future keywords
                              ;
                              ; Inherited keywords:
                              ; name      : The variable name
                              ; uvalue    : a user-defined value


   ;; Get properties from base object
   self->MGS_BaseObject::GetProperty, _Extra=extra

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error retrieving object properties!'
      RETURN
   ENDIF

   ;; Return object properties
   widget_title = self.widget_title
   widget_defaultfont=self.defaultfont
   widget_labelfont=self.labelfont
   
   hook=self.hook
   IF Ptr_Valid(self.notifyobject) THEN BEGIN
      notifyobject = *self.notifyobject
   ENDIF

END


; -----------------------------------------------------------------------------
; SetProperty:
;    This method sets specific object values. Derived objects may want
; to overwrite and extend this method to allow storing additional
; information.
; Notes:
; (1) It is not possible to change a widget font once that
; widget is realized. The ...font keywords will therefore only be
; effective if you close and rebuild the GUI. If you specify an empty
; string for either ...font keyword, the font will be reset to the
; default value. 
; (2) If you change the menu or buttons properties of the object,
; these values will be used as defaults for the GUI method. Keywords
; to the GUI method override these settings.

PRO MGS_BaseGUI::SetProperty, $
     widget_defaultfont=widget_defaultfont, $  ; A new default font for widgets
     widget_labelfont=widget_labelfont,     $  ; A new font for widget titles
     reset_fonts=reset_fonts,               $  ; reset widget fonts to their default
     window_title=window_title, $              ; A new title for the widget window
     widget_title=widget_title,             $  ; A title label for the widget 
     row_layout=row_layout,                 $  ; Layout widget in 1 row (default is column)
     no_frame=no_frame,                     $  ; Don't draw frame around layout widget
     singleton=singleton,                   $  ; Only one instance of GUI allowed
     menu=menu,                             $  ; create menu widget
     buttons=buttons,                       $  ; list of button values
     updatemode=updatemode,                 $  ; Update object value never|auto|always
     hook=hook,                             $  ; Widget_Base index number for compound widget
     _Extra=extra                  ;
                                   ; Inherited keywords:
                                   ; name      : The variable name
                                   ; no_copy   : Don't keep local copy
                                   ;             of uvalue
                                   ; no_dialog : Don't display
                                   ;             interactive dialogs
                                   ; uvalue    : a user-defined value
                                   ; debug     ; debugging level


   ;; Set Properties of base object
   self->MGS_BaseObject::SetProperty, _Extra=extra

   ;; Error Handler
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error setting object properties'
      RETURN
   ENDIF

   ;; Replace widget fonts
   self->SetWidgetFonts, $
      widget_defaultfont=widget_defaultfont, $
      widget_labelfont=widget_labelfont, $
      reset=Keyword_Set(reset_fonts)

   ;; Replace widget window title
   IF N_Elements(window_title) GT 0 THEN BEGIN
      self.window_title = window_title

      IF Widget_Info(self.tlb, /Valid_ID) THEN $
         Widget_Control, self.tlb, TLB_Set_Title=self.window_title
   ENDIF

   ;; Replace widget title label
   ;; If widget is active, this can only be done if a non-empty title
   ;; was specified before the build.
   IF N_Elements(widget_title) GT 0 THEN BEGIN
      self.widget_title = widget_title

      IF Widget_Info(self.titleID, /Valid_ID) THEN $
         Widget_Control, self.titleID, Set_Value=self.widget_title
   ENDIF

   ;; Layout properties
   IF N_Elements(row_layout) GT 0 THEN $
      self.column_layout = 1 - Keyword_Set(row_layout)

   IF N_Elements(no_frame) GT 0 THEN $
      self.layout_frame = 1 - Keyword_Set(no_frame)

   IF N_Elements(menu) GT 0 THEN self.createmenu = Keyword_Set(menu)

   IF N_Elements(buttons) GT 0 THEN BEGIN
      Ptr_Free, self.buttons
      self.buttons = Ptr_New(String(buttons))
   ENDIF 

   IF N_Elements(updatemode) GT 0 THEN $
      self.updatemode = 0 > long(updatemode[0]) < 2

   IF N_Elements(hook) GT 0 THEN $
      self.hook = long(hook[0])

   ;; Change singleton property (only effective if no GUI of this
   ;; object already present)
   IF N_Elements(singleton) GT 0 THEN self.singleton = Keyword_Set(singleton)

END


; -----------------------------------------------------------------------------
; Cleanup:
;   This method destroys the GUI widget should it still exist, the
; container with compound widgets and all pointers. Then it
; calls the inherited cleanup method.

PRO MGS_BaseGUI::Cleanup

   ;; Kill the GUI (if it is valid)
   self->DestroyTLB

   ;; Clean up all compound widgets
   Obj_Destroy, self.compound

   ;; Delete the widget baseID list and the notifyobject list
   Ptr_Free, self.value
   Ptr_Free, self.oldvalue
   Ptr_Free, self.buttons
   Ptr_Free, self.cwbaseid
   Ptr_Free, self.notifyobject

   ;; Call parent's cleanup method
   self->MGS_BaseObject::Cleanup

END


; -----------------------------------------------------------------------------
; Init:
;   This method initializes the widget object. It does not build a GUI,
; but only sets a few variables such as the default fontname.
;    Derived objects should overload this method to set the parameters
; that provide the object's functionality.
; NOTE: If you initialize a compound widget object, you cannot pass
; the parent widget ID to the Init method. Instead, you will pass it
; as argument directly to the GUI method when building the parent widget.

FUNCTION MGS_BaseGUI::Init, $
     widget_defaultfont=widget_defaultfont, $  ; A specific default font for widgets
     widget_labelfont=widget_labelfont,     $  ; A specific font for widget titles
     window_title=window_title,             $  ; The title of the widget window
     widget_title=widget_title,             $  ; A title for the widget window
     row_layout=row_layout,                 $  ; Layout widget in 1 row (default is column)
     no_frame=no_frame,                     $  ; Don't draw frame around layout widget
     singleton=singleton,                   $  ; Display this GUI only once
     menu=menu,                             $  ; Create a menu in the GUI
     buttons=buttons,                       $  ; List of button values
     updatemode=updatemode,                 $  ; Update object value never|auto|always
     hook=hook,                             $  ; An index into a base_widget where the
                                ; compound widget is attached to
     notifyobject=notifyobject,             $  ; A reference to the parent object
     Destroy_Upon_Cleanup=destroy_upon_cleanup, $ ; Set this keyword if you are creating
                                ; this object from a wrapper
                                ; routine. Then the object will be
                                ; destroyed automatically if the widget is killed 
     _Extra=extra  ; Extra keywords from inherited objects
                                ;
                                ; Inherited keywords (from MGS_BaseObject):
                                ; name      : The object name
                                ; no_copy   : Don't retain a copy
                                ;             of uvalue
                                ; no_dialog : Don't display
                                ;             interactive dialogs
                                ; uvalue    : a user-defined value


   ;; Initialize parent object first
   IF not self->MGS_BaseObject::Init(_Extra=extra) THEN RETURN, 0

   ;; Error Handler 
   Catch, theError
   IF theError NE 0 THEN BEGIN
      self->ErrorMessage, 'Error initializing object'
      RETURN, 0
   ENDIF

   ;; Check for keywords and parameters.

   ;; Widget fonts
   self->SetWidgetFonts, $
      widget_defaultfont=widget_defaultfont, $
      widget_labelfont=widget_labelfont, $
      /Reset

   ;; The widget window title
   IF N_Elements(window_title) EQ 0 THEN $
      self.window_title = 'Generic widget object' $
   ELSE $
      self.window_title = window_title

   ;; The widget header label
   IF N_Elements(widget_title) EQ 0 THEN $
      self.widget_title = '' $
   ELSE $
      self.widget_title = widget_title

   self.singleton = Keyword_Set(singleton)

   createmenu = Keyword_Set(menu)

   IF N_Elements(buttons) GT 0 THEN BEGIN
      self.buttons = Ptr_New(String(buttons))
   ENDIF ELSE BEGIN
      self.buttons = Ptr_New()   
   ENDELSE 

   IF N_Elements(updatemode) EQ 0 THEN updatemode = 1L

   IF N_Elements(hook) EQ 0 THEN hook = 0L

   ;; Populate the object.

   self.tlb = 0L
   self.titleID = 0L
   self.layoutID = 0L
   self.column_layout = 1 - Keyword_Set(row_layout)
   self.layout_frame = 1 - Keyword_Set(no_frame)
   self.createmenu = createmenu
   self.updatemode = 0 > long(updatemode) < 2
   self.hook = long(hook[0])
   self.cwbaseid = Ptr_New()
   self.parent = 0L
   IF N_Elements(notifyobject) EQ 0 THEN BEGIN
      self.notifyobject = Ptr_New()
   ENDIF ELSE BEGIN
      IF ChkStru(notifyobject[0],['object','method']) THEN BEGIN
         self.notifyobject = Ptr_New(notifyobject)
      ENDIF ELSE BEGIN
         self->ErrorMessage, 'Notifyobject must be a structure with object and method tags'
         RETURN, 0
      ENDELSE 
   ENDELSE 

   self.autodestroy = Keyword_Set(Destroy_Upon_Cleanup)
   self.registered_name = ''

   self.compound = Obj_New('MGS_Container', name='Compound widgets')

   self.value = Ptr_New()  ;; the basegui has no value
   self.oldvalue = Ptr_New() 

   RETURN, 1
END


; -----------------------------------------------------------------------------
; MGS_BaseGUI__Define:
; This is the object definition for a generic object that has a
; single GUI interface for configuration. It inherits from
; MGS_BaseObject the abilities to set and query an object name and a
; uvalue. The base object also provides a general method for display
; of error messages which can be directed to a message dialog or to
; the log screen via the no_dialog flag.

PRO MGS_BaseGUI__Define

   struct = { MGS_BASEGUI, $      ; The object class.

           ;;; Widget properties

           tlb: 0L, $             ; The identifier of the top level base widget
           titleID: 0L, $         ; The identifier of the title label
           layoutID: 0L,  $       ; The identifier of the layout base (see GUI method)
           column_layout: 0L, $   ; Major widget layout in 1 column (default true)
           layout_frame: 0L, $    ; Draw frame around layout widget (default thickness 1)
           labelfont: "", $       ; The display fontname for widget section titles
           defaultfont: "", $     ; The default display fontname for widget text
           window_title: "", $    ; The title of the widget window
           widget_title: "", $    ; The title of the GUI widget (the top label widget)
           registered_name: "", $ ; The name under which the object is registered with XManager
           autodestroy: 0,       $  ; Destroy object if widget is killed
           singleton: 0,         $  ; Defines if this widget can be displayed only once
           value: Ptr_New(), $      ; Present validated widget "value"
           oldvalue: Ptr_New(),  $  ; Remembers the initial "value" of the widget
           updatemode: 0L,       $  ; 0=never, 1=auto, 2=always update object "value"
           ;; Propeties for a menu and a button list
           buttons: Ptr_New(),   $  ; A string array with button names
           createmenu: 0,        $  ; Flag to indicate if a menu shall be built
           menuID: 0L,           $  ; The widget ID of the menu base
           ;; Properties for managing compound widgets within the widget
           compound: Obj_New(),  $  ; A container holding compound widgets for this widget
           cwbaseid: Ptr_New(),  $  ; A list of widget_base IDs which are the bases for CWs

           ;; Properties for compound widgets only
           parent:0L,  $        ; The widget's parent base
           hook:0L, $           ; An index to cwbaseid of the parent widget
           notifyobject:Ptr_New(), $  ; A list of object references and method names which shall be notified upon events

           inherits MGS_BaseObject  $ ; provides basic general properties
         }
END

