; collection of methods customized to each model

; model-specific method to convert numerical time into label time stamp
FUNCTION geov_ncfile::get_timestamp, time

  ; return time stamp, if possible
  if (obj_valid(self.times)) then begin
    stamp = self.times->get_timestamp( time )
    if strlen(stamp) gt 0 then return, stamp
  endif


;  if (ptr_valid(self.timestamps)) then begin
;      ; loop over times and identify time index
;     times = self->get_times()

;;     vid = NCDF_VARID(self.fid, 'time')
;;     IF vid LT 0 THEN Message, 'Invalid variable ID for time!'
;;     NCDF_VARGET, self.fid, vid, times

;      for i=0, n_elements(*times)-1 do begin
;          if (time eq (*times)[i]) then return, (*self.timestamps)[i]
;      endfor
;      return, ' ' ; element not found
;  endif

  return, string(time)
END

; helper routine to get timestamps from time variable
FUNCTION gnc_get_timestamp_from_time, fileid, dateformat


;; *** ACCEPT DIFFERENT TIME NAMES ***
   ;; extract coordinate values and units attribute
   timenames = [ 'Time', 'time', 'times', 'mtime', 'month' ]
   vid = -1L
   i = 0L
   WHILE vid LT 0 AND i LT n_elements(timenames) DO BEGIN 
      vid = NCDF_VARID(fileid, timenames[i])
      i = i+1
   ENDWHILE 

   IF vid LT 0 THEN begin
      ;Message, 'Invalid variable ID for time!',/continue
      return,''
   endif

   NCDF_VARGET, fileid, vid, times


;; *** replacement: first get all attribute names
   result = NCDF_VARINQ(fileid, vid)

   if (result.datatype eq "CHAR" ) then begin
      stamps = string(times)
      return,stamps
   end

   natts = result.natts
   IF natts GT 0 THEN BEGIN
      attnames = strarr(natts)
      FOR i=0L,natts-1 DO attnames[i] = NCDF_ATTNAME(fileid, vid, i)
      ;; get units
      wa = ( where(strlowcase(attnames) EQ 'units', cnt) )[0]
      IF cnt EQ 0 THEN BEGIN 
         message, 'Data file contains no units attribute for time.',/continue
         timeunits = 'unknown'  ;   the default calendar
      endif else $
         NCDF_ATTGET, fileid, vid, 'units', timeunits

      ;; get calendar
      wa = ( where(strlowcase(attnames) EQ 'calendar', cnt) )[0]
      IF cnt EQ 0 THEN BEGIN 
         tmppref = Obj_New('geov_preferences')
         default_cal = tmppref->get_default_calendar()
         Obj_Destroy, tmppref
         message, 'Data file contains no calendar attribute for time - assume '+default_cal, /continue
         calendar = default_cal ;   the default calendar
      endif else $
         NCDF_ATTGET, fileid, vid, 'calendar', calendar
   ENDIF ELSE BEGIN    ;; no time or date or month attributes
      attnames = ''
      calendar = 'unknown'
      timeunits = 'unknown'
   ENDELSE 

   ;; use utility routine to create formatted strings
   ;; define format: if time difference is about 30 days, only print
   ;; month and year
   IF n_elements(dateformat) EQ 0 THEN dateformat='%D %M %Y  %H:%I UT'
   months = [ 'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', $
              'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec' ]
   IF n_elements(times) GT 1 THEN BEGIN
      dtimes = times-shift(times,1)
      IF dtimes[1] GE 28. THEN BEGIN
         dateformat='%M %Y'
         months = ['January', 'February', 'March', 'April', 'May', 'June', $
                   'July', 'August', 'September', 'October', 'November', 'December' ]
      ENDIF
   ENDIF

   ;; NB: out of some frenzy I had once defined the ref year as
   ;; 0000-01-01. This is crazy and leads to the necessity to adjust
   ;; the dates returned by julday in order to get correct results
   result = ncdf_timelabel(times, timeunits, refyear=refyear, $
                           format=dateformat, months=months, calendar=calendar )

   IF n_elements(refyear) GT 0 THEN BEGIN 
      IF refyear LE 1582 THEN BEGIN
         offset = -365. + 1. ;; +3.   ;; magic number ;-)
         result = ncdf_timelabel(times+offset, timeunits, format=dateformat, $
                                 months=months, /ignore_gregor)
      ENDIF
   ENDIF ELSE $
      result = strtrim(times,2)
   return, result

END


; helper routine to get timestamps from date (and datesec) variables
FUNCTION gnc_get_timestamp_from_date, fileid, vid1, vid2

   ;; use date+secs to create timestamp
   if (vid1 ge 0) THEN BEGIN
      NCDF_VARGET, fileid, vid1, dates ; extract coordinate values
   ENDIF
   IF (vid2 ge 0) then begin
      NCDF_VARGET, fileid, vid2, secs ; extract coordinate values
   endif ELSE BEGIN
;; IF n_elements(dates) EQ 0 THEN stop   ; #########
      secs = LonArr(N_Elements(dates))
   endelse
   IF n_elements(dates) EQ 0 THEN Message, 'Time does not adhere to COARDS standard and '+ $
      'cannot find date variable!'

   ;; create time stamps (date assumed as YYYYMMDD)
   result = StrArr(n_elements(dates))
   dates = StrTrim(dates,2)
   FOR i=0L, n_elements(dates)-1 DO BEGIN
      year = long(StrMid(dates[i],0,4))
      month = long(StrMid(dates[i],4,2)) < 12
      day = long(StrMid(dates[i],6,2))
      hour = long(secs[i]/3600.)
      minute = long((secs[i]-3600.*hour) MOD 60)
      result[i] = StrDate([year,month,day,hour,minute], /german)
   ENDFOR

   ;; OLD RELICT: use time variable directly -- even if it does not
;;                                              adhere to COARDS
;   endif else begin
;      ;; times are expressed in julian days 0 to 365, with no leap years
;      secs = 86400*(*times-fix(*times)) ; fractional day
;      secs = 60.*fix(secs/60)   ; approximate to next minute (because of precision issues)
;      for i=0, n_elements(*self.timestamps)-1 do begin
;         (*self.timestamps)[i] = string(julian_day_to_date(1+((*times)[i] mod 365)),'-',get_time_stamp(secs[i]))
;      endfor
;   endelse

   return, result

END

; model-specific method to recompute lev values from hyam and hybm if present
PRO geov_ncfile::gnc_get_pressure_coor

   @nccoord_pars
   level = self->get_levs()
   level2 = self->get_interface_levs()
   if (not ptr_valid(level2)) then begin
;;   print, 'Null pointer LEVEL2 - creating new pointer...'
     pedge = fltarr(n_elements(level)+1)
     level2 = ptr_new(pedge)
   endif

   vida = NCDF_VARID(self.fid, 'hyam')
   vidb = NCDF_VARID(self.fid, 'hybm')

   vidai = NCDF_VARID(self.fid, 'hyai')
   vidbi = NCDF_VARID(self.fid, 'hybi')

   IF (vida GE 0 AND vidb GE 0) THEN BEGIN 
      print,'Computing pressure levels from hybrid coordinate definition ...'
      ncdf_varget, self.fid, vida, hyam
      ncdf_varget, self.fid, vidb, hybm
      ;; check if ECHAM definition (hyam includes P0) or MOZART-like
;      IF max(hyam) GT 1000. THEN p = hyam + 101300.*hybm $
;      ELSE p = 1.e5*hyam + 101300.*hybm
      IF max(hyam) GT 1000. THEN p = hyam + 100000.*hybm $
      ELSE p = 1.e5*hyam + 100000.*hybm
      p = 0.01*p
      *level = p
      self.levunits = 'hPa'

      if ( vidai gt 0 ) and ( vidai gt 0 ) then begin
 
        ncdf_varget, self.fid, vidai, hyai
        ncdf_varget, self.fid, vidbi, hybi

        IF max(hyai) GT 1000. THEN p = hyai + 101300.*hybi $
        ELSE p = 1.e5*hyai + 101300.*hybi
        p = 0.01*p
        *level2 = p

      endif

      (*self.cflags)[self.ilev] = (*self.cflags)[self.ilev] AND NOT NCCOORD_FLAG_INDEX
      (*self.cflags)[self.ilev] = (*self.cflags)[self.ilev] OR NCCOORD_FLAG_PRESSURE
   ENDIF 

END

; method called at the beginning of object initialization
PRO geov_ncfile::custom


   forward_FUNCTION gnc_get_timestamp_from_time, gnc_get_timestamp_from_date
    @geov_pars
    @nccoord_pars

    ; customize coordinate labels
;;    if (self.type eq 'mozart') then BEGIN

        if (self.ilat ge 0) then (*self.clabels)[self.ilat] = 'Latitude [degrees]'
        if (self.ilon ge 0) then (*self.clabels)[self.ilon] = 'Longitude [degrees]'
        ;if (self.ilev ge 0) then (*self.clabels)[self.ilev] = 'Levels [hPa]'
        if (self.ilev ge 0) then (*self.clabels)[self.ilev] = 'Levels ['+self.levunits+']'
        if (self.itim ge 0) then (*self.clabels)[self.itim] = 'Time [days]'

;        if (self.itim ge 0) then begin

;           ;; customized time stamps
;           dateid=-1
;           secsid=-1
;           ;; try to construct timestamps using the time coordinate value
;           ;; (requires time to follow the COARDS convention with
;           ;; units='days since ...')
;           tmppref = Obj_New('geov_preferences')
;           dateformat = tmppref->get_dateformat()
;           Obj_Destroy, tmppref

;           timestamps = gnc_get_timestamp_from_time(self.fid, dateformat)

;           ;; if unsuccessful, create labels from date and datesec values
;           IF timestamps[0] EQ '' THEN BEGIN

;              dateid = -1L
;              secsid = -1L

;              FOR vid=0L, self.nvars-1 DO BEGIN
;                 lname = strlowcase((*self.vnames)[vid])
;                 if (lname eq 'date') then dateid = vid ; found 'date'
;                 if (lname eq 'secs') OR (lname EQ 'datesec') then secsid = vid ; found 'secs'
;              ENDFOR

;              if( dateid ge 0 and secsid ge 0 ) then begin
;                timestamps = gnc_get_timestamp_from_date(self.fid, dateid, secsid)
;              endif else begin
;                times = self->get_times()
;                timestamps = string( *times )
;              endelse

;           ENDIF

;           self.timestamps = ptr_new(timestamps)
;       ENDIF 

;;     ENDIF ELSE BEGIN
        if (self.type ne 'mozart') then BEGIN
          Message, 'Warning: No or invalid file type attribute.', $
             /CONTINUE
          print,'% Program has been designed for MOZART. It should work for many other'
          print,'% files as well, but certain labels or units may be corrupt.'
;           Message, 'Unsupported file type - currently only MOZART-type implemented', $
;              /CONTINUE
;           print,'% Date labels may be corrupted.'
    ;if (self.type eq 'hank') then begin
    ;    times = self->get_times()
    ;    self.timestamps = ptr_new(strarr(n_elements(*times)))
    ;    for t=0, n_elements(*times)-1 do begin
    ;        time = (*times)[t]
    ;        yy = fix(time/1000000)
    ;        time = time mod 1000000
    ;        mm = fix(time/10000)
    ;        time = time mod 10000
    ;        dd = fix(time/100)
    ;        hh = time mod 100
    ;        (*self.timestamps)[t] = string(yy,mm,dd,hh,format='(i2.2,"/",i2.2,"/",i2.2,"-",i2.2)')
    ;    endfor
    ;endif
        ENDIF
;; endelse

; ????    ; convert model levels to pressure coordinates if possible
    if (self.ilev ge 0) then begin
       if (self->is_cflag_set(self.ilev,NCCOORD_FLAG_INDEX) gt 0) THEN BEGIN 
          self->gnc_get_pressure_coor
       ENDIF 
    ENDIF 

    ; pressure/log pressure altitude
    if (self.ilev ge 0) then begin
       if (self->is_cflag_set(self.ilev,NCCOORD_FLAG_PRESSURE) gt 0 $
           or self->is_cflag_set(self.ilev,NCCOORD_FLAG_KM) gt 0) then begin
            self.levs2 = ptr_new(*(self->get_levs()))
            if (self.ilevi ge 0) then thisP0 = max( *(self->get_interface_levs()) ) $
            else thisP0 = max(*self.levs2)
            if (self->is_cflag_set(self.ilev,NCCOORD_FLAG_PRESSURE)) then $
                 (*self.levs2)[*] =  -H0*alog((*self.levs2)[*]/thisP0) > 0. $
            else (*self.levs2)[*] = thisP0*exp(-(*self.levs2)[*]/H0)
        endif
    endif

    ; pressure/log pressure altitude at interface levels
    if (self.ilevi ge 0) then begin
       if (self->is_cflag_set(self.ilev,NCCOORD_FLAG_PRESSURE) gt 0 $
           or self->is_cflag_set(self.ilev,NCCOORD_FLAG_KM) gt 0) then begin
            self.interface_levs2 = ptr_new(*(self->get_interface_levs()))
            thisP0 = max(*self.interface_levs2)
            if (self->is_cflag_set(self.ilev,NCCOORD_FLAG_PRESSURE)) then $
                 (*self.interface_levs2)[*] =    -H0*alog((*self.interface_levs2)[*]/thisP0) > 0. $
            else (*self.interface_levs2)[*] = thisP0*exp(-(*self.interface_levs2)[*]/H0)
        endif
    endif

END
